/*
 * Copyright (C) 2022-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <optional>
#include <type_traits>

namespace API {
enum class FeatureStatus : uint8_t {
    // For customizing WebKit behavior in embedding applications.
    Embedder,
    // Feature in active development. Unfinished, no promise it is usable or safe.
    Unstable,
    // Tools for debugging the WebKit engine. Not generally useful to web developers.
    Internal,
    // Tools for web developers.
    Developer,
    // Enabled by default in test infrastructure, but not ready to ship yet.
    Testable,
    // Enabled by default in Safari Technology Preview, but not considered ready to ship yet.
    Preview,
    // Enabled by default and ready for general use.
    Stable,
    // Enabled by default and in general use for more than a year.
    Mature
};

// Helper for representing feature status as a constant type. Used by the preferences generator to
// validate feature configuration at compile time.
template<API::FeatureStatus Status>
class FeatureConstant : public std::integral_constant<API::FeatureStatus, Status> { };

constexpr std::optional<bool> defaultValueForFeatureStatus(FeatureStatus status)
{
    switch (status) {
    case FeatureStatus::Stable:
    case FeatureStatus::Mature:
        return true;
    case FeatureStatus::Unstable:
    case FeatureStatus::Developer:
    case FeatureStatus::Testable:
    case FeatureStatus::Preview:
        return false;
    case FeatureStatus::Embedder:
    case FeatureStatus::Internal:
        // Embedder features vary widely between platforms, so they have no
        // implied default.
        // FIXME: Internal features should be off by default, but they need
        // additional auditing.
        return { };
    }
}

enum class FeatureCategory : uint8_t {
    None,
    Animation,
    CSS,
    DOM,
    Extensions,
    HTML,
    Javascript,
    Media,
    Networking,
    Privacy,
    Security
};

}
