/*
 * Copyright © 2013 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include "piglit-util-gl.h"

/**
 * @file query_with_invalid_index.c
 *
 * Tests that invalid index given to BeginQueryIndexed(), EndQueryIndexed() or
 * GetQueryIndexediv() is detected and an error is set. The spec says:
 *
 * "The error INVALID_VALUE is generated by BeginQueryIndexed, EndQueryIndexed
 *  or GetQueryIndexediv if <target> is TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN
 *  or PRIMITIVES_GENERATED and <index> is greater or equal
 *  to MAX_VERTEX_STREAMS."
 */

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 32;
	config.supports_gl_core_version = 32;
	config.khr_no_error_support = PIGLIT_NO_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

void
piglit_init(int argc, char **argv)
{
	/**
	 * Separate queries for four different cases preventing the test from
	 * re-using a query that got possibly activated by the previous attempt.
	 *
	 * generated with maximum
	 * generated with maximum + 1
	 * written with maximum
	 * written with maximum + 1
	 */
	GLuint queries[4];
	GLint max_streams;
	bool pass = true;

	piglit_require_extension("GL_ARB_transform_feedback3");

	glGetIntegerv(GL_MAX_VERTEX_STREAMS, &max_streams);
	if (!piglit_check_gl_error(GL_NO_ERROR)) {
		printf("failed to resolve the maximum number of streams\n");
		piglit_report_result(PIGLIT_FAIL);
	}

	if (max_streams <= 0) {
		printf("invalid maximum number of streams: %d\n", max_streams);
		piglit_report_result(PIGLIT_FAIL);
	}

	glGenQueries(ARRAY_SIZE(queries), queries);

	glBeginQueryIndexed(GL_PRIMITIVES_GENERATED, max_streams - 1,
			queries[0]);
	if (!piglit_check_gl_error(GL_NO_ERROR)) {
		printf("failed to activate primitives generated query for the "
			"maximum supported stream\n");
		pass = false;
	}

	if (!piglit_khr_no_error) {
		glBeginQueryIndexed(GL_PRIMITIVES_GENERATED, max_streams,
				    queries[1]);
		pass = piglit_check_gl_error(GL_INVALID_VALUE) && pass;
	}

	glBeginQueryIndexed(GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN,
			max_streams - 1, queries[2]);
	if (!piglit_check_gl_error(GL_NO_ERROR)) {
		printf("failed to activate primitives written query for the "
			"maximum supported stream\n");
		pass = false;
	}

	if (!piglit_khr_no_error) {
		glBeginQueryIndexed(GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN,
				    max_streams, queries[3]);
		pass = piglit_check_gl_error(GL_INVALID_VALUE) && pass;
	}

	glDeleteQueries(ARRAY_SIZE(queries), queries);

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result
piglit_display(void)
{
	/* Should never be reached */
	return PIGLIT_FAIL;
}
