/*  TideList class.  */

/*****************************************************************************\

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

\*****************************************************************************/



#include "tideList.h"
#include "tideListHelp.h"
#include "tideDialog.h"


TideList::TideList (QWidget *parent, const char *name): 
QTabDialog (parent, name)
{
    tideEditor = (TideEditor *) parent;

    QDialog *tideQD = new QDialog (this, 0, FALSE, WDestructiveClose);
    tideQD->setCaption (name);

    tideList = new QTabWidget (tideQD, name, 0);
    tideList->resize (MAP_X_SIZE, 250);

    QWhatsThis::add (tideList, tideListText);

    connect (tideList, SIGNAL (currentChanged (QWidget *)), this, 
        SLOT (slotCurrentChanged ()));


    stationList = NULL;
    numtabs = 0;
    lastrec = 0;
    page_displayed=NULL;


    if (QApplication::desktop ()->width () > (MAP_X_SIZE + 50)
        && QApplication::desktop ()->height () > (250 + 66))
    {
        tideQD->show ();
    }
    else
    {
        tideQD->showMaximized ();
    }
}




TideList::~TideList ()
{
}



void 
TideList::updateTideList ()
{
    int         i;


    //  Clear out the old station names.

    for (i = 0 ; i < numtabs ; i++)
    {
        delete (stationList[i]);
    }
    if (stationList) free (stationList);
    if (page_displayed) free (page_displayed);


    //  Figure out how many tabs we need.

    numtabs = tideEditor->number_of_records / STATIONS_PER_PAGE;
    if (tideEditor->number_of_records % STATIONS_PER_PAGE) numtabs++;


    //  Make an array QListBox widget pointers to hold the QLIstBox widgets.

    stationList = (QListBox **) malloc (numtabs * sizeof (QListBox *));
    page_displayed = (int *) malloc (numtabs * sizeof (int));


    //  Make the QListBox widgets to hold the names and then put them in
    //  the tabs.  Also add the tab title.  Don't put the names in yet, we
    //  only do that if the user clicks on one (this saves time).

    for (i = 0 ; i < numtabs ; i++)
    {
        int start = i * STATIONS_PER_PAGE;
        int end = (i + 1) * STATIONS_PER_PAGE;
        if (tideEditor->number_of_records < end) 
            end = tideEditor->number_of_records - 1;


        stationList[i] = new QListBox (this);

        connect (stationList[i], SIGNAL (selected (int)), this, 
            SLOT (slotSelected ()));
        connect (stationList[i], SIGNAL (highlighted (int)), this, 
            SLOT (slotHighlighted ()));

        char st[4], en[4];
        for (int j = 0 ; j < 3 ; j++)
        {
            st[j] = tolower (tideEditor->station_list[start].name[j]);
            en[j] = tolower (tideEditor->station_list[end].name[j]);
        }
        st[3] = 0;
        en[3] = 0;

        QString tabname = QString ("%1 - %2")
            .arg (st)
            .arg (en);

        tideList->addTab (stationList[i], tabname);

        page_displayed[i] = 0;
    }


    //  This makes 0 the current page and makes sure it is displayed.

    tideList->showPage (stationList[0]);


    //  Load the station names and positions for the current page.

    slotCurrentChanged ();
}



//  Loads the station names for the tab the user just clicked.

void 
TideList::slotCurrentChanged ()
{
    for (int i = 0 ; i < numtabs ; i++)
    {
        if (stationList[i] == tideList->currentPage ())
        {
            page = i;
            break;
        }
    }


    //  If it hasn't already been clicked/displayed we need to populate
    //  the QListBox.

    if (!page_displayed[page])
    {
        int start = page * STATIONS_PER_PAGE;
        int end = (page + 1) * STATIONS_PER_PAGE;
        if (tideEditor->number_of_records < end) 
            end = tideEditor->number_of_records;

        double deg, min, sec;
        char hem;
        for (int j = start, k = 0 ; j < end ; j++, k++)
        {
            QString lats = 
                QString (fixpos (tideEditor->station_list[j].latitude, &deg, 
                &min, &sec, &hem, "lat", tideEditor->pos_format));
            QString lons = 
                QString (fixpos (tideEditor->station_list[j].longitude, &deg, 
                &min, &sec, &hem, "lon", tideEditor->pos_format));

            QString line = QString ("%1, %2   : %3")
                .arg (lats, 13)
                .arg (lons, 14)
                .arg (tideEditor->station_list[j].name);

            stationList[page]->insertItem (line, -1);
        }

        tideList->showPage (stationList[page]);

        page_displayed[page] = 1;
    }
}



//  When you select a station, pop up the tide constituent editor.

void 
TideList::slotSelected ()
{
    int record_number = 
        tideEditor->station_list[stationList[page]->currentItem () +
        page * STATIONS_PER_PAGE].recnum;

    TIDE_RECORD lrec;
    read_tide_record (record_number, &lrec);

    (void) new TideDialog (this, "Tide Constituent Editor", lrec, tideEditor);
}



//  When you highlight a station, highlight it on the map.

void 
TideList::slotHighlighted ()
{
    //  Unhighlight the last highlighted station.

    TIDE_RECORD lrec;
    get_partial_tide_record (tideEditor->station_list[lastrec].recnum, 
        &lrec.header);
    
    tideEditor->highlightStation (tideEditor->station_list[lastrec].latitude,
        tideEditor->station_list[lastrec].longitude, lrec.header.record_type);

    int record_number = stationList[page]->currentItem () + page * 
        STATIONS_PER_PAGE;
    lastrec = record_number;


    //  Highlight the current one.

    tideEditor->highlightStation 
        (tideEditor->station_list[record_number].latitude,
        tideEditor->station_list[record_number].longitude, 99);
}
