/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* xpcd-gate -- plugin/extention to allow xpcd and gimp to
 *              work Hand in Hand
 *              xpcd-gate and xpcd are talking using unix-sockets,
 *              therefore they can't run on different boxes.
 *
 *   (c) 1997 Gerd Knorr <kraxel@goldbach.in-berlin.de>
 *
 * parts of this are cut & pasted from jpeg and script-fu plugins
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Label.h>

#include "libgimp/gimp.h"

#include "ipc.h"

#define  XPCD_LOAD_PROC      "file_xpcd_load"
#define  XPCD_EXTENSION      "plug_in_xpcd_gate"
#define  PERROR(str)         fprintf(stderr,"%s:%d: %s: %s\n",__FILE__,__LINE__,str,strerror(errno))

#define max(x,y)         (((x) < (y)) ? (y) : (x))
#define min(x,y)         (((x) > (y)) ? (y) : (x))

/* ------------------------------------------------------------------------ */

static void     query(void);
static void     run(char *name, int nparams, GParam * param,
		    int *nreturn_vals, GParam ** return_vals);

GPlugInInfo     PLUG_IN_INFO =
{
    NULL,			/* init_proc */
    NULL,			/* quit_proc */
    query,			/* query_proc */
    run,			/* run_proc */
};

static GParamDef load_args[] =
{
    {PARAM_INT32, "run_mode", "Interactive, non-interactive"},
    {PARAM_STRING, "filename", "The name of the file to load"},
    {PARAM_STRING, "raw_filename", "The name of the file to load"},
};
static int      nload_args = sizeof(load_args) / sizeof(load_args[0]);

static GParamDef load_ret[] =
{
    {PARAM_IMAGE, "image", "Output image"},
};
static int      nload_ret = sizeof(load_ret) / sizeof(load_ret[0]);

static GParamDef gate_args[] =
{
    {PARAM_INT32, "run_mode", "Interactive, non-interactive"},
};
static int      ngate_args = sizeof(gate_args) / sizeof(gate_args[0]);

MAIN();

static void
query()
{
    /* handler for "loading" (pass a filename to xpcd) a file */
    gimp_install_procedure(XPCD_LOAD_PROC,
			   "loads PhotoCD files using xpcd",
			   "*requires* a already running xpcd process",
			   "Gerd Knorr",
			   "Gerd Knorr",
			   "1997",
			   "<Load>/PhotoCD",
			   NULL,
			   PROC_PLUG_IN,
			   nload_args, nload_ret,
			   load_args, load_ret);

#if 1
    gimp_register_load_handler(XPCD_LOAD_PROC, "pcd", "");
#endif

    /* proc receives a image from xpcd using unix sockets */
    gimp_install_procedure("plug_in_xpcd_gate",
			   "Receive a ppm image from xpcd",
			   "More help here later",
			   "Gerd Knorr",
			   "Gerd Knorr",
			   "1997",
			   "<Toolbox>/Xtns/xpcd-gate",
			   NULL,
			   PROC_EXTENSION,
			   ngate_args, 0,
			   gate_args, NULL);
}

/* ------------------------------------------------------------------------ */

XtAppContext    app_context;
Widget          app_shell, command;
int             ballout = 0;

int             fake_argc = 1;
char           *fake_argv[] =
{"xpcd-gate", NULL};

char            socket_file[256];
int             socket_fd;

static String   fallback_res[] =
{
    "xpcd-gate*background:	lightgray",
    "xpcd-gate*font:  -*-new century schoolbook-bold-r-normal-*-12-*",
    "xpcd-gate*showGrip:        false",
    "xpcd-gate*exit.label:	Close",
    "xpcd-gate*exit.skipAdjust: true",
    "xpcd-gate*help.label:	Start xpcd now.  Select \"The GIMP\"\\n"
                               "from the \"Viewers\" menu to load the\\n"
                               "PhotoCD images directly.",
    NULL
};

Boolean
exit_WP(XtPointer client_data)
{
    ballout = 1;
    return TRUE;
}

void
exit_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    XtAppAddWorkProc(app_context, exit_WP, NULL);
    XtDestroyWidget(app_shell);
}

int
create_socket()
{
    int             sock;
    struct sockaddr_un server_addr;

    sprintf(socket_file, "/tmp/xpcd-gate-%d", getpid());
    server_addr.sun_family = AF_UNIX;
    strcpy(server_addr.sun_path, socket_file);

    if (-1 == (socket_fd = sock = socket(PF_UNIX, SOCK_STREAM, 0))) {
	PERROR("socket");
	return -1;
    }
    if (-1 == bind(sock, (struct sockaddr *) &server_addr,
		   sizeof(server_addr))) {
	PERROR("bind");
	return -1;
    }
    if (-1 == listen(sock, 5)) {
	PERROR("listen");
	return -1;
    }
    chmod(socket_file, 0700);
    return sock;
}

void
kill_socket()
{
    close(socket_fd);
    unlink(socket_file);
}

struct IMG {
    int             width, height, fd, gray;
    int             img_size, buf_size;
    int             buf_start, buf_fill, buf_end;

    gint32          image_ID;
    gint32          layer_ID;
    GDrawable      *drawable;
    GPixelRgn       pixel_rgn;
    guchar         *buf;
    int             tile_height;
};

static void
read_input(XtPointer data, int *fd, XtInputId * iproc)
{
    struct IMG     *img;
    int             rc;

    img = data;

    rc = read(img->fd,
	      img->buf + img->buf_fill,
	      img->buf_end - img->buf_fill);
    if (rc <= 0) {
	PERROR("xpcd-gate: read");
	exit(1);
    }
    img->buf_fill += rc;
    if (img->buf_fill < img->buf_end)
	return;

    gimp_pixel_rgn_set_rect(&img->pixel_rgn, img->buf, 0,
		       img->buf_start / img->width / (img->gray ? 1 : 3),
			    img->drawable->width,
			img->buf_end / img->width / (img->gray ? 1 : 3));

    img->buf_start += img->buf_end;
    img->buf_fill -= img->buf_end;
    img->buf_end = min(img->img_size, img->buf_start + img->buf_size)
	- img->buf_start;
    fprintf(stderr, "(%d) ", img->fd);
    if (img->buf_start < img->img_size)
	return;

    fprintf(stderr, "xpcd-gate(%d): over\n", img->fd);
    gimp_drawable_flush(img->drawable);
    gimp_display_new(img->image_ID);
    close(*fd);
    g_free(img->buf);
    free(img);
    XtRemoveInput(*iproc);
}

static void
accept_input(XtPointer data, int *fd, XtInputId * iproc)
{
    int             r_sock, len;
    struct sockaddr_un addr;
    char            buf[33];
    int             rc, i;
    struct IMG     *img;

    len = sizeof(addr);
    fprintf(stderr, "xpcd-gate: accept... ");
    r_sock = accept(*fd, (struct sockaddr *) &addr, &len);
    fprintf(stderr, "ok (%d)\n", r_sock);

    img = malloc(sizeof(struct IMG));
    memset(img, 0, sizeof(struct IMG));

    img->fd = r_sock;
    if ((rc = read(img->fd, buf, 32)) <= 0) {
	PERROR("read");
	exit(1);
    }
    buf[rc] = 0;
    if (2 == sscanf(buf, "P6\n%d %d\n255\n%n",
		    &(img->width), &(img->height), &i)) {
	img->gray = 0;
    } else if (2 == sscanf(buf, "P5\n%d %d\n255\n%n",
			   &(img->width), &(img->height), &i)) {
	img->gray = 1;
    } else {
	fprintf(stderr, "xpcd-gate(%d): oops, sscanf failed\n", img->fd);
	exit(1);
    }
    fprintf(stderr, "xpcd-gate(%d): img size %dx%d, %s\n",
	 img->fd, img->width, img->height, img->gray ? "gray" : "color");

    img->tile_height = gimp_tile_height();
    img->buf_size = img->tile_height * img->width * (img->gray ? 1 : 3);
    img->img_size = img->height * img->width * (img->gray ? 1 : 3);
    img->buf = g_new(guchar, img->buf_size);

    img->image_ID = gimp_image_new
	(img->width, img->height, (img->gray ? GRAY : RGB));
    img->layer_ID = gimp_layer_new
	(img->image_ID, "PhotoCD", img->width, img->height,
	 (img->gray ? GRAY_IMAGE : RGB_IMAGE), 100, NORMAL_MODE);
    gimp_image_add_layer(img->image_ID, img->layer_ID, 0);
    img->drawable = gimp_drawable_get(img->layer_ID);
    gimp_pixel_rgn_init(&img->pixel_rgn, img->drawable, 0, 0,
			img->drawable->width, img->drawable->height,
			TRUE, FALSE);

    img->buf_start = 0;
    img->buf_fill = 0;
    img->buf_end = min(img->img_size, img->buf_size);
    if (i < rc) {
	fprintf(stderr, "xpcd-gate(%d): %d/%d\n", img->fd, i, rc);
	memcpy(img->buf, buf + i, rc - i);
	img->buf_fill = rc - i;
    }
    XtAppAddInput(app_context, img->fd, (XtPointer) XtInputReadMask,
		  read_input, img);
}

/* ------------------------------------------------------------------------ */

static void
run(char *name, int nparams, GParam * param,
    int *nreturn_vals, GParam ** return_vals)
{
    static GParam   values[2];
    GRunModeType    run_mode;

    Display        *dpy;
    Window          win;
    Widget          paned;
    int             sock;

    run_mode = param[0].data.d_int32;

    *nreturn_vals = 1;
    *return_vals = values;
    values[0].type = PARAM_STATUS;
    values[0].data.d_status = STATUS_CALLING_ERROR;

    if (strcmp(name, XPCD_LOAD_PROC) == 0) {
	/* pass filename to xpcd */
	fprintf(stderr, XPCD_LOAD_PROC ": %s\n", param[1].data.d_string);

	if (NULL == (dpy = XOpenDisplay(NULL))) {
	    fprintf(stderr, XPCD_LOAD_PROC ": can't open display\n");
	    values[0].data.d_status = STATUS_EXECUTION_ERROR;
	    return;
	}
	ipc_init_atoms(dpy);
	if (0 == (win = ipc_find_window(dpy, xpcd_gate))) {
	    fprintf(stderr, XPCD_LOAD_PROC
		    ": warning: xpcd-gate does not run\n");
	}
	if (0 == (win = ipc_find_window(dpy, xpcd_ver))) {
	    fprintf(stderr, XPCD_LOAD_PROC ": no running xpcd found\n");
	    values[0].data.d_status = STATUS_EXECUTION_ERROR;
	    return;
	}
	ipc_pass_cmd(dpy, win, IPC_PASS_GIMP, 1, &param[1].data.d_string);
	XFlush(dpy);
	XCloseDisplay(dpy);
	values[0].data.d_status = STATUS_SUCCESS;

    } else if (strcmp(name, XPCD_EXTENSION) == 0) {
	/* the hard part: create a socket and handle even more than one
	 * connect in parallel correctly */

	app_shell = XtAppInitialize(&app_context, "xpcd-gate",
				    NULL, 0,
				    &fake_argc, fake_argv,
				    fallback_res,
				    NULL, 0);
	dpy = XtDisplay(app_shell);
	paned = XtVaCreateManagedWidget("paned", panedWidgetClass, app_shell,
					NULL);
	XtVaCreateManagedWidget("help", labelWidgetClass, paned,
				NULL);
	command = XtVaCreateManagedWidget("exit", commandWidgetClass, paned,
					  NULL);
	XtAddCallback(command, XtNcallback, exit_CB, NULL);

	ipc_init_atoms(dpy);
	if (0 != ipc_find_window(dpy, xpcd_gate)) {
	    fprintf(stderr, XPCD_EXTENSION ": already running\n");
	} else {
	    sock = create_socket();
	    atexit(kill_socket);
	    if (-1 != sock) {
		XtRealizeWidget(app_shell);
		XChangeProperty(dpy, XtWindow(app_shell),
				xpcd_gate, XA_STRING,
				8, PropModeReplace,
				socket_file, strlen(socket_file) + 1);
		XtAppAddInput(app_context, sock, (XtPointer) XtInputReadMask,
			      accept_input, NULL);

		while (!ballout) {
		    XEvent          event;

		    XtAppNextEvent(app_context, &event);
		    XtDispatchEvent(&event);
		}
	    }
	}
	values[0].data.d_status = STATUS_SUCCESS;
	return;
    }
}
