#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Simple.h>
#include <X11/Xaw/SimpleMenu.h>

#include "config.h"

#include "toolbox.h"
#include "pcd.h"
#include "x11.h"
#include "dither.h"
#include "xpcd.h"
#include "ipc.h"

/* ---------------------------------------------------------------------- */

struct THUMBNAILS_WP {
    Widget          box;
    int             i, count;
    struct PCD_IMAGE *img;
    XtWorkProcId    wpid;
    char           *filename;
};

static void
thumbnail_command_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    char            filename[256];

    strcpy(filename, XtName(XtParent(widget)));
    strcat(filename, "/images/");
    strcat(filename, XtName(widget));
    XtAppAddWorkProc(app_context, open_file_WP,
	     (XtPointer) strcpy(malloc(strlen(filename) + 1), filename));
}

void
delete_pixmap_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    Pixmap          pix;

    pix = (Pixmap) client_data;
    XFreePixmap(dpy, pix);
}

Boolean
load_thumbnails_WP(XtPointer client_data)
{
    int             left, top, width, height, y;
    struct THUMBNAILS_WP *thumbnails;
    unsigned char  *image;
    Pixmap          pix;
    Widget          button;
    char            filename[15];
    unsigned char  *rgb_line;

    left = top = width = height = 0;
    thumbnails = (struct THUMBNAILS_WP *) client_data;
    if (-1 == pcd_select(thumbnails->img, 1, thumbnails->i, use_grays, 0,
	     tiny_turn ? pcd_get_rot(thumbnails->img, thumbnails->i) : 0,
			 &left, &top, &width, &height)) {
	fprintf(stderr, "%s:%d: libpcd: %s\n",
		__FILE__, __LINE__, pcd_errmsg);
	exit(1);
    }
    width = width >> 1;
    height = height >> 1;

    image = malloc(width * height * display_depth);
    if (display_type == PSEUDOCOLOR || use_grays) {
	rgb_line = malloc((width) * (use_grays ? 1 : 3));

	for (y = 0; y < height; y++) {
	    if (-1 == pcd_get_image_line(thumbnails->img, y, rgb_line,
			(use_grays ? PCD_TYPE_GRAY : PCD_TYPE_RGB), 1)) {
		fprintf(stderr, "%s:%d: libpcd: %s\n",
			__FILE__, __LINE__, pcd_errmsg);
		exit(1);
	    }
	    x11_data_to_ximage(rgb_line, image + y * width * display_depth,
			       width, 1, y, use_grays);
	}
	free(rgb_line);
    } else {
	pcd_set_lookup(thumbnails->img, x11_lut_red, x11_lut_green, x11_lut_blue);
	if (-1 == pcd_get_image(thumbnails->img, image, PCD_TYPE_LUT_LONG, 1)) {
	    fprintf(stderr, "%s:%d: libpcd: %s\n",
		    __FILE__, __LINE__, pcd_errmsg);
	    exit(1);
	}
    }
    pix = x11_create_pixmap(app_shell, image, width, height, use_grays);
    free(image);

    sprintf(filename, "img%04d.pcd", thumbnails->i + 1);

    button = XtVaCreateManagedWidget(filename,
				     commandWidgetClass, thumbnails->box,
				     XtNbitmap, pix,
			      XtNheight, tiny_turn ? 96 + 8 : height + 8,
				XtNwidth, tiny_turn ? 96 + 8 : width + 8,
				     NULL);
    XtAddCallback(button, XtNcallback, thumbnail_command_CB, NULL);
    XtAddCallback(button, XtNdestroyCallback,
		  (XtCallbackProc) delete_pixmap_CB, (XtPointer) pix);

    thumbnails->i++;
    if (thumbnails->i < thumbnails->count) {
	return FALSE;
    } else {
	thumbnails->wpid = 0;
	return TRUE;
    }
}

void
destroy_thumbnails_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    struct THUMBNAILS_WP *thumbnails;

    thumbnails = (struct THUMBNAILS_WP *) client_data;
    pcd_close(thumbnails->img);

    if (thumbnails->wpid)
	XtRemoveWorkProc(thumbnails->wpid);

    free(thumbnails->img);
    free(thumbnails->filename);
    free(thumbnails);
}

/* ---------------------------------------------------------------------- */

static struct MENU reslist[] =
{
    {1, "s1", NULL, 0},
    {2, "s2", NULL, 0},
    {3, "s3", NULL, 0},
    {4, "s4", NULL, 0},
    {5, "s5", NULL, 0},
    {0, NULL}
};

struct IMAGE_WP {
    Widget          label;
    Pixmap          pixmap;
    GC              wgc;
    XtWorkProcId    wpid;
    struct PCD_IMAGE *img;
    unsigned char  *image;
    int             y, width, height;
    int             x1, x2, y1, y2;
    char           *filename;
};

Boolean
load_image_WP(XtPointer client_data)
{
    int             left, top, width, height;
    struct IMAGE_WP *wpdata;

    wpdata = (struct IMAGE_WP *) client_data;
    if (wpdata->y == 0) {
	left = top = width = height = 0;
	if (-1 == pcd_select(wpdata->img, 2, 0, use_grays, 0,
			     pcd_get_rot(wpdata->img, 0),
			     &left, &top, &width, &height)) {
	    fprintf(stderr, "%s:%d: libpcd: %s\n",
		    __FILE__, __LINE__, pcd_errmsg);
	    exit(1);
	}
	wpdata->width = width;
	wpdata->height = height;
	wpdata->image = malloc(width * height * display_depth);
	if (display_type == TRUECOLOR)
	    pcd_set_lookup(wpdata->img, x11_lut_red, x11_lut_green, x11_lut_blue);
    } else {
	width = wpdata->width;
	height = wpdata->height;
    }

    if (display_type == PSEUDOCOLOR || use_grays) {
	unsigned char  *rgb_line = malloc((width) * (use_grays ? 1 : 3));

	if (-1 == pcd_get_image_line(wpdata->img, wpdata->y, rgb_line,
			(use_grays ? PCD_TYPE_GRAY : PCD_TYPE_RGB), 0)) {
	    fprintf(stderr, "%s:%d: libpcd: %s\n",
		    __FILE__, __LINE__, pcd_errmsg);
	    exit(1);
	}
	x11_data_to_ximage(rgb_line,
		       wpdata->image + wpdata->y * width * display_depth,
			   width, 1, wpdata->y, use_grays);
	free(rgb_line);
    } else {
	if (-1 == pcd_get_image_line(wpdata->img, wpdata->y,
				   wpdata->image + wpdata->y * width * 4,
				     PCD_TYPE_LUT_LONG, 0)) {
	    fprintf(stderr, "%s:%d: libpcd: %s\n",
		    __FILE__, __LINE__, pcd_errmsg);
	    exit(1);
	}
    }
    wpdata->y++;

    if (wpdata->y < wpdata->height) {
	return FALSE;
    } else {
	wpdata->pixmap =
	    x11_create_pixmap(app_shell, wpdata->image, width, height, use_grays);
	XtVaSetValues(wpdata->label,
		      XtNbackgroundPixmap, wpdata->pixmap,
		      NULL);
	free(wpdata->image);
	wpdata->image = NULL;
	wpdata->wpid = 0;
	return TRUE;
    }
}

void
destroy_image_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    struct IMAGE_WP *image;

    image = (struct IMAGE_WP *) client_data;
    pcd_close(image->img);

    if (image->wpid)
	XtRemoveWorkProc(image->wpid);
    if (image->pixmap)
	XFreePixmap(dpy, image->pixmap);
    if (image->image)
	free(image->image);
    free(image->img);
    free(image->filename);
    free(image);
}

void
image_expose_event(Widget widget,
		   XtPointer client_data,
		   XExposeEvent * event)
{
    struct IMAGE_WP *image;

    image = (struct IMAGE_WP *) client_data;
    if (image->wgc == 0) {
	XGCValues       values;

	values.foreground = x11_red;
	image->wgc = XCreateGC(dpy, XtWindow(widget),
			       GCForeground, &values);
    }
#if 0
    XCopyArea(dpy,
	      image->pixmap,
	      XtWindow(widget),
	      image->wgc,
	      event->x, event->y,
	      event->width, event->height,
	      event->x, event->y);
#endif

    if ((image->x2 != 0) && (image->y2 != 0)) {
	XDrawRectangle(dpy, XtWindow(widget), image->wgc,
		       image->x1, image->y1,
		       image->x2 - image->x1,
		       image->y2 - image->y1);
    }
}

void
image_mouse_event(Widget widget, XtPointer client_data, XEvent * event)
{
    static int      startx, starty;	/* XXX threads */
    struct IMAGE_WP *image;
    int             sel, i, x, y;

    image = (struct IMAGE_WP *) client_data;

    switch (event->type) {
    case ButtonPress:
	if (event->xbutton.button == Button1) {
	    startx = event->xbutton.x;
	    starty = event->xbutton.y;
	}
	break;
    case ButtonRelease:
	if (event->xbutton.button == Button1) {
	    if ((event->xbutton.x == startx) || (event->xbutton.y == starty)) {
		image->x1 = 0;
		image->x2 = 0;
		image->y1 = 0;
		image->y2 = 0;
	    } else {
		if (event->xbutton.x < startx) {
		    image->x1 = max(0, event->xbutton.x);
		    image->x2 = startx;
		} else {
		    image->x2 = min(image->width - 1, event->xbutton.x);
		    image->x1 = startx;
		}
		if (event->xbutton.y < starty) {
		    image->y1 = max(0, event->xbutton.y);
		    image->y2 = starty;
		} else {
		    image->y2 = min(image->y - 1, event->xbutton.y);
		    image->y1 = starty;
		}
	    }
	    XClearArea(dpy, XtWindow(widget), 0, 0, 0, 0, True);
	}
	if (event->xbutton.button == Button3) {
	    for (i = 0; i < 5; i++) {
		if (NULL == reslist[i].title) {
		    reslist[i].title = malloc(20);
		}
		x = image->x2 - image->x1;
		y = image->y2 - image->y1;
		if (0 == x)
		    x = PCD_WIDTH(2, pcd_get_rot(image->img, 0));
		if (0 == y)
		    y = PCD_HEIGHT(2, pcd_get_rot(image->img, 0));
		sprintf(reslist[i].title, "%i x %i", (x << i) / 2, (y << i) / 2);
	    }
	    if (pcd_get_maxres(image->img) == 3)
		reslist[3].disabled = reslist[4].disabled = 1;
	    else
		reslist[3].disabled = reslist[4].disabled = 0;
	    sel = popup_menu(widget, reslist);
	    if (sel != -1)
		loadimage(app_shell, image->img,
			  image->x1, image->x2, image->y1, image->y2, sel,
			  viewer_cmd[viewer_current], image->filename);
	}
	break;
    case MotionNotify:
	if (event->xbutton.x < startx) {
	    image->x1 = max(0, event->xbutton.x);
	    image->x2 = startx;
	} else {
	    image->x2 = min(image->width - 1, event->xbutton.x);
	    image->x1 = startx;
	}
	if (event->xbutton.y < starty) {
	    image->y1 = max(0, event->xbutton.y);
	    image->y2 = starty;
	} else {
	    image->y2 = min(image->y - 1, event->xbutton.y);
	    image->y1 = starty;
	}
	XClearArea(dpy, XtWindow(widget), 0, 0, 0, 0, True);
	break;
    }
}

/* ---------------------------------------------------------------------- */

struct OPEN_FILES {
    Widget          shell;
    char           *filename;
    struct OPEN_FILES *next;
};

struct OPEN_FILES *filelist = NULL;

void
add_window_to_list(Widget shell, char *filename)
{
    struct OPEN_FILES *thisfile;

    thisfile = malloc(sizeof(struct OPEN_FILES));
    memset(thisfile, 0, sizeof(struct OPEN_FILES));

    thisfile->shell = shell;
    thisfile->filename = filename;
    thisfile->next = filelist;
    filelist = thisfile;
}

Boolean
open_file_WP(XtPointer client_data)
{
    char           *filename, *h;
    struct PCD_IMAGE *img;
    int             rc;
    struct THUMBNAILS_WP *thumbnails;
    struct IMAGE_WP *image;
    struct OPEN_FILES *thisfile;
    Widget          shell, viewport, box, label;

    filename = (char *) client_data;

    for (thisfile = filelist; thisfile != NULL; thisfile = thisfile->next)
	if (0 == strcmp(thisfile->filename, filename))
	    break;
    if (thisfile != NULL) {
	/* XXX does not work perfect... */
	XtPopdown(thisfile->shell);
	XtPopup(thisfile->shell, XtGrabNone);
	return TRUE;
    }
    img = malloc(sizeof(struct PCD_IMAGE));
    memset(img, 0, sizeof(struct PCD_IMAGE));

    rc = pcd_open(img, filename);
    if (rc == -1) {
	/* error */
	tell_user(app_shell, "str_perror_title", pcd_errmsg);
	free(filename);
	free(img);
	return TRUE;
    }
    if (0 == rc) {
	/* image */
	shell = XtVaAppCreateShell("image", "Xpcd-2",
				   applicationShellWidgetClass,
				   dpy,
				   XtNtitle, filename,
				   XtNclientLeader, app_shell,
				   NULL);
	XtOverrideTranslations(shell, XtParseTranslationTable
			       ("<Message>WM_PROTOCOLS: CloseFile()"));
	label = XtVaCreateManagedWidget(filename,
					simpleWidgetClass, shell,
					NULL);

	XtVaSetValues(label,
		      XtNwidth, PCD_WIDTH(2, pcd_get_rot(img, 0)),
		      XtNheight, PCD_HEIGHT(2, pcd_get_rot(img, 0)),
		      NULL);
	XtVaSetValues(shell,
		      XtNmaxWidth, PCD_WIDTH(2, pcd_get_rot(img, 0)),
		      XtNmaxHeight, PCD_HEIGHT(2, pcd_get_rot(img, 0)),
		      XtNminWidth, PCD_WIDTH(2, pcd_get_rot(img, 0)),
		      XtNminHeight, PCD_HEIGHT(2, pcd_get_rot(img, 0)),
		      NULL);

	XtRealizeWidget(shell);
	XDefineCursor(dpy, XtWindow(shell), left_ptr);
	XSetWMProtocols(dpy, XtWindow(shell), &wm_close, 1);

	image = malloc(sizeof(struct IMAGE_WP));
	memset(image, 0, sizeof(struct IMAGE_WP));

	image->label = label;
	image->img = img;
	image->y = 0;
	image->filename = filename;
	add_window_to_list(shell, filename);
	XtAddCallback(shell, XtNdestroyCallback,
		   (XtCallbackProc) destroy_image_CB, (XtPointer) image);
	XtAddEventHandler(label, ExposureMask, False,
			  (XtEventHandler) image_expose_event,
			  (XtPointer) image);
	XtAddEventHandler(label,
		 ButtonPressMask | ButtonReleaseMask | Button1MotionMask,
			  False, (XtEventHandler) image_mouse_event,
			  (XtPointer) image);
	image->wpid = XtAppAddWorkProc
	    (app_context, load_image_WP, (XtPointer) image);
	XtInstallAllAccelerators(label, shell);
    } else {
	/* thumbnails */
	shell = XtVaAppCreateShell("thumbnails", "Xpcd-2",
				   applicationShellWidgetClass,
				   dpy,
				   XtNtitle, filename,
				   XtNclientLeader, app_shell,
				   NULL);
	XtOverrideTranslations(shell, XtParseTranslationTable
			       ("<Message>WM_PROTOCOLS: CloseFile()"));
	viewport = XtVaCreateManagedWidget("viewport",
					   viewportWidgetClass, shell,
					   XtNallowHoriz, False,
					   XtNallowVert, True,
					   NULL);
	h = strrchr(filename, '/');
	*h = 0;
	box = XtVaCreateManagedWidget(filename,
				      boxWidgetClass, viewport,
				      XtNsensitive, True,
				      NULL);
	*h = '/';
	XtRealizeWidget(shell);
	XDefineCursor(dpy, XtWindow(shell), left_ptr);
	XSetWMProtocols(dpy, XtWindow(shell), &wm_close, 1);
	thumbnails = malloc(sizeof(struct THUMBNAILS_WP));
	memset(thumbnails, 0, sizeof(struct THUMBNAILS_WP));

	thumbnails->count = rc;
	thumbnails->img = img;
	thumbnails->box = box;
	thumbnails->filename = filename;
	add_window_to_list(shell, filename);
	XtAddCallback(shell, XtNdestroyCallback,
	 (XtCallbackProc) destroy_thumbnails_CB, (XtPointer) thumbnails);
	thumbnails->wpid = XtAppAddWorkProc
	    (app_context, load_thumbnails_WP, (XtPointer) thumbnails);
	/* keyboard scroll stuff */
	XtAddCallback(viewport, XtNreportCallback,
		   (XtCallbackProc) report_viewport_CB, (XtPointer) box);
    }
    return TRUE;
}

void
close_all_files(void)
{
    struct OPEN_FILES *this, *next;

    for (this = filelist; this != NULL; this = next) {
	next = this->next;
	XtDestroyWidget(this->shell);
	free(this);
    }
}

void
close_file_AC(Widget widget, XEvent * event,
	      String * params, Cardinal * num_params)
{
    struct OPEN_FILES *thisfile, *prevfile;

    for (prevfile = NULL, thisfile = filelist;
	 thisfile != NULL;
	 prevfile = thisfile, thisfile = thisfile->next)
	if (thisfile->shell == widget)
	    break;

    if (thisfile == NULL)
	oops("widget not found in close_file_AC()");

    XtDestroyWidget(widget);

    if (prevfile)
	prevfile->next = thisfile->next;
    else
	filelist = thisfile->next;
    free(thisfile);
}
