/* aguix.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: aguix.cc,v 1.28 2002/03/18 01:13:53 ralf Exp $ */

#include "aguix.h"
#include "awindow.h"

AGUIX::AGUIX()
{
  initOK=False;
  classname=NULL;
  myfont=NULL;
  mainfont=NULL;
  dsp=NULL;
  firstwin=NULL;
  colors=0;
  wins=new List();
  messages=new List();
  fonts=new List();
  privatecmap=false;
  cutstart=NULL;
  cutbuffer=NULL;
}
  
int AGUIX::initX(int targc,char **targv,char *tclassname)
{
#ifdef DEBUG
  printf("Opening display...");
#endif
  char *displayname=getenv("DISPLAY");
  dsp=XOpenDisplay(displayname);
  if(dsp==NULL) {
#ifdef DEBUG
    printf("failed\n");
#endif
    return 5;
  }
#ifdef DEBUG
  printf("Ok\n");
#endif
  this->classname=dupstring(tclassname);
  scr=XDefaultScreen(dsp);
  cmap=DefaultColormap(dsp,scr);
  white=WhitePixel(dsp,scr);
  black=BlackPixel(dsp,scr);

  gc=XCreateGC(dsp,RootWindow(dsp,scr),0,0);
  XGCValues gc_values;
  gc_values.graphics_exposures=False;
  gc_values.foreground=black;
  gc_values.background=white;
  gc_values.line_width=0;
  gc_values.cap_style=CapButt;
#ifdef DEBUG
  printf("Loading font...");
#endif
  myfont=XLoadQueryFont(dsp,"fixed");
  if(myfont==NULL) {
    printf("no font, trying default: ");
    myfont=XLoadQueryFont(dsp,"fixed");
    if(myfont==NULL) panic("failed");
#ifndef DEBUG
    printf("Ok\n");
#endif
  }
#ifdef DEBUG
  printf("Ok\n");
#endif
  CharHeight=myfont->ascent+myfont->descent;
#if 0
  CharWidth=myfont->max_bounds.rbearing-myfont->min_bounds.lbearing;
#else
  CharWidth=XTextWidth(myfont,"M",1);
#endif
  gc_values.font=myfont->fid;
  unsigned long gc_valuemask=GCGraphicsExposures|GCForeground|GCBackground|GCLineWidth|GCFont|GCCapStyle;
  XChangeGC(dsp,gc,gc_valuemask,&gc_values);
  if(getFont("fixed")==NULL) printf("Opps, no \"fixed\"\n");
  WM_delete_window=XInternAtom(dsp,"WM_DELETE_WINDOW",False);
  this->argc=targc;
  this->argv=targv;
  createGroupWin();

  cursors[WAIT_CURSOR]=XCreateFontCursor(dsp,XC_watch);
  cursors[SCROLLH_CURSOR]=XCreateFontCursor(dsp,XC_sb_h_double_arrow);
  cursors[SCROLLV_CURSOR]=XCreateFontCursor(dsp,XC_sb_v_double_arrow);

  unsigned int rd,rbw;
  int rx, ry;
  Window rootw;
  XGetGeometry( dsp, DefaultRootWindow( dsp ), &rootw, &rx, &ry, &rootWindowWidth, &rootWindowHeight, &rbw, &rd );

  initOK=True;
  return 0;
}

AGUIX::~AGUIX()
{
  AGMessage *agmsg;
  while((agmsg=getAGMsg())!=NULL) ReplyMessage(agmsg);
  if(checkX()==True) {
    if(dsp!=NULL) {
      closeX();
    }
  }
  cancelCut();
  if(cutbuffer!=NULL) _freesafe(cutbuffer);
  delete wins;
  delete messages;
  delete fonts;
}

void AGUIX::panic(const char *msg)
{
  fprintf(stderr,"%s\n",msg);
  exit(1);
}

void AGUIX::closeX()
{
  destroyGroupWin();
  int id=fonts->initEnum();
  AGUIXFont *tf=(AGUIXFont*)fonts->getFirstElement(id);
  while(tf!=NULL) {
    delete tf;
    tf=(AGUIXFont*)fonts->getNextElement(id);
  }
  fonts->closeEnum(id);
  if(classname!=NULL) _freesafe(classname);
  if(myfont!=NULL) XFreeFont(dsp,myfont);
  if(privatecmap==true) {
    XFreeColormap(dsp,cmap);
  }
  XFreeCursor(dsp,cursors[WAIT_CURSOR]);
  XFreeCursor(dsp,cursors[SCROLLH_CURSOR]);
  XFreeCursor(dsp,cursors[SCROLLV_CURSOR]);
#ifdef DEBUG
  printf("Closing display\n");
#endif
  XCloseDisplay(dsp);
  dsp=NULL;
}

int AGUIX::checkX()
{
  return initOK;
}

int AGUIX::getDepth()
{
  return DefaultDepth(dsp,scr);
}

Display *AGUIX::getDisplay()
{
  return dsp;
}

int AGUIX::getScreen()
{
  return scr;
}

int AGUIX::getCharHeight()
{
  return CharHeight;
}

int AGUIX::getCharWidth()
{
  return CharWidth;
}

void AGUIX::setFG(int color)
{
  if(color<0) color=0;
  if(color<colors) {
    XSetForeground(dsp,gc,ColBuf[color]);
  } else {
    XSetForeground(dsp,gc,ColBuf[colors-1]);
  }
}

void AGUIX::setFG(GC tgc,int color)
{
  if(tgc==0) setFG(color);
  else {
    if(color<0) color=0;
    if(color<colors) {
      XSetForeground(dsp,tgc,ColBuf[color]);
    } else {
      XSetForeground(dsp,tgc,ColBuf[colors-1]);
    }
  }
}

void AGUIX::setBG(int color)
{
  if(color<0) color=0;
  if(color<colors) {
    XSetBackground(dsp,gc,ColBuf[color]);
  } else {
    XSetBackground(dsp,gc,ColBuf[colors-1]);
  }
}

void AGUIX::setBG(GC tgc,int color)
{
  if(tgc==0) setBG(color);
  else {
    if(color<0) color=0;
    if(color<colors) {
      XSetBackground(dsp,tgc,ColBuf[color]);
    } else {
      XSetBackground(dsp,tgc,ColBuf[colors-1]);
    }
  }
}

void AGUIX::FillRectangle(Drawable buffer,int x,int y,int w,int h)
{
  XFillRectangle(dsp,buffer,gc,x,y,w,h);
}

void AGUIX::FillRectangle(Drawable buffer,GC tgc,int x,int y,int w,int h)
{
  if(tgc==0) FillRectangle(buffer,x,y,w,h);
  else XFillRectangle(dsp,buffer,tgc,x,y,w,h);
}

void AGUIX::DrawText(Drawable buffer,const char *text,int x,int y)
{
  XDrawString(dsp,buffer,gc,x,y+getFontBaseline(),text,strlen(text));
}

void AGUIX::DrawText(Drawable buffer,AGUIXFont *tf,const char *text,int x,int y)
{
  XDrawString(dsp,buffer,tf->getGC(),x,y+tf->getBaseline(),text,strlen(text));
}

int AGUIX::getFontBaseline()
{
  if(mainfont!=NULL) return mainfont->getBaseline();
  return myfont->ascent;
}

int AGUIX::AddColor(int red,int green,int blue)
{
  XColor col;
  if(colors<256) {
    col.flags=DoRed|DoGreen|DoBlue;
    col.red=red*256;
    col.green=green*256;
    col.blue=blue*256;
    if(!XAllocColor(dsp,cmap,&col)) {
      // error
      if(privatecmap==true) return -1;
      else {
        // now try to go to a private colormap
	changeColormap();
	return AddColor(red,green,blue);
      }
    }
    ColBuf[colors++]=col.pixel;
  }
  return colors-1;
}

void AGUIX::removeLastColor()
{
  if(colors>0) {
    XFreeColors(dsp,cmap,ColBuf+colors-1,1,0);
    colors--;
  }
}

int AGUIX::getMaxCols()
{
  return colors;
}

char *AGUIX::getClassname()
{
  char *tstr;
  tstr=dupstring(classname);
  return tstr;
}

unsigned long AGUIX::getPixel(int color)
{
  if((color<colors)&&(color>=0)) {
    return ColBuf[color];
  } else return 0;
}

Atom *AGUIX::getCloseAtom()
{
  return &WM_delete_window;
}

void AGUIX::insertWindow(AWindow *win)
{
  if(win==NULL) return;
  wins->addElement(win);
}

void AGUIX::removeWindow(AWindow *win)
{
  if(win==NULL) return;
  wins->removeElement(win);
}

Message *AGUIX::wait4mess(int mode,bool createagmsg)
{
  XButtonEvent xbutton;
  XMotionEvent xmotion;
  XExposeEvent xexpose;
  Message *newmsg;
  Window twin;
  AGMessage *agmsg[2];
  char buf[10];
  int neww,newh;
  int buttonstat,mousex,mousey;
  KeySym J;
  //  struct aguix_windowlist *tentry;
  if(mode==MES_GET) {
    if(XEventsQueued(dsp,QueuedAfterFlush)==0) return NULL;
  }
  XNextEvent(dsp,&LastEvent);
  newmsg=(Message*)_allocsafe(sizeof(Message));
  newmsg->type=LastEvent.type;
  newmsg->gadgettype=NON_GADGET;
  newmsg->gadget=NULL;
  newmsg->window=0;
  newmsg->time=0;
  if(createagmsg==true) {
    agmsg[0]=(AGMessage*)_allocsafe(sizeof(AGMessage));  // Hauptmessage
    agmsg[0]->type=AG_NONE;
    agmsg[1]=NULL;           // eventl. Nebenmessage;
  }
  switch(LastEvent.type) {
    case MappingNotify:
      XRefreshKeyboardMapping((XMappingEvent*)&LastEvent);
      break;
    case KeyPress:if(createagmsg==true) agmsg[0]->type=AG_KEYPRESSED;
    case KeyRelease:
      XLookupString(&(LastEvent.xkey),buf,9,&J,NULL);
      newmsg->key=J;
      newmsg->keystate=LastEvent.xkey.state;
      newmsg->window=LastEvent.xkey.window;
      newmsg->mousex=LastEvent.xkey.x;
      newmsg->mousey=LastEvent.xkey.y;
      if(createagmsg==true) {
        if(agmsg[0]->type==AG_NONE) agmsg[0]->type=AG_KEYRELEASED;
        agmsg[0]->key.window=newmsg->window;
        agmsg[0]->key.key=J;
        agmsg[0]->key.keystate=newmsg->keystate;
        strcpy(agmsg[0]->key.keyname,buf);
      }
      break;
    case ButtonPress:if(createagmsg==true) agmsg[0]->type=AG_MOUSEPRESSED;
    case ButtonRelease:
      xbutton=LastEvent.xbutton;
      buttonstat=xbutton.button;
      mousex=xbutton.x;
      mousey=xbutton.y;
      if(LastEvent.type==ButtonRelease) buttonstat=0;
      newmsg->button=xbutton.button;
      newmsg->mousex=xbutton.x;
      newmsg->mousey=xbutton.y;
      newmsg->window=LastEvent.xbutton.window;
      newmsg->time=LastEvent.xbutton.time;
      if(createagmsg==true) {
        if(agmsg[0]->type==AG_NONE) {
	  agmsg[0]->type=AG_MOUSERELEASED;
          agmsg[1]=(AGMessage*)_allocsafe(sizeof(AGMessage));
          agmsg[1]->type=AG_MOUSECLICKED;
          agmsg[1]->mouse.window=newmsg->window;
          agmsg[1]->mouse.button=xbutton.button;
          agmsg[1]->mouse.x=newmsg->mousex;
          agmsg[1]->mouse.y=newmsg->mousey;
	  agmsg[1]->mouse.time=newmsg->time;
	}
        agmsg[0]->mouse.window=newmsg->window;
        agmsg[0]->mouse.button=xbutton.button;
        agmsg[0]->mouse.x=newmsg->mousex;
        agmsg[0]->mouse.y=newmsg->mousey;
	agmsg[0]->mouse.time=newmsg->time;
      }
      break;
    case MotionNotify:
      xmotion=LastEvent.xmotion;
      /* I no longer use XQueryPointer here because of error if the window
         this message is for no longer exists and then XQueryPointer fails
	 Instead use queryPointer when get AG_MOUSEMOVE */
      /*Window root,child;
      int rootx,rooty;
      unsigned int button;
      XQueryPointer(dsp,xmotion.window,&root,&child,&rootx,&rooty,&mousex,&mousey,&button);
      */
      newmsg->mousex=xmotion.x;
      newmsg->mousey=xmotion.y;
      newmsg->mouserx=xmotion.x_root;
      newmsg->mousery=xmotion.y_root;
      newmsg->window=xmotion.window;
      if(createagmsg==true) {
        agmsg[0]->type=AG_MOUSEMOVE;
        agmsg[0]->mouse.window=newmsg->window;
	/* button is wrong, must be quest with queryPointer */
        agmsg[0]->mouse.button=0;
        agmsg[0]->mouse.x=newmsg->mousex;
        agmsg[0]->mouse.y=newmsg->mousey;
      }
      break;
    case Expose:
      xexpose=LastEvent.xexpose;
      newmsg->window=xexpose.window;
      if(createagmsg==true) {
        agmsg[0]->type=AG_EXPOSE;
        agmsg[0]->expose.window=newmsg->window;
        agmsg[0]->expose.x=xexpose.x;
        agmsg[0]->expose.y=xexpose.y;
        agmsg[0]->expose.w=xexpose.width;
        agmsg[0]->expose.h=xexpose.height;
      }
      break;
    case ConfigureNotify:
      neww=LastEvent.xconfigure.width;
      newh=LastEvent.xconfigure.height;
      newmsg->window=LastEvent.xconfigure.window;
      newmsg->mousex=LastEvent.xconfigure.x;
      newmsg->mousey=LastEvent.xconfigure.y;
      twin=newmsg->window;
      //!!!TODO: Was soll der Schei hier: firstwin wird eh nicht erstellt, also wenn, dann
      // wins benutzen
      /*      tentry=firstwin;
      while(tentry!=NULL) {
        if((tentry->window->getWindow())==twin) break;
        tentry=tentry->next;
      }
      if(tentry!=NULL) {
        tentry->window->sizeChanged(neww,newh);
	}*/
      break;
    case ClientMessage:
      if(LastEvent.xclient.data.l[0]==(long)WM_delete_window) {
        newmsg->gadgettype=CLOSE_GADGET;
        newmsg->gadget=NULL;
        newmsg->window=LastEvent.xclient.window;
        if(createagmsg==true) {
          agmsg[0]->type=AG_CLOSEWINDOW;
          agmsg[0]->closewindow.window=newmsg->window;
        }
      }
      break;
    case EnterNotify:if(createagmsg==true) agmsg[0]->type=AG_ENTERWINDOW;
    case LeaveNotify:
      newmsg->window=LastEvent.xcrossing.window;
      newmsg->mousex=LastEvent.xcrossing.x;
      newmsg->mousey=LastEvent.xcrossing.y;
      if(createagmsg==true) {
        if(agmsg[0]->type==AG_NONE) agmsg[0]->type=AG_LEAVEWINDOW;
        agmsg[0]->cross.window=newmsg->window;
      }
      break;
    case SelectionClear:
      if(cutstart!=NULL) {
	if( ( LastEvent.xselectionclear.window==cutstart->getWindow() ) &&
	    ( LastEvent.xselection.selection==XA_PRIMARY ) ) {
	  cancelCut();
	}
      }
      break;
    case SelectionNotify:
      if( (LastEvent.xselection.selection==XA_PRIMARY) &&
	  (LastEvent.xselection.property!=None) ) {
	Atom ret_type;
	int ret_format;
	unsigned long ret_len,ret_after;
	unsigned char *ret_prop;
	XGetWindowProperty(dsp,
			   LastEvent.xselection.requestor,
			   LastEvent.xselection.property,
			   0,
			   8192,
			   False,
			   LastEvent.xselection.target,
			   &ret_type,
			   &ret_format,
			   &ret_len,
			   &ret_after,
			   &ret_prop);
	if(ret_len>0) {
	  //	  ret_prop[ret_len-1]=0;
	  if(pastestart!=NULL) pastestart->paste(ret_prop);
	  XFree(ret_prop);
	  XDeleteProperty(dsp,LastEvent.xselection.requestor,LastEvent.xselection.property);
	} else {
	  if(pastestart!=NULL) pastestart->cancelpaste();
	  pastestart=NULL;
	}
      } else {
	if(pastestart!=NULL) pastestart->cancelpaste();
	pastestart=NULL;
      }
      break;
    case SelectionRequest:
      XEvent ev;
      ev.type=SelectionNotify;
      ev.xselection.requestor=LastEvent.xselectionrequest.requestor;
      ev.xselection.selection=LastEvent.xselectionrequest.selection;
      ev.xselection.target=LastEvent.xselectionrequest.target;
      ev.xselection.time=LastEvent.xselectionrequest.time;
      ev.xselection.property=LastEvent.xselectionrequest.property;
      if( ( LastEvent.xselectionrequest.selection==XA_PRIMARY ) &&
	  ( LastEvent.xselectionrequest.target==XA_STRING ) ) {
	XChangeProperty(dsp,
			ev.xselection.requestor,
			ev.xselection.property,
			ev.xselection.target,
			8,
			PropModeReplace,
			(unsigned char*)cutbuffer,
			strlen(cutbuffer));
      } else {
	ev.xselection.property=None;
      }
      XSendEvent(dsp,ev.xselection.requestor,False,0,&ev);
      break;
  }
  if(createagmsg==true) {
    // agmsg in FIFO speichern
    putAGMsg(agmsg[0]);
    if(agmsg[1]!=NULL) putAGMsg(agmsg[1]);
  }
  return newmsg;
}

void AGUIX::buildAGMessage()
{
  XButtonEvent xbutton;
  XMotionEvent xmotion;
  XExposeEvent xexpose;
  AGMessage *agmsg[2];
  char buf[10];
  KeySym J;
  agmsg[0]=(AGMessage*)_allocsafe(sizeof(AGMessage));  // Hauptmessage
  agmsg[0]->type=AG_NONE;
  agmsg[1]=NULL;           // eventl. Nebenmessage;
  switch(LastEvent.type) {
    case MappingNotify:
      XRefreshKeyboardMapping((XMappingEvent*)&LastEvent);
      break;
    case KeyPress:agmsg[0]->type=AG_KEYPRESSED;
    case KeyRelease:
      XLookupString(&(LastEvent.xkey),buf,9,&J,NULL);
      if(agmsg[0]->type==AG_NONE) agmsg[0]->type=AG_KEYRELEASED;
      agmsg[0]->key.window=LastEvent.xkey.window;
      agmsg[0]->key.key=J;
      agmsg[0]->key.keystate=LastEvent.xkey.state;
      strcpy(agmsg[0]->key.keyname,buf);
      break;
    case ButtonPress:agmsg[0]->type=AG_MOUSEPRESSED;
    case ButtonRelease:
      xbutton=LastEvent.xbutton;
      if(agmsg[0]->type==AG_NONE) {
        agmsg[0]->type=AG_MOUSERELEASED;
        agmsg[1]=(AGMessage*)_allocsafe(sizeof(AGMessage));
        agmsg[1]->type=AG_MOUSECLICKED;
        agmsg[1]->mouse.window=LastEvent.xbutton.window;
        agmsg[1]->mouse.button=xbutton.button;
        agmsg[1]->mouse.x=xbutton.x;
        agmsg[1]->mouse.y=xbutton.y;
        agmsg[1]->mouse.time=LastEvent.xbutton.time;
      }
      agmsg[0]->mouse.window=LastEvent.xbutton.window;
      agmsg[0]->mouse.button=xbutton.button;
      agmsg[0]->mouse.x=xbutton.x;
      agmsg[0]->mouse.y=xbutton.y;
      agmsg[0]->mouse.time=LastEvent.xbutton.time;
      break;
    case MotionNotify:
      xmotion=LastEvent.xmotion;
      agmsg[0]->type=AG_MOUSEMOVE;
      agmsg[0]->mouse.window=xmotion.window;
      /* button is wrong, must be quest with queryPointer */
      agmsg[0]->mouse.button=0;
      agmsg[0]->mouse.x=xmotion.x;
      agmsg[0]->mouse.y=xmotion.y;
      break;
    case Expose:
      xexpose=LastEvent.xexpose;
      agmsg[0]->type=AG_EXPOSE;
      agmsg[0]->expose.window=xexpose.window;
      agmsg[0]->expose.x=xexpose.x;
      agmsg[0]->expose.y=xexpose.y;
      agmsg[0]->expose.w=xexpose.width;
      agmsg[0]->expose.h=xexpose.height;
      break;
    case ClientMessage:
      if(LastEvent.xclient.data.l[0]==(long)WM_delete_window) {
        agmsg[0]->type=AG_CLOSEWINDOW;
        agmsg[0]->closewindow.window=LastEvent.xclient.window;
      }
      break;
    case EnterNotify:agmsg[0]->type=AG_ENTERWINDOW;
    case LeaveNotify:
      if(agmsg[0]->type==AG_NONE) agmsg[0]->type=AG_LEAVEWINDOW;
      agmsg[0]->cross.window=LastEvent.xcrossing.window;
      break;
  }
  // agmsg in FIFO speichern
  putAGMsg(agmsg[0]);
  if(agmsg[1]!=NULL) putAGMsg(agmsg[1]);
}

AGMessage *AGUIX::GetMessage(AWindow *parent)
{
  Message *tmsg;
  tmsg=wait4mess(MES_GET,false);
  if(tmsg!=NULL) {
    if(ReactMessage(tmsg,parent)==false) buildAGMessage();
    _freesafe(tmsg);
  }
  return getAGMsg();
}

AGMessage *AGUIX::WaitMessage(AWindow *parent)
{
  while(messages->size()==0) {
    Message *tmsg;
    tmsg=wait4mess(MES_WAIT,false);
    if(tmsg!=NULL) {
      if(ReactMessage(tmsg,parent)==false) buildAGMessage();
      _freesafe(tmsg);
    }
  }
  return getAGMsg();
}

Message *AGUIX::wait4messReact(int mode,bool createagmsg)
{
  Message *tmsg;
  tmsg=wait4mess(mode,createagmsg);
  if(tmsg!=NULL)
    if(tmsg->type==Expose) ReactMessage(tmsg,NULL);
  return tmsg;
}

void AGUIX::copyArea(Drawable source,Drawable dest,int s_x,int s_y,int width,int height,int d_x,int d_y)
{
  XCopyArea(dsp,source,dest,gc,s_x,s_y,width,height,d_x,d_y);
}

void AGUIX::DrawLine(Drawable buffer,int x1,int ty1,int x2,int y2)
{
  XDrawLine(dsp,buffer,gc,x1,ty1,x2,y2);
}

void AGUIX::DrawLine(Drawable buffer,GC tgc,int x1,int ty1,int x2,int y2)
{
  if(tgc==0) DrawLine(buffer,x1,ty1,x2,y2);
  else XDrawLine(dsp,buffer,tgc,x1,ty1,x2,y2);
}

void AGUIX::DrawPoint(Drawable buffer,int px,int py)
{
  XDrawPoint(dsp,buffer,gc,px,py);
}

void AGUIX::DrawPoint(Drawable buffer,GC tgc,int px,int py)
{
  if(tgc==0) DrawPoint(buffer,px,py);
  else XDrawPoint(dsp,buffer,tgc,px,py);
}

bool AGUIX::ReactMessage(Message *msg,AWindow *parent)
{
  // als erstes das Fenster ausfindig machen
  // dann ReactMessage des Fenster aufrufen
  bool returnvalue=false;
  AWindow *win;
  if(msg->type==ClientMessage) {
    return returnvalue;
  }
  if(msg->window!=0) {
    int id;
    id=wins->initEnum();
    win=(AWindow*)wins->getFirstElement(id);
    while((win!=NULL)&&(returnvalue==false)) {
      if((parent==NULL)||(msg->type==Expose)) {
        returnvalue=win->handleMessage(&LastEvent,msg);
      } else {
        if(parent->isParent(win->getWindow(),false)==true) {
	  returnvalue=win->handleMessage(&LastEvent,msg);
        }
      }
      win=(AWindow*)wins->getNextElement(id);
    }
    wins->closeEnum(id);
  }
  return returnvalue;
}

void AGUIX::Flush()
{
  XFlush(dsp);
}

int AGUIX::getargc()
{
  return argc;
}

char **AGUIX::getargv()
{
  return argv;
}

void AGUIX::putAGMsg(AGMessage *msg)
{
  messages->addElement(msg);
}

AGMessage *AGUIX::getAGMsg()
{
  AGMessage *msg=(AGMessage*)messages->getFirstElement();
  messages->removeFirstElement();
  return msg;
}

void AGUIX::ReplyMessage(AGMessage *msg)
{
  _freesafe(msg);
}

void AGUIX::ClearWin(Window win)
{
  XClearWindow(dsp,win);
}

void AGUIX::SetWindowBG(Window win,int color)
{
  if(color<0) color=0;
  if(color<colors) {
    XSetWindowBackground(dsp,win,ColBuf[color]);
  } else {
    XSetWindowBackground(dsp,win,ColBuf[color-1]);
  }
}

void AGUIX::WindowtoBack(Window win)
{
  XLowerWindow(dsp,win);
}

void AGUIX::WindowtoFront(Window win)
{
  XRaiseWindow(dsp,win);
}

int AGUIX::changeColor(int index2,int red,int green,int blue)
{
  XColor col;
#ifdef DEBUG
  printf("changeColor:%d/%d\n",index2,colors);
#endif
  XFreeColors(dsp,cmap,ColBuf+index2,1,0);
  col.flags=DoRed|DoGreen|DoBlue;
  col.red=red*256;
  col.green=green*256;
  col.blue=blue*256;
  if(!XAllocColor(dsp,cmap,&col)) return -1;
  ColBuf[index2]=col.pixel;
  return 0;
}

AGUIXFont::AGUIXFont(AGUIX *parent)
{
  name=NULL;
  gc=0;
  font=NULL;
  aguix=parent;
}

AGUIXFont::~AGUIXFont()
{
  removeFont();
}

void AGUIXFont::removeFont()
{
  if(gc!=0) {
    XFreeGC(aguix->getDisplay(),gc);
    gc=0;
  }
  if(font!=NULL) {
    XFreeFont(aguix->getDisplay(),font);
    font=NULL;
  }
  if(name!=NULL) {
    _freesafe(name);
    name=NULL;
  }
}

int AGUIXFont::setFont(char *fontname)
{
  Display *dsp=aguix->getDisplay();
  int scr=aguix->getScreen();
  unsigned long white=WhitePixel(dsp,scr);
  unsigned long black=BlackPixel(dsp,scr);
  gc=XCreateGC(dsp,RootWindow(dsp,scr),0,0);
  XGCValues gc_values;
  gc_values.graphics_exposures=False;
  gc_values.foreground=black;
  gc_values.background=white;
  gc_values.line_width=0;
  gc_values.cap_style=CapButt;
  font=XLoadQueryFont(dsp,fontname);
  if(font==NULL) {
    XFreeGC(dsp,gc);
    gc=0;
    return 1;
  }
  charHeight=font->ascent+font->descent;
#if 0
  charWidth=font->max_bounds.rbearing-font->min_bounds.lbearing;
#else
  charWidth=XTextWidth(font,"M",1);
#endif
  gc_values.font=font->fid;
  unsigned long gc_valuemask=GCGraphicsExposures|GCForeground|GCBackground|GCLineWidth|GCFont|GCCapStyle;
  XChangeGC(dsp,gc,gc_valuemask,&gc_values);
  if(name!=NULL) _freesafe(name);
  name=dupstring(fontname);
  return 0;
}

const char *AGUIXFont::getName()
{
  return name;
}

GC AGUIXFont::getGC()
{
  return gc;
}

int AGUIXFont::getCharWidth()
{
  return charWidth;
}

int AGUIXFont::getCharHeight()
{
  return charHeight;
}

AGUIXFont *AGUIX::getFont(char *name)
{
  AGUIXFont *tf=new AGUIXFont(this);
  if(tf->setFont(name)!=0) {
    delete tf;
    return NULL;
  }
  fonts->addElement(tf);
  return tf;
}

int AGUIXFont::getBaseline()
{
  return font->ascent;
}

void AGUIX::ExposeHandler(Message *msg)
{
  AWindow *win;
  int id;
  if(msg->window!=0) {
    id=wins->initEnum();
    win=(AWindow*)wins->getFirstElement(id);
    while(win!=NULL) {
      if(msg->type==Expose) {
        win->handleMessage(&LastEvent,msg);
      }
      win=(AWindow*)wins->getNextElement(id);
    }
    wins->closeEnum(id);
  }
}

AWindow *AGUIX::findAWindow(Window win)
{
  AWindow *awin;
  int id=wins->initEnum();
  awin=(AWindow*)wins->getFirstElement(id);
  while(awin!=NULL) {
    if(awin->isParent(win,true)==true) break;
    awin=(AWindow*)wins->getNextElement(id);
  }
  wins->closeEnum(id);
  return awin;
}

char *AGUIX::getNameOfKey(KeySym key,unsigned int mod)
{
  char *tstr,*tstr2;
  tstr2=XKeysymToString(key); // Nicht freilassen
  if(tstr2==NULL) return NULL;
  tstr=dupstring(tstr2);
  if((mod&Mod1Mask)==Mod1Mask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Left Alt + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  if((mod&ControlMask)==ControlMask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Control + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  if((mod&ShiftMask)==ShiftMask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Shift + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  return tstr;
}

Window AGUIX::getGroupWin()
{
  return groupwin;
}

void AGUIX::createGroupWin()
{
  XTextProperty windowname,iconname;
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask;
  XSetWindowAttributes attr;
  char *tstr;
  attr.event_mask=0;
  groupwin=XCreateWindow(dsp,RootWindow(dsp,scr),0,0,10,10,0,getDepth(),InputOutput,vis,mask,&attr);
  if(!groupwin) {
    return;
  }
  XClassHint classhint;
  tstr=getClassname();
  classhint.res_name=tstr;
  classhint.res_class=tstr;
  XSetClassHint(dsp,groupwin,&classhint);
  XSetWMProtocols(dsp,groupwin,getCloseAtom(),1);
  XSizeHints *SizeHints;
  XWMHints *WMHints;
  SizeHints=XAllocSizeHints();
  WMHints=XAllocWMHints();
  SizeHints->flags=PSize;
  SizeHints->width=10;
  SizeHints->height=10;
  WMHints->input=True;

  WMHints->window_group=groupwin;

  WMHints->flags=InputHint|WindowGroupHint;
  XStringListToTextProperty(&tstr,1,&windowname);
  XStringListToTextProperty(&tstr,1,&iconname);
  XSetWMName(dsp,groupwin,&windowname);
  XSetWMIconName(dsp,groupwin,&iconname);
  XSetWMHints(dsp,groupwin,WMHints);
  XSetWMNormalHints(dsp,groupwin,SizeHints);
  XStoreName(dsp,groupwin,tstr);
  XFree(SizeHints);
  XFree(WMHints);
  _freesafe(tstr);
  XSetCommand(dsp,groupwin,getargv(),getargc());
}

void AGUIX::destroyGroupWin()
{
  XDestroyWindow(dsp,groupwin);
  groupwin=0;
}

int
AGUIX::queryPointer(Window win,int *x,int *y)
{
  Window usewin,root,child;
  int root_x,root_y;
  unsigned int keys_buttons;
  if(win==0) usewin=DefaultRootWindow(dsp);
  else usewin=win;
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,&keys_buttons);
  return 0;
}

int
AGUIX::queryPointer(Window win,int *x,int *y,unsigned int *buttons)
{
  Window usewin,root,child;
  int root_x,root_y;
  if(win==0) usewin=DefaultRootWindow(dsp);
  else usewin=win;
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,buttons);
  return 0;
}

int
AGUIX::queryRootPointer(int *x,int *y)
{
  Window usewin,root,child;
  int root_x,root_y;
  unsigned int keys_buttons;
  usewin=DefaultRootWindow(dsp);
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,&keys_buttons);
  return 0;
}

int
AGUIX::setFont(char *newfont)
{
  Flush();
  AGUIXFont *afont=getFont(newfont);
  if(afont==NULL) return 1;
  mainfont=afont;

  XGCValues gc_values;
  XFontStruct *xfont;
  xfont=mainfont->getFontStruct();
  
  CharHeight=mainfont->getCharHeight();
  CharWidth=mainfont->getCharWidth();
  gc_values.font=xfont->fid;
  unsigned long gc_valuemask=GCFont;
  XChangeGC(dsp,gc,gc_valuemask,&gc_values);
  Flush();
  return 0;
}

XFontStruct*
AGUIXFont::getFontStruct()
{
  return font;
}

void
AGUIX::changeColormap()
{
  Colormap newcmap;
  int id;
  AWindow *win;
  if(privatecmap==false) {
#ifdef DEBUG
    printf("go to private cmap\n");
#endif
    newcmap=XCopyColormapAndFree(dsp,cmap);
    cmap=newcmap;
    privatecmap=true;
    id=wins->initEnum();
    win=(AWindow*)wins->getFirstElement(id);
    while(win!=NULL) {
      XSetWindowColormap(dsp,win->getWindow(),cmap);
      win=(AWindow*)wins->getNextElement(id);
    }
    wins->closeEnum(id);
  }
}

Colormap
AGUIX::getColormap()
{
  return cmap;
}

void AGUIX::setCursor(Window win,int type)
{
  if((type>=WAIT_CURSOR)&&(type<MAXCURSORS)) {
    XDefineCursor(dsp,win,cursors[type]);
  }
}

void AGUIX::unsetCursor(Window win)
{
  XUndefineCursor(dsp,win);
}

int AGUIX::startCut(GUIElement *elem,const char *buffer)
{
  XSetSelectionOwner(dsp,XA_PRIMARY,elem->getWindow(),CurrentTime);
  if(elem->getWindow()==XGetSelectionOwner(dsp,XA_PRIMARY)) {
    cutstart=elem;
    if(cutbuffer!=NULL) _freesafe(cutbuffer);
    cutbuffer=dupstring(buffer);
  } else {
    cancelCut();
    return 1;
  }
  return 0;
}

void AGUIX::cancelCut()
{
  if(cutstart!=NULL) cutstart->cancelcut();
  cutstart=NULL;
  /*  if(cutbuffer!=NULL) _freesafe(cutbuffer);
      cutbuffer=NULL;*/
}

int AGUIX::startPaste(GUIElement *elem)
{
  pastestart=elem;
  return 0;
}

bool AGUIX::amiOwner()
{
  if(cutstart!=NULL) return true;
  return false;
}

const char *AGUIX::getCutBuffer()
{
  return cutbuffer;
}

void AGUIX::requestCut(Window win)
{
  Atom myproperty=XInternAtom(dsp,"Worker Paste Property",False);
  XConvertSelection(dsp,
		    XA_PRIMARY,
		    XA_STRING,
		    myproperty,
		    win,
		    CurrentTime);
}

void AGUIX::DrawTriangleFilled(Drawable buffer, int x1, int y1, int x2, int y2, int x3, int y3)
{
  DrawTriangleFilled( buffer, 0, x1, y1, x2, y2, x3, y3 );
}

void AGUIX::DrawTriangleFilled(Drawable buffer, GC tgc, int x1, int y1, int x2, int y2, int x3, int y3)
{
  GC usegc;
  XPoint points[3];

  if( tgc == 0 )
    usegc = gc;
  else
    usegc = tgc;

  points[0].x = x1;
  points[0].y = y1;
  points[1].x = x2;
  points[1].y = y2;
  points[2].x = x3;
  points[2].y = y3;

  XFillPolygon( dsp, buffer, usegc, points, 3, Convex, CoordModeOrigin );
  XDrawLines( dsp, buffer, usegc, points ,3, CoordModeOrigin );
}

void AGUIX::cancelCutPaste(GUIElement *elem)
{
  if(cutstart==elem) {
    cancelCut();
  }
  if(pastestart==elem) {
    pastestart=NULL;
  }
}

bool AGUIX::isDoubleClick(struct timeval *t1,struct timeval *t2)
{
  int dt=0;
  int s,us;
  s=abs(t1->tv_sec-t2->tv_sec);
  us=t1->tv_usec-t2->tv_usec;
  if(us<0) {
    us+=1000000;
    s--;
  }
  dt=us+s*1000000;
  if(dt<350000) return true;
  return false;
}

bool AGUIX::isDoubleClick( Time t1, Time t2)
{
  if ( abs( t2 - t1 ) < 350 ) return true;
  return false;
}

#define ADJUST_SPACE()				\
{						\
  w1 = 0;					\
  for ( i = 0; i < nr; i++ ) {			\
    w1 += elems[i]->getWidth();			\
  }						\
  						\
  sp = wantedWidth - w1 - borderwidth * 2;	\
  						\
  last = borderwidth;				\
  for ( i = 0; i < nr; i++ ) {			\
    widths[i] = elems[i]->getWidth();		\
    xpos[i] = last;				\
    if ( i < ( nr - 1 ) )			\
      tsp = sp / ( nr - 1 - i );		\
    else					\
      tsp = 0;					\
    sp -= tsp;					\
    last += widths[i] + tsp;			\
  }						\
}

#define ARRANGE_MINSPACE()			\
{						\
  last = borderwidth;				\
  for ( i = 0; i < nr; i++ ) {			\
    widths[i] = elems[i]->getWidth();		\
    xpos[i] = last;				\
    last += widths[i];				\
    if ( minSpace > 0 ) last += minSpace;	\
  }						\
}

/*
 * resize and pos all elems to fit given width
 *
 * args:
 *   wantedWidth: width for resize or <1 for any
 *   borderwidth: left and right border
 *                <0 means 0
 *   minSpace: minimal space between elems
 *             <0 means any
 *   maxSpace: maximal space between elems
 *             <0 means any
 *   allowShrink: true when shrinking is allowed
 *   allowStretch: true when stretching is allowed
 *   elems: Array of elements
 *   nr: Arraysize
 *
 * returnvalue:
 *   used width
 * used width = wantedWidth is NOT guaranteed
 *
 */
int AGUIX::scaleElementsW( int wantedWidth,
			   int borderwidth,
			   int minSpace,
			   int maxSpace,
			   bool allowShrink,
			   bool allowStretch,
			   GUIElement **elems,
			   int nr )
{
  //  GUIElement **elems;
  int cw, i, last, nw, snr;
  int *widths, *xpos;
  int w1, w2, sp, tsp;
  int usedWidth = -1;

  // for ,... notation
  // nr = 0;
  // elems = &elem;
  // while ( elems[nr] != NULL )
  //   nr++;

  if ( elems == NULL ) return -1;
  if ( nr < 1 ) return -1;
  if ( ( minSpace >= 0 ) && ( maxSpace >= 0 ) && ( minSpace > maxSpace ) ) return -1;

  widths = new int[nr];
  xpos = new int[nr];

  if ( borderwidth < 0 ) borderwidth = 0;

  // calc the currently needed width
  cw = 2 * borderwidth;
  if ( nr > 1 )
    cw += ( nr - 1 ) * ( ( minSpace < 0 ) ? 0 : minSpace );
  
  for ( i = 0; i < nr; i++ ) {
    cw += elems[i]->getWidth();
  }
  
  if ( nr == 1 ) {
    // special case
    // center the only element
    // 2 possibilities (when cw<wantedWidth):
    //   1.:stretch never
    //   2.:stretch when allowStretch == true
    // currently I use the second one
    if ( wantedWidth < 1 ) {
      // align to border
      widths[0] = elems[0]->getWidth();
      xpos[0] = borderwidth;
    } else {
      if ( cw < wantedWidth ) {
	// stretching needed
	if ( allowStretch == false ) {
	  // place it in the center
	  widths[0] = elems[0]->getWidth();
	  xpos[0] = ( wantedWidth / 2 ) - ( widths[0] / 2 );
	  
	  // we cannot calc the used width later so set it here
	  usedWidth = wantedWidth;
	} else {
	  widths[0] = wantedWidth - 2 * borderwidth;
	  if ( widths[0] < 4 ) widths[0] = 4;
	  xpos[0] = borderwidth;
	}
      } else if ( cw > wantedWidth ) {
	// shrinking needed
	if ( allowShrink == false ) {
	  // not allowed
	  widths[0] = elems[0]->getWidth();
	  xpos[0] = borderwidth;
	} else {
	  widths[0] = wantedWidth - 2 * borderwidth;
	  if ( widths[0] < 4 ) widths[0] = 4;
	  xpos[0] = borderwidth;
	}
      } else {
	// no changes needed
	widths[0] = elems[0]->getWidth();
	xpos[0] = borderwidth;
      }
    }
  } else {
    if ( wantedWidth < 1 ) {
      // means any width so just arrange them with minSpace
      ARRANGE_MINSPACE();
    } else {
      // try to get the elems to fit wantedWidth
      // prio: 1.change space between elems
      //       2.always stretch or shrink, not both
      
      if ( cw == wantedWidth ) {
	// nothing to do
	last = borderwidth;
	for ( i = 0; i < nr; i++ ) {
	  widths[i] = elems[i]->getWidth();
	  xpos[i] = last;
	  last += widths[i];
	  if ( minSpace > 0 ) last += minSpace;
	}
      } else if ( cw < wantedWidth ) {
	// stretching needed
	if ( maxSpace < 0 ) {
	  // no maxSpace set so just adjust space
	  ADJUST_SPACE();
	} else {
	  // first test if we need to stretch when using maxSpace
	  
	  w1 = 2 * borderwidth;
	  if ( nr > 1 )
	    w1 += ( nr - 1 ) * maxSpace;
	  
	  for ( i = 0; i < nr; i++ ) {
	    w1 += elems[i]->getWidth();
	  }
	  // w1 is now the width when using maxSpace
	  
	  if ( w1 > wantedWidth ) {
	    // we can adjust the space
	    ADJUST_SPACE();
	  } else {
	    // maxSpace isn't enough
	    // we have to stretch some elems
	    if ( allowStretch == false ) {
	      // not allowed
	      // we cannot give wantedWidth so use maxSpace
	      last = borderwidth;
	      for ( i = 0; i < nr; i++ ) {
		widths[i] = elems[i]->getWidth();
		xpos[i] = last;
		last += widths[i] + maxSpace;
	      }
	    } else {
	      w1 = wantedWidth - ( nr - 1 ) * maxSpace;
	      w1 -= 2 * borderwidth;
	      
	      // now divide w1 to all elems
	      // but instead of resize all elems
	      // only stretch elems which are smaller then w1/nr
	      w2 = w1 / nr;
	      snr = nr;
	      for ( i = 0; i < nr; i++ ) {
		if ( elems[i]->getWidth() >= w2 ) {
		  w1 -= elems[i]->getWidth();
		  snr--;
		}
	      }
	      // divide w1 to snr elements
	      for ( i = 0; i < nr; i++ ) {
		if ( elems[i]->getWidth() < w2 ) {
		  widths[i] = w1 / snr;
		  w1 -= widths[i];
		  snr--;
		} else {
		  widths[i] = elems[i]->getWidth();
		}
	      }
	      // now calc the xpos
	      last = borderwidth;
	      for ( i = 0; i < nr; i++ ) {
		xpos[i] = last;
		last += widths[i] + maxSpace;
	      }
	    }
	  }
	}
      } else {
	// shrinking needed
	// works different from stretch-case
	// because we calced the width need when using minSpace ( or 0 when <0)
	// so in any case we have to shrink the elems
	// maxSpace doesn't count at all
	if ( allowShrink == false ) {
	  // nothing we can do to fit wantedWidth
	  // just arrange them with minSpace
	  ARRANGE_MINSPACE();
	} else {
	  // again only shrink elems which are not smaller then the needed average
	  w1 = wantedWidth - ( nr - 1 ) * ( minSpace < 0 ? 0 : minSpace );
	  w1 -= 2 * borderwidth;
	  
	  // now divide w1 to all elems
	  // but instead of resize all elems
	  // only shrink elems which are greater then w1/nr
	  w2 = w1 / nr;
	  snr = nr;
	  for ( i = 0; i < nr; i++ ) {
	    if ( elems[i]->getWidth() <= w2 ) {
	      w1 -= elems[i]->getWidth();
	      snr--;
	    }
	  }
	  // divide w1 to snr elements
	  for ( i = 0; i < nr; i++ ) {
	    if ( elems[i]->getWidth() > w2 ) {
	      widths[i] = w1 / snr;
	      
	      // no less then 4 pixel
	      if ( widths[i] < 4 ) widths[i] = 4;
	      
	      w1 -= widths[i];
	      snr--;
	    } else {
	      widths[i] = elems[i]->getWidth();
	    }
	  }
	  // now calc the xpos
	  last = borderwidth;
	  for ( i = 0; i < nr; i++ ) {
	    xpos[i] = last;
	    last += widths[i];
	    if ( minSpace > 0 ) last += minSpace;
	  }
	}
      }
    } 
  }
  
  // now arrange the elements
  for ( i = 0; i < nr; i++ ) {
    elems[i]->move( xpos[i], elems[i]->getY() );
    elems[i]->resize( widths[i], elems[i]->getHeight() );
  }
  
  delete widths;
  delete xpos;

  if ( usedWidth >= 0 ) {
    return usedWidth;
  } else {
    nw = elems[nr - 1]->getX() + elems[nr - 1]->getWidth() + borderwidth;
    return nw;
  }
}

#undef ADJUST_SPACE
#undef ARRANGE_MINSPACE

int AGUIX::centerElementsY( GUIElement *element,
			    GUIElement *center2element )
{
  int h1,h2;

  if ( ( element == NULL ) || ( center2element == NULL ) )
    return -1;

  h1 = element->getHeight();
  h2 = center2element->getHeight();

  element->move( element->getX(),
		 center2element->getY() +
		 h2 / 2 -
		 h1 / 2 );
  return 0;
}

int AGUIX::getRootWindowWidth()
{
  return rootWindowWidth;
}

int AGUIX::getRootWindowHeight()
{
  return rootWindowHeight;
}
