/*
** DocProperties.c, DocProperties.h
**
** Functions to manage the index's Document Properties 
**
** File Created.
** Mark Gaulin 11/24/98
**
** change sprintf to snprintf to avoid corruption, 
** and use MAXSTRLEN from swish.h instead of literal "200"
** SRE 11/17/99
**
** 04/00 Jose ruiz
** storeDocProperties and readNextDocPropEntry modified to store
** the int numbers compressed. This makes integers "portable"
**
** 04/00 Jose Ruiz
** Added sorting results by property
**
*/

#include "swish.h"
#include "file.h"
#include "hash.h"
#include "mem.h"
#include "merge.h"
#include "error.h"
#include "search.h"
#include "string.h"
#include "docprop.h"

/* 04/00 Jose Ruiz */
/* Simple routing for comparing pointers to strings in order to
get an ascending sort with qsort */
int scompasc(const void *s1,const void *s2)
{
        return(strcasecmp((char *)s2,(char *)s1));
}

void freeDocProperties(docProperties)
     struct docPropertyEntry **docProperties;
{
	/* delete the linked list of doc properties */
	struct docPropertyEntry *prop = NULL;

	prop = *docProperties;
	while (prop != NULL)
	{
		struct docPropertyEntry *nextOne = prop->next;
		efree(prop->propValue);
		efree(prop);

		prop = nextOne;
	}

	/* replace the ptr to the head of the list with a NULL */
	*docProperties = NULL;
}

void addDocProperty(docProperties, metaName, propValue)
     struct docPropertyEntry **docProperties;
     int metaName;
     char* propValue;
{
	/* Add the given file/metaName/propValue data to the File object */
	struct docPropertyEntry *docProp = NULL;

	/*printf("File#: %s %03d %s\n", thisFileEntry->filename, metaName, propValue);*/

	/* new prop object */
	docProp = (struct docPropertyEntry *) emalloc(sizeof(struct docPropertyEntry));
	docProp->metaName = metaName;
	docProp->propValue = (char *) estrdup(propValue);

	/* insert at head of file objects list of properties */
	docProp->next = *docProperties;	/* "*docProperties" is the ptr to the head of the list */
	*docProperties = docProp;	/* update head-of-list ptr */
}

void storeDocProperties(docProperties, fp)
     struct docPropertyEntry *docProperties;
     FILE *fp;
{
	/*
	 * Dump the document properties into the index file 
	 * The format is:
	 *	<PropID:int><PropValueLen:int><PropValue:not-null-terminated>
	 *	  ...
	 *	<PropID:int><PropValueLen:int><PropValue:not-null-terminated>
	 *	<EndofList:null char>
	 * All ints are compressed to save space. Jose Ruiz 04/00
	 *
	 * The list is terminated with a PropID with a value of zero
	 */
	int propID;
	int len;

	while (docProperties != NULL)
	{
		/* the length of the property value */
		len = strlen(docProperties->propValue);
		if (len > 0)
		{
			/* the ID of the property */
			propID = docProperties->metaName;
			/* Do not store 0!! - compress does not like them */
			compress(++propID,fp);
			/* including the length will make retrieval faster */
			compress(len,fp);
			fwrite(docProperties->propValue, len, 1, fp);
		}

		docProperties = docProperties->next;
	}

	/* set is terminated by a "zero" ID  */
	fputc(0,fp);
}

static char* readNextDocPropEntry(fp, metaName, targetMetaName)
      FILE* fp;
      int* metaName;
      int targetMetaName;
{
	/* read one entry and return it; also set the metaName.
 	 * if targetMetaName is zero then return values for all entries.
	 * if targetMetaName is non-zero than only return the value
	 * for the property matching that value.
	 * In all cases, metaName will be zero when the end of the
	 * set is reached.
	 */
	static char* propValueBuf = NULL;
	static int propValueBufLen = 0;
	

	int tempPropID;
	int len;
	long propPos;	/* file pos */

	uncompress(tempPropID,fp);
	if(tempPropID) tempPropID--;

	*metaName = tempPropID;

	if (tempPropID == 0)
		return NULL;		/* end of list */

	/* grab the string length */
	uncompress(len,fp);

	if ((targetMetaName != 0) && (tempPropID != (short int) targetMetaName))
	{
		/* we were looking for something specific, and this is not it */
		/* move to the next property */
		propPos = ftell(fp);
		fseek(fp, propPos+len, 0);
		return "";
	}
	else
	{
		/* return the value */
		if (propValueBufLen < len+1)
		{
			/* allocate buffer for prop value */
			/* the buffer will be reused on the next call */
			if(propValueBufLen) efree(propValueBuf);
			propValueBufLen = len+100;
			propValueBuf = (char *) emalloc(propValueBufLen);
		}
		fread(propValueBuf, len, 1, fp);
		propValueBuf[len]='\0';
		return propValueBuf;
	}
}

void fetchDocProperties(docProperties, fp)
     struct docPropertyEntry **docProperties;
     FILE *fp;
{
	/*
	 * Read the docProperties section that the file pointer is
	 * currently pointing to.
	 * If docProperties is NULL then throw away the results,
	 * which has the desired side effect of moving the file
	 * pointer to the next file entry in the index
	 */

	char* tempPropValue;
	int tempMetaName;
	int targetMetaName = 0;	/* 0 = no target; return all data */

	if (docProperties == NULL)
	{
		targetMetaName = -1;	/* invalid target; matches nothing */
	}
	else
	{
		*docProperties = NULL;	/* initialize linked list - empty */
	}

	/* read all of the properties */
	tempPropValue = readNextDocPropEntry(fp, &tempMetaName, targetMetaName);
	while (tempMetaName > 0)
	{
		if (docProperties != NULL)
		{
			/* add the entry to the list of properties */
			addDocProperty(docProperties, tempMetaName, tempPropValue);
		}

		tempPropValue = readNextDocPropEntry(fp, &tempMetaName, targetMetaName);
	}
}


char* lookupDocPropertyValue(metaName, propPos, fp)
     int metaName;
     long propPos;	/* from struct sortresult.propPos */
     FILE *fp;
{
	/*
	 * Returns the string containing the document's
	 * property value, or an empty string if it was not found.
	 */
	char* tempPropValue;
	int tempMetaName;

	fseek(fp, propPos, 0);

	tempPropValue = readNextDocPropEntry(fp, &tempMetaName, metaName);
	while (tempMetaName > 0)
	{
		/* a match? */
		if (tempMetaName == metaName)
			return tempPropValue;
		tempPropValue = readNextDocPropEntry(fp, &tempMetaName, metaName);
	}

	return "";
}

static int numPropertiesToDisplay = 0;
static char* propNameToDisplay[MAX_PROPS_TO_DISPLAY];
static int propIDToDisplay[MAX_PROPS_TO_DISPLAY];

static int numPropertiesToSort = 0;
static char* propNameToSort[MAX_PROPS_TO_SORT];
static int propIDToSort[MAX_PROPS_TO_SORT];

void addSearchResultDisplayProperty(propName)
     char* propName;
{
	/* add a property to the list of properties that will be displayed */
	if (numPropertiesToDisplay < MAX_PROPS_TO_DISPLAY)
	{
		propNameToDisplay[numPropertiesToDisplay++] = propName;
	}
	else
	{
		progerr("Too many properties to display.");
	}
}

void addSearchResultSortProperty(propName)
     char* propName;
{
	/* add a property to the list of properties that will be displayed */
	if (numPropertiesToSort < MAX_PROPS_TO_SORT)
	{
		propNameToSort[numPropertiesToSort++] = propName;
	}
	else
	{
		progerr("Too many properties to sort.");
	}
}


void initSortResultProperties()
{
int i;
        if (numPropertiesToSort == 0)
                return;
        for (i = 0; i<numPropertiesToSort; i++)
        {
                printf("%s %d: %s\n", SORTDOCPROPHEADER, i+1, propNameToSort[i]);
                makeItLow(propNameToSort[i]);
                propIDToSort[i] = getMetaName(propNameToSort[i]);
                if (propIDToSort[i] == 1)
                {
                        snprintf(errorstr, MAXSTRLEN, "Unknown Sort property name \"%s\"", propNameToSort[i]);
                        progerr(errorstr);
                        return;
                }
        }

}

int isSortProp()
{
	return numPropertiesToSort;
}

void initSearchResultProperties()
{
	/* lookup selected property names */
	int i;

	if (numPropertiesToDisplay == 0)
		return;
	for (i = 0; i<numPropertiesToDisplay; i++)
	{
		printf("%s %d: %s\n", DOCPROPHEADER, i+1, propNameToDisplay[i]);
		/*strlwr(propNameToDisplay[i]);*/
		makeItLow(propNameToDisplay[i]);
		propIDToDisplay[i] = getMetaName(propNameToDisplay[i]);
		if (propIDToDisplay[i] == 1)
		{
			snprintf(errorstr, MAXSTRLEN, "Unknown property name \"%s\"", propNameToDisplay[i]);
			progerr(errorstr);
			return;
		}
	}
}

void printSearchResultProperties(prop)
char **prop;
{
	int i;

	if (numPropertiesToDisplay == 0)
		return;

	for (i = 0; i<numPropertiesToDisplay; i++)
	{
		char* propValue;
		propValue = prop[i];
		
		if (useCustomOutputDelimiter)
			printf("%s", customOutputDelimiter);
		else
			printf(" \"");	/* default is to quote the string, with leading space */

		/* print value, handling newlines and quotes */
		while (*propValue)

		{
			if (*propValue == '\n')
				printf(" ");
			else if (*propValue == '\"')	/* should not happen */
				printf("&quot;");
			else
				printf("%c", *propValue);
			propValue++;
		}
		printf("%s", propValue);

		if (!useCustomOutputDelimiter)
			printf("\"");	/* default is to quote the string */
		efree(prop[i]); /* Property is no longer needed */
	}
}

void getResultProperties(propPos, props, fp)
     long propPos;	/* from struct sortresult.propPos */
     char **props;      /* Array to Store properties */
     FILE *fp;
{
	int i;

	if (numPropertiesToDisplay == 0)
		return;

	for (i = 0; i<numPropertiesToDisplay; i++)
		props[i] = estrdup(lookupDocPropertyValue(propIDToDisplay[i], propPos, fp));

}

void swapDocPropertyMetaNames(docProperties, metaFile)
     struct docPropertyEntry *docProperties;
     struct metaMergeEntry* metaFile;
{
	/* swap metaName values for properties */
	while (docProperties)
	{
		struct metaMergeEntry* metaFileTemp;
		/* scan the metaFile list to get the new metaName value */
		metaFileTemp = metaFile;
		while (metaFileTemp)
		{
			if (docProperties->metaName == metaFileTemp->oldIndex)
			{
				docProperties->metaName = metaFileTemp->newIndex;
				break;
			}

			metaFileTemp = metaFileTemp->next;
		}
		docProperties = docProperties->next;
	}
}

/* Jose Ruiz 04/00
** Sort results by property
*/
struct result *sortresultsbyproperty(rp, structure, fp)
struct result *rp;
int structure;
FILE *fp;
{ 
int i, j, k, l;
unsigned char *ptmp,*ptmp2;
char *ps;
struct result *pv;
struct result *rtmp;
struct result *sortresultlist;
long propPos;
char *fileInfo;
int MaxWide,maxWide[MAX_PROPS_TO_SORT],len;
		/* Trivial case */
	if (!rp) return NULL;
	initSortResultProperties();
	for (i = 0; i<numPropertiesToSort; i++) maxWide[i] = 0;
	sortresultlist = NULL;
		/* Compute results */
	for(i=0,rtmp=rp;rtmp;rtmp = rtmp->next) {
		if (rtmp->structure & structure) {
			i++;
			fileInfo = lookupfile(rtmp->filenum, fp, &propPos);
			rtmp->propPos = propPos;
			for (l = 0; l<numPropertiesToSort; l++){
				rtmp->propSort[l] = estrdup(lookupDocPropertyValue(propIDToSort[l], propPos, fp));
				len=strlen(rtmp->propSort[l]);
				if(len > maxWide[l]) maxWide[l] = len;
			}
		}
	}
		/* Another trivial case */
	if (!i) return NULL;

	for (k = 0,MaxWide=0; k<numPropertiesToSort; k++) MaxWide+=maxWide[k];
		/* We need to compute array wide */
	j=MaxWide+1+sizeof(void *);
		/* Compute array size */
	ptmp=(void *)emalloc((j*i));
		/* Build an array with the elements to compare
			 and pointers to data */
	for(ptmp2=ptmp,rtmp=rp;rtmp;rtmp = rtmp->next) {
		if (rtmp->structure & structure) {
			ps=(char *)ptmp2;
			memset(ps,1,MaxWide); /* padded with 1 */
			for(l=0,k=0;l<numPropertiesToSort; l++){
				len=strlen(rtmp->propSort[l]);
				if(len)memcpy(ps+k,rtmp->propSort[l],len);
				k+=maxWide[l];
			}
			ps[MaxWide]='\0';
			ptmp2+=(MaxWide+1);
			memcpy((char *)ptmp2,(char *)&rtmp,sizeof(struct result *));
			ptmp2+=sizeof(void *);
		}
	}
		/* Sort them */
	qsort(ptmp,i,j,&scompasc);
		/* Build the list */
	for(j=0,ptmp2=ptmp;j<i;j++){
		ps=(char *)ptmp2;
		ptmp2+=(MaxWide + 1);
		memcpy((char *)&pv,(char *)ptmp2,sizeof(struct result *));
		ptmp2+=sizeof(void *);
		pv->fileInfo = estrdup(lookupfile(pv->filenum, fp, &pv->propPos));
		sortresultlist = (struct result *) addsortresult(sortresultlist, pv);
	}
		/* Free the memory of the array */
	efree(ptmp);
		/* Free memory for propSort */
	for(ptmp2=ptmp,rtmp=rp;rtmp;rtmp = rtmp->next) 
		if (rtmp->structure & structure) 
			for(i=0;i<numPropertiesToSort; i++)
				efree(rtmp->propSort[i]);

	return sortresultlist;
}

