/*
 * Linked list functions
 *
 * Copyright IBM Corp. 2001, 2010
 * Author(s): Carsten Otte (cotte@de.ibm.com)
 *            Michael Holzheu <holzheu@linux.vnet.ibm.com>
 */

#ifndef LIST_H
#define LIST_H

#include <stddef.h>

struct list {
	struct list *next, *prev;
};

#define EMPTY_LIST(list) { &(list), &(list) }

/*
 * Add entry to begining of list
 */
static inline void list_add(struct list *entry, struct list *head)
{
	entry->next = head->next;
	entry->next->prev = entry;
	head->next = entry;
	entry->prev = head;
}

/*
 * Add entry to end of list
 */
static inline void list_add_end(struct list *entry, struct list *head)
{
	entry->prev = head->prev;
	entry->prev->next = entry;
	head->prev = entry;
	entry->next = head;
}

/*
 * Remove entry
 */
static inline void list_del(struct list *entry)
{
	entry->next->prev = entry->prev;
	entry->prev->next = entry->next;
	entry->next = entry;
	entry->prev = entry;
}

/*
 * Check if list is empty
 */
static inline int list_is_empty(struct list *head)
{
	if ((head->next == head) && (head->prev == head))
		return 1;
	else
		return 0;
}

/*
 * Initialize list
 */
static inline void list_init(struct list *head)
{
	head->next = head;
	head->prev = head;
}

#define list_entry(ptr, type, member) ({ \
	const typeof(((type *) 0)->member) *__member_ptr = (ptr); \
	(type *)((char *)__member_ptr - offsetof(type, member) ); })

#define list_entry_first(ptr, type, member) \
	list_entry((ptr)->next, type, member)

#define list_entry_next(ptr, type, member) \
	list_entry((ptr)->next, type, member)

#define list_entry_prev(ptr, type, member) \
	list_entry((ptr)->prev, type, member)

/*
 * List iterators
 */
#define list_get(entry, type, member) \
	((type *)((char *)(entry)-(unsigned long)(&((type *)0)->member)))

#define list_iterate(i, head, member)				\
	for (i = list_get((head)->next, typeof(*i), member);	\
	     &i->member != (head);				\
	     i = list_get(i->member.next, typeof(*i), member))

#define list_iterate_safe(i, head, member, n)			\
	for (i = list_get((head)->next, typeof(*i), member),	\
	     n = list_get(i->member.next, typeof(*i), member);	\
	     &i->member != (head);				\
	     i = n,						\
	     n = list_get(n->member.next, typeof(*n), member))

#endif /* LIST_H */
