/*
 * RTF-to-LaTeX2e translation writer code.
 * (c) 1999 Ujwal S. Sathyam
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

# include	<stdio.h>
# include	<string.h>
# include	<stdlib.h>
# include	<ctype.h>

# include	"rtf.h"
# include	"tokenscan.h"
# include 	"cole.h"
# include	"rtf2LaTeX2e.h"
# if RTF2LATEX2E_MAC
	# include	"wmf2pict.h"
	# include	"FullPath.h"
# endif
#include "eqn.h"

char	outputMapName[255];
# define	prefFileName	"r2l-pref"
# define	MAX_BLANK_LINES		2

char texMapQualifier[rtfBufSiz];



void RTFSetOutputStream (FILE	*stream);
int Figure2eps (char *inFile, char *outFile);
void JPEGtoEPS(pictureStruct *picturePtr);

long groupLevel = 0; /* a counter for keeping track of opening and closing braces */
extern FILE *ifp, *ofp;

# define EQUATION_OFFSET 35
# define UNKNOWN_EQUATION_OFFSET 28

# define	NumberOfPreferences 16

char *preferenceList[] = {		 "outputMapFile",
								 "ignoreRulerSettings",
								 "paperWidth",
								 "leftMargin",
								 "rightMargin",
								 "ignoreColor",
								 "ignoreParagraphAlignment",
								 "ignoreTextStyle",
								 "ignoreHypertext",
								 "ignoreSpacing",
								 "convertEquations",
								 "pict2eps_translate",
								 "pict2eps_fonts",
								 "pict2eps_preview",
								 "fileCreator",
								 "swpMode"};
float preferenceValue[NumberOfPreferences];



static long codePage;
char	fileCreator[10];


static char	*outMap[rtfSC_MaxChar];

#define NumberOfR2LMappings	11
static char *r2lMap[NumberOfR2LMappings];
static boolean	r2lMapPresent = true;
char *r2lList[] = {"documentclass",
				   "bold",
				   "nobold",
				   "italic",
				   "noitalic",
				   "underline",
				   "smallcaps",
				   "heading1",
				   "heading2",
				   "heading3",
				   "table"};
static char *boldString, *noBoldString, *italicString, *noItalicString;
static char *underlineString, *smallcapsString;
static char	*documentclassString, *heading1String, *heading2String, *heading3String;
static char *tableString;
static short itemNumber;

				   

static FILE	*ostream;
static textStyleStruct textStyle;
static pictureStruct picture;
static equationStruct oleEquation;
static tableStruct table;

static boolean dblQuoteLeft = false;
static boolean wroteCellHeader = false;
static boolean startFootnoteText = false;
static boolean continueTextStyle = false;
static boolean lineIsBlank = true;




/*
 * Initialize the writer. Basically reads the output map.
 */

void
WriterInit (void)
{
	/* read character map files */
	if (RTFReadCharSetMap ("cp1252.map", rtfCS1252) == 0)
		RTFPanic ("Cannot read character set map file cp1252.map for code page 1252!\n");
	if (RTFReadCharSetMap ("cp1250.map", rtfCS1250) == 0)
		RTFPanic ("Cannot read character set map file cp1250.map for code page 1250!\n");
	if (RTFReadCharSetMap ("cp1254.map", rtfCS1254) == 0)
		RTFPanic ("Cannot read character set map file cp1254.map for code page 1254!\n");
	if (RTFReadCharSetMap ("applemac.map", rtfCSMac) == 0)
		RTFPanic ("Cannot read character set map file applemac.map!\n");
	if (RTFReadCharSetMap ("cp437.map", rtfCS437) == 0)
		RTFPanic ("Cannot read character set map file cp437.map for code page 437!\n");
	if (RTFReadCharSetMap ("cp850.map", rtfCS850) == 0)
		RTFPanic ("Cannot read character set map file cp850.map for code page 850!\n");
	
	/* read preferences */
	if (ReadPrefFile (prefFileName) == 0)
		RTFPanic ("Cannot read preferences file %s", prefFileName);
	
	/* read output map file TeX-map */
	strcpy(texMapQualifier, "");
	
	/* if a TeX-map file was not specified, set it to the default */
	if(strcmp(outputMapName, "") == 0)
		strcpy(outputMapName, "TeX-map");
	
	if (RTFReadOutputMap (outputMapName, outMap, 1) == 0)
		RTFPanic ("Cannot read output map %s", outputMapName);

	/* read r2l mapping file if present */
	if (ReadR2LMap () == 0)
		r2lMapPresent = false;
				
}

static short
ReadR2LMap (void)
{
FILE	*f;
char	buf[rtfBufSiz];
char	*name, *seq;
short	stdCode;
short	i;
TSScanner	scanner;
char		*scanEscape;
char		*fn = "ReadR2LMap";

	/* clobber current mapping */
	for (i = 0; i < NumberOfR2LMappings; i++)
	{
		RTFFree (r2lMap[i]);
		r2lMap[i] = (char *) NULL;
	}


	if ((f = RTFOpenLibFile ("r2l-map", "r")) == (FILE *) NULL)
		return (0);

	/*
	 * Turn off scanner's backslash escape mechanism while reading
	 * file.  Restore it later.
	 */
	TSGetScanner (&scanner);
	scanEscape = scanner.scanEscape;
	scanner.scanEscape = "";
	TSSetScanner (&scanner);

	/* read file */

	while (fgets (buf, (int) sizeof (buf), f) != (char *) NULL)
	{
		if(buf[0] == '#')	/* skip comment lines */
			continue;
		TSScanInit (buf);
		if ((name = TSScan ()) == (char *) NULL)
			continue;	/* skip blank lines */
				
		if ((stdCode = R2LItem (name)) < 0)
		{
			RTFMsg ("%s: unknown preference: %s\n", fn, name);
			continue;
		}
		if ((seq = TSScan ()) == (char *) NULL)
		{
			RTFMsg ("%s: malformed output sequence line for preference %s\n", 
			        fn, name);
			continue;
		}
		if ((seq = RTFStrSave (seq)) == (char *) NULL)
			RTFPanic ("%s: out of memory", fn);
		r2lMap[stdCode] = seq;
	}
	scanner.scanEscape = scanEscape;
	TSSetScanner (&scanner);
	fclose(f);
/*	
	for (i = 0; i < NumberOfR2LMappings; i++)
		printf ("%s\n", r2lMap[i]);
*/	
	return (1);
}

static short
R2LItem (char *name)
{
short i;

	for (i = 0; i < NumberOfR2LMappings; i++)
	{
		if (strcmp (name, r2lList[i]) == 0)
			return (i);
	}
	printf ("R2LItem: invalid preference %s!\n", name);
	return (-1);
}

short
ReadPrefFile (char *file)
{
char	*fn = "ReadPrefFile";
FILE	*f;
char	buf[rtfBufSiz];
char	*name, *seq;
TSScanner	scanner;
char		*scanEscape;
int		whichPref;
short	i;

	/* initialize preferences */
	for (i = 0; i < NumberOfPreferences; i++)
		preferenceValue[i] = 0;

#if	(RTF2LATEX2E_UNIX || RTF2LATEX2E_DOS) /* attach path to rtf2latex2e */
	if ((f = RTFOpenLibFile (file, "r")) == (FILE *) NULL)
			{
				RTFMsg ("can't open file %s\n", file);
				return (0);
			}
#else
	if ((f = fopen (file, "r")) == (FILE *) NULL)
	{
		sprintf(buf, ":pref:%s", file);
		if ((f = fopen (buf, "r")) == (FILE *) NULL)
		{
			RTFMsg ("can't open file %s\n", file);
			return (0);
		}	
	}
#endif			


	/*
	 * Turn off scanner's backslash escape mechanism while reading
	 * file.  Restore it later.
	 */
	TSGetScanner (&scanner);
	scanEscape = scanner.scanEscape;
	scanner.scanEscape = "";
	TSSetScanner (&scanner);

	/* read file */

	while (fgets (buf, (int) sizeof (buf), f) != (char *) NULL)
	{
		if(buf[0] == '#')	/* skip comment lines */
			continue;
		TSScanInit (buf);
		if ((name = TSScan ()) == (char *) NULL)
			continue;	/* skip blank lines */
		if ((whichPref = GetPreferenceNum(name)) < 0)
		{
			RTFMsg ("%s: unkown preference: %s\n", fn, name);
			continue;
		}
		if ((seq = TSScan ()) == (char *) NULL || !IsValidPref (name, seq))
		{
			RTFMsg ("%s: malformed preference setting for %s\n", fn, name);
			continue;
		}

		if (strcmp (name, "fileCreator") == 0)
		{
			strcpy (fileCreator, seq);
			strcpy (seq, "true");
		}
		
		if (strcmp (name, "outputMapFile") == 0)
		{
			if(!strcmp(outputMapName, ""))
				strcpy (outputMapName, seq);
			strcpy (seq, "true");
		}

		if (strcmp(seq, "true") == 0)
			strcpy(seq, "1");
		else if (strcmp(seq, "false") == 0)
			strcpy(seq, "0");
		
		preferenceValue[whichPref] = atof(seq);
	}
	scanner.scanEscape = scanEscape;
	TSSetScanner (&scanner);
	if (fclose(f) != 0) printf(" error closing pref file %s\n", file);
	
	/* set the preferences here to values in file or to default */
	if ((page.width = preferenceValue[GetPreferenceNum("paperWidth")]) == 0)
		page.width = 8.5;
	if ((page.leftMargin = preferenceValue[GetPreferenceNum("leftMargin")]) == 0)
		page.leftMargin = 1.0;
	if ((page.rightMargin = preferenceValue[GetPreferenceNum("rightMargin")]) == 0)
		page.rightMargin = 1.0;
	
	return (1);

}

static boolean
IsValidPref (char *name, char *pref)
{
short i;
	
	if (!strcmp(name, "fileCreator") || !strcmp(name, "outputMapFile"))
		return true;

	if (strcmp(pref, "true") == 0 || strcmp(pref, "false") == 0)
		return true;
		
	for (i = 0; i < strlen(pref); i++)
		if (!(isdigit(pref[i]) || pref[i] =='.'))
			return false;
	
	return true;
}


int
GetPreferenceNum (char *name)
{
short i;

	for (i = 0; i < NumberOfPreferences; i++)
	{
		if (strcmp (name, preferenceList[i]) == 0)
			return (i);
	}
	printf ("GetPreferenceNum: invalid preference %s!\n", name);
	return (-1);

}

/*
 * Prepares output TeX file for each input RTF file. 
 * Sets globals and installs callbacks.
 */
int
BeginLaTeXFile (void)
{
	/* set some globals */
	
	RTFSetDefaultFont (-1);
	codePage = 0;
	wrapCount = 0;
	charAttrCount = 0;
	groupLevel = 0;
	paragraph.alignment = left;
	textStyle.newStyle = 0;
	lastCharWasLineBreak = true;
	seenLeftDoubleQuotes = false;
	wroteBeginDocument = false;
	spaceCount = 0;
	blankLineCount = 0;
	suppressLineBreak = true;
	continueTextStyle = false;
	writingHeading1 = false;
	writingHeading2 = false;
	writingHeading3 = false;
	insideFootnote = false;
	justWroteFootnote = false;
	insideHyperlink = false;
	paragraph.lineSpacing = singleSpace;
	paragraph.newStyle = false;
	paragraph.parbox = false;
	paragraph.wroteAlignment = false;
	paragraph.wroteSpacing = false;
	section.newStyle = false;
	section.cols = 1;
	dblQuoteLeft = false;
	lineIsBlank = true;
	
	if (preferenceValue[GetPreferenceNum("ignoreColor")])
		requireColorPackage = false;
	else
		requireColorPackage = true;
	requireSetspacePackage = false;
	requireTablePackage = false;
	requireGraphicxPackage = false;
	requireAmsSymbPackage = false;
	requireMultiColPackage = false;
	requireUlemPackage = false;
	requireHyperrefPackage = false;
	requireMultirowPackage = false;
	requireAmsMathPackage = false;


	picture.count = 0;
	picture.type = unknownPict;
	oleEquation.count = 0;
	object.class = unknownObj;
	object.word97 = 0;
	table.inside = false;
	table.cellCount = 0;
	table.cellInfo = (cell *)NULL;
	table.cellMergePar = none;
    table.multiCol = false;
    table.multiRow = false;
	InitializeTextStyle ();
	

	/* install class callbacks */
	RTFSetClassCallback (rtfText, TextClass);
	RTFSetClassCallback (rtfControl, ControlClass);
	
	/* install destination callbacks */
	RTFSetDestinationCallback (rtfColorTbl, WriteColours);
	RTFSetDestinationCallback (rtfParNumTextAfter, SkipGroup);
	RTFSetDestinationCallback (rtfParNumTextBefore, SkipGroup);
	RTFSetDestinationCallback (rtfFieldInst, ReadFieldInst);
	RTFSetDestinationCallback (rtfObject, ReadObject);
	RTFSetDestinationCallback (rtfWord97Object, ReadWord97Object);
	RTFSetDestinationCallback (rtfWord97NoPicture, SkipGroup);
	RTFSetDestinationCallback (rtfRevisionTbl, SkipGroup);
	RTFSetDestinationCallback (rtfPict, ReadPicture);
	RTFSetDestinationCallback (rtfFootnote, ReadFootnote);
	RTFSetDestinationCallback (rtfBookmarkStart, SkipGroup);
	RTFSetDestinationCallback (rtfBookmarkEnd, SkipGroup);
	RTFSetDestinationCallback (rtfDataField, SkipGroup);
	RTFSetDestinationCallback (rtfTemplate, SkipGroup);
	RTFSetDestinationCallback (rtfDocvar, SkipGroup);
	RTFSetDestinationCallback (rtfFchars, SkipGroup);
	RTFSetDestinationCallback (rtfLchars, SkipGroup);
	RTFSetDestinationCallback (rtfPgdsctbl, SkipGroup);
	
	
	/* use r2l-map if present */
	/* defaults */
	documentclassString = "{article}";
	boldString = "\\textbf{";
	noBoldString = "\\textmd{";
	italicString = "\\textit{";
	noItalicString = "\\textup{";
	underlineString = "{\\underline {";
	smallcapsString = "\\textsc{";
	heading1String = "\\section*{";
	heading2String = "\\subsection*{";
	heading3String = "\\subsubsection*{";
	tableString = "longtable";

	if (r2lMapPresent)
	{
		itemNumber = R2LItem ("documentclass");
		if (r2lMap[itemNumber] != (char *)NULL)
			documentclassString = r2lMap[itemNumber];
		itemNumber = R2LItem ("bold");
		if (r2lMap[itemNumber] != (char *)NULL)
			boldString = r2lMap[itemNumber];
		itemNumber = R2LItem ("nobold");
		if (r2lMap[itemNumber] != (char *)NULL)
			noBoldString = r2lMap[itemNumber];
		itemNumber = R2LItem ("italic");
		if (r2lMap[itemNumber] != (char *)NULL)
			italicString = r2lMap[itemNumber];
		itemNumber = R2LItem ("noitalic");
		if (r2lMap[itemNumber] != (char *)NULL)
			noItalicString = r2lMap[itemNumber];
		itemNumber = R2LItem ("underline");
		if (r2lMap[itemNumber] != (char *)NULL)
			underlineString = r2lMap[itemNumber];
		itemNumber = R2LItem ("smallcaps");
		if (r2lMap[itemNumber] != (char *)NULL)
			smallcapsString = r2lMap[itemNumber];
		itemNumber = R2LItem ("heading1");
		if (r2lMap[itemNumber] != (char *)NULL)
			heading1String = r2lMap[itemNumber];
		itemNumber = R2LItem ("heading2");
		if (r2lMap[itemNumber] != (char *)NULL)
			heading2String = r2lMap[itemNumber];
		itemNumber = R2LItem ("heading3");
		if (r2lMap[itemNumber] != (char *)NULL)
			heading3String = r2lMap[itemNumber];
		itemNumber = R2LItem ("table");
		if (r2lMap[itemNumber] != (char *)NULL)
			tableString = r2lMap[itemNumber];
	}

	/* write LaTeX header */
	WriteLaTeXHeader ();
	
	return (1);
}



/* make sure we write this all important stuff. This routine is called 
 * whenever something is written to the output file.
 */
static void CheckForBeginDocument (void)
{
char buf[100];

	if (!wroteBeginDocument)
	{
		
		if (!preferenceValue[GetPreferenceNum("ignoreRulerSettings")])
		{
			sprintf (buf, "\\setlength{\\oddsidemargin}{%3.2fin}\n", 
			        1-page.leftMargin);		
			PutLitStr (buf);
			sprintf (buf, "\\setlength{\\evensidemargin}{%3.2fin}\n", 
			         1-page.rightMargin);		
			PutLitStr (buf);
			sprintf (buf, "\\setlength{\\textwidth}{%3.2fin}\n", 
				 	page.width - page.leftMargin - page.rightMargin);		
			PutLitStr (buf);
			blankLineCount++;
		}
		if (blankLineCount < MAX_BLANK_LINES)
		{
			InsertNewLine ();
			blankLineCount++;
		}
		
		PutLitStr ("\\begin{document}\n\n");
		blankLineCount++;
	}

	wroteBeginDocument = true;
}


static void
InitializeGroupLevels (void)
{
	textStyle.boldGroupLevel = 0;
	textStyle.noBoldGroupLevel = 0;
	textStyle.italicGroupLevel = 0;
	textStyle.noItalicGroupLevel = 0;
	textStyle.underlinedGroupLevel = 0;
	textStyle.noUnderlinedGroupLevel = 0;
	textStyle.dbUnderlinedGroupLevel = 0;
	textStyle.noDbUnderlinedGroupLevel = 0;
	textStyle.foreColorGroupLevel = 0;
	textStyle.backColorGroupLevel = 0;
	textStyle.subScriptGroupLevel = 0;
	textStyle.noSubScriptGroupLevel = 0;
	textStyle.superScriptGroupLevel = 0;
	textStyle.noSuperScriptGroupLevel = 0;
	textStyle.fontSizeGroupLevel = 0;
	textStyle.allCapsGroupLevel = 0;
	textStyle.smallCapsGroupLevel = 0;
	textStyle.open = false;

}

static void
StoreGroupLevels (void)
{
	temp1 = textStyle.boldGroupLevel;
	temp2 = textStyle.italicGroupLevel;
	temp3 = textStyle.underlinedGroupLevel;
	temp4 = textStyle.foreColorGroupLevel;
	temp5 = textStyle.backColorGroupLevel;
	temp6 = textStyle.subScriptGroupLevel;
	temp7 = textStyle.superScriptGroupLevel;
	temp8 = textStyle.fontSizeGroupLevel;
	temp10 = textStyle.noBoldGroupLevel;
	temp11 = textStyle.noItalicGroupLevel;
	rememberFontSize = textStyle.fontSize;
}

static void 
RestoreGroupLevels (void)
{
	textStyle.boldGroupLevel = temp1;
	textStyle.italicGroupLevel = temp2;
	textStyle.underlinedGroupLevel = temp3;
	textStyle.foreColorGroupLevel = temp4;
	textStyle.backColorGroupLevel = temp5;
	textStyle.subScriptGroupLevel = temp6;
	textStyle.superScriptGroupLevel = temp7;
	textStyle.fontSizeGroupLevel = temp8;
	textStyle.noBoldGroupLevel = temp10;
	textStyle.noItalicGroupLevel = temp11;
	textStyle.fontSize = rememberFontSize;

}


/*
 * This function initializes the text style. Pretty much self-explanatory.
 */

static void
InitializeTextStyle (void)
{
	textStyle.foreColor = -1;
	textStyle.backColor = -1;
	textStyle.fontSize = normalSize;
	textStyle.newStyle = false;
	textStyle.wroteBold = false;
	textStyle.wroteNoBold = false;
	textStyle.wroteItalic = false;
	textStyle.wroteNoItalic = false;
	textStyle.wroteUnderlined = false;
	textStyle.wroteDbUnderlined = false;
	textStyle.wroteShadowed = false;
	textStyle.wroteAllcaps = false;
	textStyle.wroteSmallCaps = false;
	textStyle.wroteForeColor = false;
	textStyle.wroteFontSize = false;
	textStyle.wroteSubScript = false;
	textStyle.wroteNoSubScript = false;
	textStyle.wroteSuperScript = false;
	textStyle.wroteNoSuperScript = false;

	paragraph.firstIndent = 0;
	paragraph.leftIndent = 0;
	paragraph.rightIndent = 0;
	
	InitializeGroupLevels ();
}

/*
 * This function stores the text style. Pretty much self-explanatory.
 */

static void
SetTextStyle (void)
{
	if (insideHyperlink)
		return;
			
	switch (rtfMinor)
	{
		case rtfSmallCaps:
			textStyle.smallCapsGroupLevel = groupLevel;
			break;
		case rtfAllCaps:
			textStyle.allCapsGroupLevel = groupLevel;
			break;
		case rtfItalic:
			if (rtfParam != 0)
			{
				textStyle.italicGroupLevel = groupLevel;
				if (textStyle.noItalicGroupLevel == groupLevel)
					textStyle.noItalicGroupLevel = 0;
			}
			else if(textStyle.wroteItalic)
			{
				textStyle.noItalicGroupLevel = groupLevel;
				if (textStyle.italicGroupLevel == groupLevel)
				{
					StoreGroupLevels ();
					CheckForCharAttr ();
					RestoreGroupLevels ();
					textStyle.italicGroupLevel = 0;
				}
			}
			break;
		case rtfBold:
			if (rtfParam != 0)
			{
				textStyle.boldGroupLevel = groupLevel;
				if (textStyle.noBoldGroupLevel == groupLevel)
					textStyle.noBoldGroupLevel = 0;
			}
			else if(textStyle.wroteBold)
			{
				textStyle.noBoldGroupLevel = groupLevel;
				if (textStyle.boldGroupLevel == groupLevel)
				{
					StoreGroupLevels ();
					CheckForCharAttr ();
					RestoreGroupLevels ();				
					textStyle.boldGroupLevel = 0;
				}
			}
			break;
		case rtfUnderline:
			if (rtfParam != 0)
			{
				textStyle.underlinedGroupLevel = groupLevel;
				if (textStyle.noUnderlinedGroupLevel == groupLevel)
					textStyle.noUnderlinedGroupLevel = 0;
			}
			else if(textStyle.wroteUnderlined)
			{
				textStyle.noUnderlinedGroupLevel = groupLevel;
				if (textStyle.underlinedGroupLevel == groupLevel)
				{
					StoreGroupLevels ();
					CheckForCharAttr ();
					RestoreGroupLevels ();				
					textStyle.underlinedGroupLevel = 0;
				}
			}
			break;
		case rtfDbUnderline:
			if (rtfParam != 0)
			{
				textStyle.dbUnderlinedGroupLevel = groupLevel;
				if (textStyle.noDbUnderlinedGroupLevel == groupLevel)
					textStyle.noDbUnderlinedGroupLevel = 0;
			}
			else if(textStyle.wroteDbUnderlined)
			{
				textStyle.noDbUnderlinedGroupLevel = groupLevel;
				if (textStyle.dbUnderlinedGroupLevel == groupLevel)
				{
					StoreGroupLevels ();
					CheckForCharAttr ();
					RestoreGroupLevels ();				
					textStyle.dbUnderlinedGroupLevel = 0;
				}
			}
			break;
		case rtfForeColor:
			textStyle.foreColor = rtfParam;
			textStyle.foreColorGroupLevel = groupLevel;
			textStyle.foreColor = rtfParam;
			break;
		case rtfSubScrShrink: case rtfSubScript:
			if (rtfParam != 0)
			{
				textStyle.subScriptGroupLevel = groupLevel;
				if (textStyle.noSubScriptGroupLevel == groupLevel)
					textStyle.noSubScriptGroupLevel = 0;
			}
			else if(textStyle.wroteSubScript)
			{
				textStyle.noSubScriptGroupLevel = groupLevel;
				if (textStyle.subScriptGroupLevel == groupLevel)
				{
					StoreGroupLevels ();
					CheckForCharAttr ();
					RestoreGroupLevels ();				
					textStyle.subScriptGroupLevel = 0;
				}
			}
//			textStyle.subScriptGroupLevel = groupLevel;
			break;
		case rtfSuperScript:
			if (rtfParam != 0)
			{
				textStyle.superScriptGroupLevel = groupLevel;
				if (textStyle.noSuperScriptGroupLevel == groupLevel)
					textStyle.noSuperScriptGroupLevel = 0;
			}
			else if(textStyle.wroteSuperScript)
			{
				textStyle.noSuperScriptGroupLevel = groupLevel;
				if (textStyle.superScriptGroupLevel == groupLevel)
				{
					StoreGroupLevels ();
					CheckForCharAttr ();
					RestoreGroupLevels ();				
					textStyle.superScriptGroupLevel = 0;
				}
			}
			break;		
		case rtfSuperScrShrink:
			RTFGetToken ();
			if (strcmp (rtfTextBuf, "\\chftn") != 0 && !RTFCheckCM(rtfGroup, rtfEndGroup))
				textStyle.superScriptGroupLevel = groupLevel;
			RTFUngetToken ();	
			break;
		case rtfFontSize:
			if (rtfParam <= 18)
			{
				textStyle.fontSize = smallSize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			if (rtfParam <= 14)
			{
				textStyle.fontSize = footNoteSize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			if (rtfParam <= 12)
			{
				textStyle.fontSize = scriptSize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			if (rtfParam >= 28)
			{
				textStyle.fontSize = largeSize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			if (rtfParam >= 32)
			{
				textStyle.fontSize = LargeSize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			if (rtfParam >= 36)
			{
				textStyle.fontSize = LARGESize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			if (rtfParam >= 48)
			{
				textStyle.fontSize = giganticSize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			if (rtfParam >= 72)
			{
				textStyle.fontSize = GiganticSize;
				textStyle.fontSizeGroupLevel = groupLevel;
			}
			break;
		
	}
		textStyle.newStyle = true;

}



/*
 * This function writes the text style. Pretty much self-explanatory.
 */

static void
WriteTextStyle (void)
{
char buf[rtfBufSiz];
int italicGL, noItalicGL;
int	boldGL, noBoldGL;
int	underlinedGL, noUnderlinedGL;
int	dbUnderlinedGL, noDbUnderlinedGL;
int foreColorGL, backColorGL;
int	subScriptGL, superScriptGL;
int	fontSizeGL, allCapsGL, smallCapsGL;

	if (writingHeading1 || writingHeading2 || writingHeading3 || insideHyperlink)
		return;

	italicGL = textStyle.italicGroupLevel;
	noItalicGL = textStyle.noItalicGroupLevel;
	boldGL = textStyle.boldGroupLevel;
	noBoldGL = textStyle.noBoldGroupLevel;
	underlinedGL = textStyle.underlinedGroupLevel;
	noUnderlinedGL = textStyle.noUnderlinedGroupLevel;
	dbUnderlinedGL = textStyle.dbUnderlinedGroupLevel;
	noDbUnderlinedGL = textStyle.noDbUnderlinedGroupLevel;
	foreColorGL = textStyle.foreColorGroupLevel;
	backColorGL = textStyle.backColorGroupLevel;
	subScriptGL = textStyle.subScriptGroupLevel;
	superScriptGL = textStyle.superScriptGroupLevel;
	fontSizeGL = textStyle.fontSizeGroupLevel;
	allCapsGL = textStyle.allCapsGroupLevel;
	smallCapsGL = textStyle.smallCapsGroupLevel;
	
	CheckForBeginDocument ();

	if (rtfClass == 2 && rtfMajor == 32) PutLitChar (' ');
	

	if (foreColorGL <= groupLevel && foreColorGL > 0 && (textStyle.foreColor) > 0 
		&& !(textStyle.wroteForeColor))
	{
		
		sprintf(buf, "{\\color{color%ld} ", (long)(textStyle.foreColor));
		PutLitStr (buf);	
		wrapCount += 17;
		charAttrCount++;
		textStyle.wroteForeColor = true;

	}
	if (subScriptGL <= groupLevel && subScriptGL > 0 && !(textStyle.wroteSubScript))
	{
			if (boldGL <= superScriptGL && boldGL > 0 && textStyle.wroteBold)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.wroteBold = false;
			}
			if (italicGL <= superScriptGL && italicGL > 0 && textStyle.wroteItalic)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.wroteItalic = false;
			}
		CheckForMathMode ();
		PutLitStr ("_{");
		wrapCount += 2;
		charAttrCount += 1;
		mathMode = 1;
		textStyle.wroteSubScript = true;
		textStyle.open = true;
	}
	if (superScriptGL <= groupLevel && superScriptGL > 0 && 
	    !(textStyle.wroteSuperScript))
	{
			if (boldGL <= superScriptGL && boldGL > 0 && textStyle.wroteBold)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.wroteBold = false;
			}
			if (italicGL <= superScriptGL && italicGL > 0 && textStyle.wroteItalic)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.wroteItalic = false;
			}
		CheckForMathMode ();
		PutLitStr ("^{");
		wrapCount += 2;
		charAttrCount += 1;
		mathMode = 1;
		textStyle.wroteSuperScript = true;
		textStyle.open = true;
	}
	if (boldGL <= groupLevel && boldGL > 0 && !(textStyle.wroteBold))
	{
		if (!mathMode)
		{
			PutLitStr (boldString);
			wrapCount += strlen (boldString);
			charAttrCount += CountCharInString (boldString, '{');
		}
		else
		{
			PutLitStr ("\\mathbf{");
			wrapCount += 8;
			charAttrCount++;
		}
		textStyle.wroteBold = true;
//		textStyle.open = true;
	}
	if (noBoldGL <= groupLevel && noBoldGL > 0 && !(textStyle.wroteNoBold)
		&& textStyle.wroteBold && !mathMode)
	{
		PutLitStr (noBoldString);
		wrapCount += strlen (noBoldString);
		charAttrCount += CountCharInString (noBoldString, '{');
		textStyle.wroteNoBold = true;
	}
	if (italicGL <= groupLevel && italicGL > 0 && !(textStyle.wroteItalic))
	{
		if (!mathMode)
		{
			PutLitStr (italicString);
			wrapCount += strlen (italicString);
			charAttrCount += CountCharInString (italicString, '{');
		}
		else
		{
			PutLitStr ("\\mathit{");
			wrapCount += 8;
			charAttrCount++;
		}
		textStyle.wroteItalic = true;
//		textStyle.open = true;
	}
	if (noItalicGL <= groupLevel && noItalicGL > 0 && !(textStyle.wroteNoItalic)
		&& textStyle.wroteItalic && !mathMode)
	{
		PutLitStr (noItalicString);
		wrapCount += strlen (noItalicString);
		charAttrCount += CountCharInString (noItalicString, '{');
		textStyle.wroteNoItalic = true;
	}
	if (underlinedGL <= groupLevel && underlinedGL > 0 
		&& !(textStyle.wroteUnderlined) && !mathMode)
	{
		PutLitStr (underlineString);
		wrapCount += strlen (underlineString);
		charAttrCount += CountCharInString (underlineString, '{');
		textStyle.wroteUnderlined = true;
		textStyle.open = true;

	}
	if (dbUnderlinedGL <= groupLevel && dbUnderlinedGL > 0 
		&& !(textStyle.wroteDbUnderlined) && !mathMode)
	{
		PutLitStr ("{\\uuline {");
		wrapCount += 12;
		charAttrCount += 2;
		textStyle.wroteDbUnderlined = true;
		textStyle.open = true;
		requireUlemPackage = true;
	}
	if (smallCapsGL <= groupLevel && smallCapsGL > 0 && !(textStyle.wroteSmallCaps) && !mathMode)
	{
		PutLitStr (smallcapsString);
		wrapCount += strlen (smallcapsString);
		charAttrCount += CountCharInString (smallcapsString, '{');
		textStyle.wroteSmallCaps = true;
	}
	if (textStyle.fontSize != normalSize && !mathMode && !(textStyle.wroteFontSize) && !mathMode)
	{
		sprintf (buf, "{%s ", fontSizeList[textStyle.fontSize]);
		PutLitStr (buf);
		charAttrCount++;
		wrapCount += 7;
		textStyle.wroteFontSize = true;
	}
		
}

/*
 * This function writes the LaTeX header and includes some basic packages.
 */
static void
WriteLaTeXHeader (void)
{
	char	*fn = "WriteLaTeXHeader";
	FILE	*f;
	char	buf[rtfBufSiz];
	char	*item;
	TSScanner	scanner;
	char		*scanEscape;
	int i, j;
	boolean preambleFilePresent = true;

	if ((f = RTFOpenLibFile ("r2l-head", "r")) == (FILE *) NULL)
		preambleFilePresent = false;
			
	PutLitStr ("%&LaTeX\n");
	PutLitStr ("\\documentclass");
	PutLitStr (documentclassString);
	InsertNewLine ();

	if (preambleFilePresent)
	{
		/*
		 * Turn off scanner's backslash escape mechanism while reading
		 * file.  Restore it later.
		 */
		TSGetScanner (&scanner);
		scanEscape = scanner.scanEscape;
		scanner.scanEscape = "";
		TSSetScanner (&scanner);
	
		while (fgets (buf, (int) sizeof (buf), f) != (char *) NULL)
		{
			if(buf[0] == '#')	/* skip comment lines */
				continue;
			TSScanInit (buf);
			if ((item = TSScan ()) == (char *) NULL)
				continue;	/* skip blank lines */
			
			PutLitStr (item);
			PutLitChar ('\n');
		}
	
		scanner.scanEscape = scanEscape;
		TSSetScanner (&scanner);
	
		if (fclose(f) != 0) printf(" error closing preamble file\n");
	}
	
	/* insert TeX-map qualifier */
	if(strcmp(texMapQualifier, "") != 0)
		PutLitStr(texMapQualifier);
	
	/* to come back and write necessary \usepackage{...}
     * commands if necessary */
     packagePos = ftell (ofp); 
	
	for (j = 0; j < PACKAGES + 1; j++)
	{
		for (i = 0; i < 100; i++) 
			PutLitChar (' ');
		PutLitChar ('\n');
	}
	
	
	PutLitStr ("\\newcommand{\\tab}{\\hspace{5mm}}\n\n");
	blankLineCount++;

}

static void
WriteColours (void)
{
	/* This function is in the reader. I made it public. */
	ReadColorTbl (); 
	if (requireColorPackage)
		DefineColors ();
}

/* 
 * This function reads colors from the color table and defines them in 
 * LaTeX format to be included in the 
 * LaTeX preamble. This is done after the color table has been read (see above).
*/

static void
DefineColors (void)
{
RTFColor *rtfColorPtr;
int i = 1;
float textRed, textBlue, textGreen;
char buf[rtfBufSiz];
	

	while ((rtfColorPtr = RTFGetColor (i)) != (RTFColor *)NULL) {
		textRed = (float)((rtfColorPtr->rtfCRed)/255.0);
		textGreen = (float)((rtfColorPtr->rtfCGreen)/255.0);
		textBlue = (float)((rtfColorPtr->rtfCBlue)/255.0);
		
		sprintf(buf, "\\definecolor{color%d}{rgb}{%1.3f,%1.3f,%1.3f}\n", 
		        i, textRed, textGreen, textBlue);
		PutLitStr (buf);
		wrapCount = 0;
		
		i++;
	}

}

static void
WriteLaTeXFooter (void)
{
	
	DoParagraphCleanUp ();
	DoSectionCleanUp ();

	
	PutLitStr("\n\n\\end{document}\n"); 
	fseek (ofp, packagePos, 0);
	
	/* load required packages */
	if (requireSetspacePackage)
		PutLitStr ("\\usepackage{setspace}\n");	
	if (requireColorPackage)
		PutLitStr ("\\usepackage{color}\n");
	if (requireGraphicxPackage)
		PutLitStr ("\\usepackage{graphicx}\n");
	if (requireTablePackage)
		PutLitStr ("\\usepackage{longtable}\n");
	if (requireMultirowPackage)
		PutLitStr ("\\usepackage{multirow}\n");
	if (requireAmsSymbPackage)
		PutLitStr ("\\usepackage{amssymb}\n");
	if (requireMultiColPackage)
		PutLitStr ("\\usepackage{multicol}\n");
	if (requireUlemPackage)
		PutLitStr ("\\usepackage{ulem}\n");
	if (requireAmsMathPackage)
		PutLitStr ("\\usepackage{amsmath}\n");
	if (requireHyperrefPackage)
	{
		PutLitStr ("\\usepackage{hyperref}\n");
		PutLitStr ("\\def\\R2Lurl#1#2{\\mbox{\\href{#1}{\\tt #2}}}");
	}

	
	fseek (ofp, 0L, 2); /* go back to end of stream */
	
	

}

/* This function causes the present group to be skipped.
 */
static void
SkipGroup (void)
{

	RTFSkipGroup ();
	RTFRouteToken ();
}


/*
 * Write out a character.  rtfMajor contains the input character, rtfMinor
 * contains the corresponding standard character code.
 *
 * If the input character isn't in the charset map, try to print some
 * representation of it.
 */

static void
TextClass (void)
{

	CheckForBeginDocument ();

	if (section.newStyle && !table.inside)
		WriteSectionStyle ();
	if (paragraph.newStyle && !table.inside)
		WriteParagraphStyle ();
	if (textStyle.newStyle && rtfMinor != rtfSC_space)
	{
		WriteTextStyle ();
		textStyle.newStyle = false;
	}

	if (insideFootnote && !startFootnoteText && rtfMinor == rtfSC_space)
		return;

	if (insideFootnote)
		startFootnoteText = true;
	else
		justWroteFootnote = false;
	
	lastCharWasLineBreak = false;
	

	if (rtfMinor == rtfSC_space)
	{
		spaceCount++;
		
 		/* 
 		 * Here we check whether there is an end group '}' just after the space
 		 * If so, we process that first and then process the space. This is mainly
 		 * for aesthetic reasons to more produce "hello \textbf{and} hello" instead 
 		 * of "hello \textbf{and }hello". Just some of Scott's nitpicking...
 		 */
 		while (RTFCheckCMM (rtfText, 32, rtfSC_space))
 			RTFGetToken ();
		if (RTFCheckCM (rtfGroup, rtfEndGroup))
		{
			RTFRouteToken ();
			PutStdChar (rtfSC_space);
			return;
		}
		else if (rtfClass == rtfText && rtfMinor == rtfSC_period)
		{
			RTFUngetToken ();
			return; /* sometimes RTF stupidly puts a space before a period */
		}
		else
		{
			RTFUngetToken ();
			RTFSetToken (rtfText, 32, rtfSC_space, rtfNoParam, " ");
		} 
	}		
	else 
	{
		spaceCount = 0;
 		suppressLineBreak = false;
 		lineIsBlank = false; 
	}
			
	if (spaceCount < 5)
	{
		if (rtfMinor >= rtfSC_therefore && rtfMinor < rtfSC_currency)
			requireAmsSymbPackage = true;
			
		/* check for a few special characters if we are translating hyperlinks */
		if(insideHyperlink)
		{
			switch(rtfMinor)
			{
				case rtfSC_underscore:
					PutLitChar('_');
					return;
			}
		}	
		
		PutStdChar (rtfMinor);
		WrapText ();
		if (rtfMinor != rtfSC_space)
			blankLineCount = 0;
	}
	
}



/* This function make sure that the output TeX file does not 
 * contain one very, very long line of text.
 */
static void
WrapText (void)
{
	if (wrapCount >= WRAP_LIMIT) {
		if (rtfMinor == rtfSC_space) {
			PutLitChar('\n');
			wrapCount = 0;
			return;
		}	
	}
	wrapCount++;

}


/* decides what to do when a control word is encountered */
static void
ControlClass (void)
{
	
	switch (rtfMajor)
	{
	case rtfDefFont:
		RTFSetDefaultFont (rtfParam);
	case rtfFontAttr:
		switch (rtfMinor)
		{
			case rtfAnsiCodePage: case rtfFontCodePage:
				codePage = rtfParam;
				break;
		}
		break;
	case rtfDestination:
		Destination ();
		break;
	case rtfSpecialChar:
		SpecialChar ();
		break;
	case rtfCharAttr:
		CharAttr ();
		break;
	case rtfListAttr:
		RTFSkipGroup ();
		RTFRouteToken ();
		break;
	case rtfTblAttr: /* trigger for reading table */
		if (rtfMinor == rtfRowDef && !(table.inside)) 
		{
			DoTable (); /* if we are not already inside a table, get into it */
		}
		else 
			DoTableAttr (); /* if we are already inside 
			                 * a table, set table attributes */
		break;
	case rtfParAttr:
		ParAttr ();
		break;
	case rtfSectAttr:
		SectAttr ();
		break;
	case rtfWord97ObjAttr:
		if (rtfMinor == rtfShapeName || rtfMinor == rtfShapeValue)
			SkipGroup ();
		break;
	}
}



/*
 * This function makes sure any braces opened for text styles are closed;
 * rather messy, but it will do for now.
 */
void
CheckForCharAttr (void)
{
int italicGL, noItalicGL;
int	boldGL, noBoldGL;
int	underlinedGL, noUnderlinedGL;
int	dbUnderlinedGL, noDbUnderlinedGL;
int foreColorGL, backColorGL;
int	subScriptGL, superScriptGL;
int noSubScriptGL, noSuperScriptGL;
int	fontSizeGL, allCapsGL, smallCapsGL;
int i;

	
	italicGL = textStyle.italicGroupLevel;
	noItalicGL = textStyle.noItalicGroupLevel;
	boldGL = textStyle.boldGroupLevel;
	noBoldGL = textStyle.noBoldGroupLevel;
	underlinedGL = textStyle.underlinedGroupLevel;
	noUnderlinedGL = textStyle.noUnderlinedGroupLevel;
	dbUnderlinedGL = textStyle.dbUnderlinedGroupLevel;
	noDbUnderlinedGL = textStyle.noDbUnderlinedGroupLevel;
	foreColorGL = textStyle.foreColorGroupLevel;
	backColorGL = textStyle.backColorGroupLevel;
	subScriptGL = textStyle.subScriptGroupLevel;
	noSubScriptGL = textStyle.noSubScriptGroupLevel;
	superScriptGL = textStyle.superScriptGroupLevel;
	noSuperScriptGL = textStyle.noSuperScriptGroupLevel;
	fontSizeGL = textStyle.fontSizeGroupLevel;
	allCapsGL = textStyle.allCapsGroupLevel;
	smallCapsGL = textStyle.smallCapsGroupLevel;


		if (smallCapsGL == groupLevel && smallCapsGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteSmallCaps)
			{
				for (i = 0; i < CountCharInString (smallcapsString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (smallcapsString, '{');
			}
			textStyle.smallCapsGroupLevel = 0;
			textStyle.wroteSmallCaps = false;
		}
		if (underlinedGL == groupLevel && underlinedGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteUnderlined)
			{
				for (i = 0; i < CountCharInString (underlineString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (underlineString, '{');
			}
			textStyle.underlinedGroupLevel = 0;
			textStyle.wroteUnderlined = false;
		}	
		if (dbUnderlinedGL == groupLevel && dbUnderlinedGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteDbUnderlined)
			{
				PutLitStr ("}}");
				charAttrCount -= 2;
			}
			textStyle.dbUnderlinedGroupLevel = 0;
			textStyle.wroteDbUnderlined = false;
		}	
		if (noItalicGL == groupLevel && noItalicGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteNoItalic)
			{
				for (i = 0; i < CountCharInString (noItalicString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (noItalicString, '{');
			}
			textStyle.noItalicGroupLevel = 0;
			textStyle.wroteNoItalic = false;
		}	
		if (italicGL == groupLevel && italicGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteItalic)
			{
				for (i = 0; i < CountCharInString (italicString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (italicString, '{');
			}
			textStyle.italicGroupLevel = 0;
			textStyle.wroteItalic = false;
		}	
		if (noBoldGL == groupLevel && noBoldGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteNoBold)
			{
				for (i = 0; i < CountCharInString (noBoldString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (noBoldString, '{');
			}
			textStyle.noBoldGroupLevel = 0;
			textStyle.wroteNoBold = false;
		}	
		if (boldGL == groupLevel && boldGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteBold)
			{
				for (i = 0; i < CountCharInString (boldString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (boldString, '{');
			}
			textStyle.boldGroupLevel = 0;
			textStyle.wroteBold = false;
		}	
		if (superScriptGL == groupLevel && superScriptGL > 0)
		{
			if (boldGL < superScriptGL && boldGL > 0 && textStyle.wroteBold)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.newStyle = true;
				textStyle.wroteBold = false;
			}
			if (italicGL < superScriptGL && italicGL > 0 && textStyle.wroteItalic)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.newStyle = true;
				textStyle.wroteItalic = false;
			}
			if (charAttrCount > 0)
			{
				PutLitStr ("}");
				charAttrCount -= 1;
				if (mathMode) 
				{
					PutLitChar ('$');
					mathMode = 0;
				}
			}
			textStyle.superScriptGroupLevel = 0;
			textStyle.wroteSuperScript = false;
		}
		if (subScriptGL == groupLevel && subScriptGL > 0)
		{
			if (boldGL < superScriptGL && boldGL > 0 && textStyle.wroteBold)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.newStyle = true;
				textStyle.wroteBold = false;
			}
			if (italicGL < superScriptGL && italicGL > 0 && textStyle.wroteItalic)
			{
				PutLitStr ("}");
				charAttrCount--;
				textStyle.newStyle = true;
				textStyle.wroteItalic = false;
			}
			if (charAttrCount > 0)
			{
				PutLitStr ("}");
				charAttrCount -= 1;
				if (mathMode) 
				{
					PutLitChar ('$');
					mathMode = 0;
				}
			}
			textStyle.subScriptGroupLevel = 0;
			textStyle.wroteSubScript = false;
		}	
		if (foreColorGL == groupLevel && foreColorGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteForeColor)
			{
				PutLitStr ("}");
				charAttrCount--;
			}
			textStyle.foreColorGroupLevel = 0;
			textStyle.wroteForeColor = false;
		}	
		if (backColorGL == groupLevel && backColorGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteBackColor)
			{
				PutLitStr ("}");
				charAttrCount--;
			}
			textStyle.backColorGroupLevel = 0;
			textStyle.wroteBackColor = false;
		}
		if (fontSizeGL == groupLevel && fontSizeGL > 0)
		{
			if (charAttrCount > 0 && textStyle.wroteFontSize)
			{
				PutLitStr ("}");
				charAttrCount--;
			}
			textStyle.fontSizeGroupLevel = 0;
			textStyle.fontSize = normalSize;
			textStyle.wroteFontSize = false;
		}
		if (charAttrCount == 0)
			textStyle.open = false;
			
}

static void 
ForceCloseCharAttr (void)
{
int i;
			if (charAttrCount > 0 && textStyle.wroteSmallCaps)
			{
				for (i = 0; i < CountCharInString (smallcapsString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (smallcapsString, '{');
			}
			textStyle.smallCapsGroupLevel = 0;
			textStyle.wroteSmallCaps = false;

			if (charAttrCount > 0 && textStyle.wroteUnderlined)
			{
				for (i = 0; i < CountCharInString (underlineString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (underlineString, '{');
			}
			textStyle.underlinedGroupLevel = 0;
			textStyle.wroteUnderlined = false;

			if (charAttrCount > 0 && textStyle.wroteNoItalic)
			{
				for (i = 0; i < CountCharInString (noItalicString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (noItalicString, '{');
			}
			textStyle.noItalicGroupLevel = 0;
			textStyle.wroteNoItalic = false;

			if (charAttrCount > 0 && textStyle.wroteItalic)
			{
				for (i = 0; i < CountCharInString (italicString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (italicString, '{');
			}
			textStyle.italicGroupLevel = 0;
			textStyle.wroteItalic = false;

			if (charAttrCount > 0 && textStyle.wroteBold)
			{
				for (i = 0; i < CountCharInString (boldString, '{'); i++)
					PutLitStr ("}");
				charAttrCount -= CountCharInString (boldString, '{');
			}
			textStyle.boldGroupLevel = 0;
			textStyle.wroteBold = false;

			if (charAttrCount > 0 && textStyle.wroteSuperScript)
			{
				PutLitStr ("}");
				charAttrCount -= 1;
				if (mathMode)
					PutLitChar ('$');
				mathMode = 0;
				textStyle.superScriptGroupLevel = 0;
				textStyle.wroteSuperScript = false;
			}
		
			if (charAttrCount > 0 && textStyle.wroteSubScript)
			{
				PutLitStr ("}");
				charAttrCount -= 1;
				if (mathMode)
					PutLitChar ('$');
				mathMode = 0;
				textStyle.subScriptGroupLevel = 0;
				textStyle.wroteSubScript = false;
			}

			if (charAttrCount > 0 && textStyle.wroteForeColor)
			{
				PutLitStr ("}");
				charAttrCount--;
			}
			textStyle.foreColorGroupLevel = 0;
			textStyle.wroteForeColor = false;

			if (charAttrCount > 0 && textStyle.wroteBackColor)
			{
				PutLitStr ("}");
				charAttrCount--;
			}
			textStyle.backColorGroupLevel = 0;
			textStyle.wroteBackColor = false;

			if (charAttrCount > 0 && textStyle.wroteFontSize)
			{
				PutLitStr ("}");
				charAttrCount--;
			}
			textStyle.fontSizeGroupLevel = 0;
			textStyle.fontSize = normalSize;
			textStyle.wroteFontSize = false;

	for (i = 0; i < charAttrCount; i++)
		PutLitStr ("}");
	charAttrCount = 0;


}


/*
 * This function notices destinations that should be ignored
 * and skips to their ends.  This keeps, for instance, picture
 * data from being considered as plain text.
 */

static void
Destination (void)
{
	switch (rtfMinor)
	{
	case rtfFNContSep:
	case rtfFNContNotice:
	case rtfInfo:
	case rtfIndexRange:
	case rtfITitle:
	case rtfISubject:
	case rtfIAuthor:
	case rtfIOperator:
	case rtfIKeywords:
	case rtfIComment:
	case rtfIVersion:
	case rtfIDoccomm:
		RTFSkipGroup ();
		break;
	}
}

/* reads footnote. Just puts a footnote wrapper around whatever is
 * inside the footnote. Table footnotes are skipped for now
 * until I figure out a way that TeX likes.
 */
static void
ReadFootnote (void)
{
int	footnoteGL;
	
	suppressLineBreak = true;
	startFootnoteText = false;
	
	
	/* if we are in a table, skip the footnote. Later, I'll try to implement
	 * it in a way LaTeX likes
	 */
	/* 
	if (table.inside)
	{
		RTFMsg ("* Warning! Skipping footnote in table\n");
		SkipGroup ();
		return;
	}
	*/
	StoreGroupLevels ();
	
	ForceCloseCharAttr ();
	
	footnoteGL = groupLevel;
	PutLitStr ("\\footnote{");
	wrapCount += 10;
	insideFootnote = true;
	while (groupLevel >= footnoteGL)
	{
		RTFGetToken ();
		RTFRouteToken ();
	}
	if (insideFootnote)
		PutLitStr ("}");
	insideFootnote = false;
	justWroteFootnote = true;
	wrapCount++;
	
	RestoreGroupLevels ();
	textStyle.newStyle = true;
	suppressLineBreak = false;

}


/*
 * The reason these use the rtfSC_xxx thingies instead of just writing
 * out ' ', '-', '"', etc., is so that the mapping for these characters
 * can be controlled by the TeX-map file.
 */

static void
SpecialChar (void)
{
int i;

	switch (rtfMinor)
	{
	case rtfSect:
	case rtfLine:
	case rtfPar:
		if (!suppressLineBreak)
			CheckForParagraph ();
		break;
/*	case rtfCurFNote:
		if (mathMode)
		{
			PutLitChar ('}');
			PutLitChar ('$');
			mathMode = 0;
			charAttrCount--;
		}
		textStyle.subScriptGroupLevel = 0;
		textStyle.superScriptGroupLevel = 0;
		break; */
	case rtfNoBrkSpace:
		PutStdChar (rtfSC_nobrkspace);
		break;
	case rtfTab:
		PutLitStr ("\\tab ");
		break;
	case rtfNoBrkHyphen:
		PutStdChar (rtfSC_nobrkhyphen);
		break;
	case rtfBullet:
		PutStdChar (rtfSC_bullet);
		suppressLineBreak = false;
		break;
	case rtfEmDash:
		PutStdChar (rtfSC_emdash);
		suppressLineBreak = false;
		break;
	case rtfEnDash:
		PutStdChar (rtfSC_endash);
		suppressLineBreak = false;
		break;
	case rtfLQuote:
		PutStdChar (rtfSC_quoteleft);
		suppressLineBreak = false;
		break;
	case rtfRQuote:
		PutStdChar (rtfSC_quoteright);
		suppressLineBreak = false;
		break;
	case rtfLDblQuote:
		PutStdChar (rtfSC_quotedblleft);
		suppressLineBreak = false;
		break;
	case rtfRDblQuote:
		PutStdChar (rtfSC_quotedblright);
		suppressLineBreak = false;
		break;
	case rtfPage:
		if (!(table.inside))
		{
			if (writingHeading1)
			{	
				for (i = 0; i < CountCharInString (heading1String, '{'); i++)
					PutLitStr ("}");
				InsertNewLine ();
				wrapCount = 0;
				writingHeading1 = false;
				suppressLineBreak = true;
				blankLineCount++;
			}
			else if (writingHeading2)
			{	
				for (i = 0; i < CountCharInString (heading2String, '{'); i++)
					PutLitStr ("}");
				InsertNewLine ();
				wrapCount = 0;
				writingHeading2 = false;
				suppressLineBreak = true;
				blankLineCount++;
			}
			else if (writingHeading3)
			{	
				for (i = 0; i < CountCharInString (heading3String, '{'); i++)
					PutLitStr ("}");
				InsertNewLine ();
				wrapCount = 0;
				writingHeading3 = false;
				suppressLineBreak = true;
				blankLineCount++;
			}
			CheckForCharAttr ();
			if (blankLineCount < MAX_BLANK_LINES)
			{
				PutLitStr ("\n\n");
				blankLineCount++;
			}
			
			PutLitStr ("\\newpage\n");
			if (!preferenceValue[GetPreferenceNum("ignoreRulerSettings")])
				IndentParagraph ();
			wrapCount = 0;
			suppressLineBreak = true;
		}
		break;
	}
}


/* This function looks at a few character attributes that are relevant to LaTeX */
static void
CharAttr (void)
{
	switch (rtfMinor)
	{

	case rtfItalic: case rtfBold: case rtfUnderline:	
	case rtfFontSize: case rtfSmallCaps: 
	case rtfAllCaps: case rtfDbUnderline:
		if (!(int)preferenceValue[GetPreferenceNum("ignoreTextStyle")])
			SetTextStyle ();
		break;
	case rtfSubScript: case rtfSubScrShrink: case rtfSuperScript: 
	case rtfSuperScrShrink:
		SetTextStyle ();
		break;
	case rtfForeColor:
		if (requireColorPackage)
			SetTextStyle ();
		break;
	case rtfPlain:
		CheckForCharAttr (); /* suggested by Jens Ricky */
		InitializeTextStyle ();
		break;
	case rtfDeleted:
		RTFSkipGroup ();
		break;		
	}
	

}


/* this is a diagnostic function that I call when I get into trouble 
 * with text styles.
 */
static void ExamineTextStyle (void)
{
	RTFMsg ("* GL is %d\n", groupLevel);
	RTFMsg ("* charAttrCount is %d\n", charAttrCount);
	RTFMsg ("* Bgl is %d\n", textStyle.boldGroupLevel);
	RTFMsg ("* TBgl is %d\n", textStyle.noBoldGroupLevel);
	RTFMsg ("* Igl is %d\n", textStyle.italicGroupLevel);
	RTFMsg ("* TIgl is %d\n", textStyle.noItalicGroupLevel);
	RTFMsg ("* Ugl is %d\n", textStyle.underlinedGroupLevel);
	RTFMsg ("* TUgl is %d\n", textStyle.noUnderlinedGroupLevel);
	RTFMsg ("* FCgl is %d\n", textStyle.foreColorGroupLevel);
	RTFMsg ("* BCgl is %d\n", textStyle.backColorGroupLevel);
	RTFMsg ("* Sbgl is %d\n", textStyle.subScriptGroupLevel);
	RTFMsg ("* TSbgl is %d\n", textStyle.noSubScriptGroupLevel);
	RTFMsg ("* Spgl is %d\n", textStyle.superScriptGroupLevel);
	RTFMsg ("* TSpgl is %d\n", textStyle.noSuperScriptGroupLevel);
	RTFMsg ("* Fsgl is %d\n\n", textStyle.fontSizeGroupLevel);

}

static void CheckForMathMode (void)
{
	if (!mathMode) 
	{
		PutLitStr ("$");
		wrapCount++;
		mathMode = 1;
		return;
	}
	mathMode = 0;
}


/*
 This function gets called when a \par token meaning line break is encountered.
 We need to figure out if this line break is just an ordinary line break or the
 end of a paragraph, since LaTeX treats paragraph endings differently from line 
 breaks. We do this check by scanning forward and looking if either another \par 
 (line break) or a \pard (paragraph definition) follows, both of which indicates
 that the paragraph has ended. If, however, a text item follows, it means that
 the line break was just that, a line break.
*/
static void
CheckForParagraph (void)
{
boolean stop = false;
int i;
int storeGroupLevel = groupLevel;
boolean newParagraph = false;
	

	if (suppressLineBreak == true)
		return;
		
	if (writingHeading1 || writingHeading2 || writingHeading3)
	{
		
		RTFGetToken();
		if(RTFCheckCM(rtfGroup, rtfEndGroup))
		{
			RTFUngetToken();
			return;
		}
		RTFUngetToken();
			
		if (writingHeading1)
		{
			DoParagraphCleanUp ();
			PutLitStr (heading1String);
			writingHeading1 = true;
			wrapCount = strlen (heading1String);
			paragraph.newStyle = false;
		}	
		else if (writingHeading2)
		{
			DoParagraphCleanUp ();
			PutLitStr (heading2String);
			writingHeading2 = true;
			wrapCount = strlen (heading2String);
			paragraph.newStyle = false;
		}	
		else if (writingHeading3)
		{
			DoParagraphCleanUp ();
			PutLitStr (heading3String);
			writingHeading3 = true;
			wrapCount = strlen (heading3String);
			paragraph.newStyle = false;
		}	
		return;
	}
		
	if (table.inside)
	{
		if (!suppressLineBreak) 
		{
			PutLitStr (" \\linebreak");
			InsertNewLine ();
			wrapCount = 0;
		}
		return;
	}
	
	
	if (lastCharWasLineBreak)
	{
		if (charAttrCount > 0)
		{
			StoreGroupLevels ();
			continueTextStyle = true;
		}		
		CheckForCharAttr ();
		for (i = 0; i < charAttrCount; i++)
			PutLitChar ('}');
		if (blankLineCount < MAX_BLANK_LINES)
		{
			CheckForCharAttr ();
			InsertNewLine ();
			InsertNewLine ();
			blankLineCount += 2;
		}
		if (!preferenceValue[GetPreferenceNum("ignoreRulerSettings")])
			IndentParagraph ();
			
		charAttrCount = 0;
		if (continueTextStyle)
		{
			RestoreGroupLevels ();
			textStyle.newStyle = true;
			continueTextStyle = false;
		} 		
		return;
	}
	
	RTFGetToken ();
	
	while (!stop)
	{
		/* keep going if an unknown token is encountered */
		if (rtfClass == rtfUnknown)
			stop = false;
	
		/* stop if we see a line break */
		else if (rtfMajor == rtfSpecialChar && (rtfMinor == rtfPar || rtfMinor == rtfPage))
		{
			stop = true;
			newParagraph = true;
			break;
		}
		
		/* or if we see a paragraph definition */
		else if (rtfMajor == rtfParAttr && rtfMinor == rtfParDef)
		{
			stop = true;
			newParagraph = true;
			break;
		}
		
		/* if it is a destination */
		else if (RTFCheckCM(rtfControl, rtfDestination))
		{
			stop = true;
			newParagraph = false;
			break;
		}
		
		/* or something else */
		else if ((rtfClass == rtfText && rtfMinor != rtfSC_space) || rtfClass == rtfEOF || 
				 rtfMajor == rtfSpecialChar ||
				 rtfMajor == rtfTblAttr || rtfMajor == rtfFontAttr || rtfMajor == rtfSectAttr ||
				 rtfMajor == rtfPictAttr || rtfMajor == rtfObjAttr)
		{
			stop = true;
			newParagraph = false;
			break;
		}
		
		/* otherwise keep looking and route the tokens scanned */
		if (!stop)
		{
			RTFRouteToken ();
			RTFGetToken ();
		}

	}

	/* if we saw a line break or a paragraph definition, the paragraph has ended.
	 */
	if (newParagraph)
	{
		
				
		if (charAttrCount > 0)
		{
			StoreGroupLevels ();
			continueTextStyle = true;
		}		
		CheckForCharAttr ();
		for (i = 0; i < charAttrCount; i++)
			PutLitChar ('}');
		charAttrCount = 0; 
		if (groupLevel <= storeGroupLevel && blankLineCount < MAX_BLANK_LINES) 
		{
			CheckForCharAttr ();
			InsertNewLine ();
			InsertNewLine ();
			blankLineCount += 2;
		}
		else if (!suppressLineBreak && !(textStyle.open) && !lineIsBlank) {
			PutLitStr ("\\\\");
			InsertNewLine ();
		}	
		wrapCount = 0;
		lastCharWasLineBreak = true;
		RTFUngetToken ();
		if (continueTextStyle)
		{
			RestoreGroupLevels ();
			textStyle.newStyle = true;
		} 		
		return;
	}
	
	
	else
	{
		if (rtfClass != rtfEOF && !(textStyle.open) && !lineIsBlank) {
			PutLitStr ("\\\\");
			InsertNewLine ();
		}	
		wrapCount = 0;
		RTFUngetToken ();
		return;
	}

			

}

/*
 * Eventually this should keep track of the destination of the
 * current state and only write text when in the initial state.
 *
 * If the output sequence is unspecified in the output map, write
 * the character's standard name instead.  This makes map deficiencies
 * obvious and provides incentive to fix it. :-)
 */

static void
PutStdChar (stdCode)
int	stdCode;
{
char	*oStr = (char *) NULL;
char	buf[rtfBufSiz];

	if (stdCode == rtfSC_nothing)
	{
		RTFMsg ("* Warning: I don't know this character %c (0x%x) in character set %d!\n", 
		       rtfTextBuf[0], rtfTextBuf[0], RTFGetCharSet ());
		ExamineToken ();
		PutLitStr ("(unknown char)");
		return;
	}
	oStr = outMap[stdCode];
	if (oStr == (char *) NULL)	/* no output sequence in map */
	{
		sprintf (buf, "(%s)", RTFStdCharName (stdCode));
		oStr = buf;
	}
	PutLitStr (oStr);
	wrapCount += strlen(oStr);
}


/* duh */
static void
PutLitChar (c)
int	c;
{
	fputc (c, ostream);
}

/* duh again */
static void
PutLitStr (s)
char	*s;
{
	fputs (s, ostream);
}

/* and again */
static void
WriteColor (void)
{
char buf[rtfBufSiz];

	
	sprintf(buf, "{\\color{color%ld} ", (long)rtfParam);
	PutLitStr (buf);	
	wrapCount += 17;

}


/* called when we are done reading the RTF file. */
void
EndLaTeXFile (void)
{
	WriteLaTeXFooter ();
}

/* sets the output stream */
void
RTFSetOutputStream (stream)
FILE	*stream;
{
	ostream = stream;
}


/* This function reads in a picture */
static void
ReadPicture (void)
{
char dummyBuf[rtfBufSiz], *strPtr;
char inputPicture[rtfBufSiz], outputPicture[rtfBufSiz];
int width, height;
boolean convertedPict = false;
# if RTF2LATEX2E_MAC
	FSSpec fileSpec;
	OSErr	err;
# endif
	
	requireGraphicxPackage = true;
	picture.type = unknownPict;
	picture.width = 0;
	picture.height = 0;
	picture.goalWidth = 0;
	picture.goalHeight = 0;
	picture.scaleX = 100;
	picture.scaleY = 100;
	
	/* skip everything until we reach hex data */
	while (!HexData ());

	/* Put back the first hex character into the stream (removed by HexData) */
	RTFUngetToken ();
	
	/* Process picture */
	switch (picture.type)
	{
		case pict:
			RTFMsg ("* Warning: PICT format image encountered.\n");
			#if __POWERPC__ || __CFM68K__ || __MC68K__
   		 		/* defined in unix.mac.h --- sets the creator */
   		 		_fcreator = 'ttxt';    
   		 		/* defined in unix.mac.h --- sets the type */
   		 		_ftype = 'PICT';       
   			#endif
			ConvertHexPicture ("pict");
			#if __POWERPC__ || __CFM68K__ || __MC68K__
				if ((int)preferenceValue[GetPreferenceNum("pict2eps_translate")])
				{	
					strcpy (dummyBuf, picture.name);
					ConvertPICT2EPS (dummyBuf);
					sprintf (outputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "eps");
					if(!ScanBoundingBox(outputPicture))
					{
						IncludeGraphics ("eps");
						convertedPict = true;
					}
					else
					{
						RTFMsg("* PICT->EPS conversion using Laserwriter failed...\n");
						RTFMsg("* Attempting PICT->EPS using ImageMagick...\n");
					}		
				}
			#endif
			if(!convertedPict)
			{
#if !RTF2LATEX2E_MAC			
				sprintf (inputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "pict");
				sprintf (outputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "eps");
				if(Figure2eps(inputPicture, outputPicture))
            	{
                	RTFMsg("* Figure conversion to EPS failed! Is ImageMagick installed?\n");
                	IncludeGraphics ("pict");
            	}
		    	else if(!ScanBoundingBox(outputPicture))
					IncludeGraphics ("eps");
				else				
					IncludeGraphics ("pict");
#else
				IncludeGraphics ("pict");
#endif									
			}			
			suppressLineBreak = true;						
			break;
		case wmf:
			RTFMsg ("* Warning: WMF format image encountered.\n");
			#if __POWERPC__ || __CFM68K__ || __MC68K__
   		 		/* defined in unix.mac.h --- sets the creator */
   		 		_fcreator = 'GKON';    
   		 		/* defined in unix.mac.h --- sets the type */
   		 		_ftype = 'WMF ';       
   			#endif   			
			ConvertHexPicture ("wmf");
			/* Do wmf->pict->eps */
			#if __POWERPC__ || __CFM68K__ || __MC68K__
				RTFMsg ("* Converting WMF to PICT...\n");
				strcpy (dummyBuf, picture.name);
				if ((strPtr = strstr (dummyBuf, ".wmf")) != (char *)NULL)
					*strPtr = '\0';
   		 		/* defined in unix.mac.h --- sets the creator */
   		 		_fcreator = 'ttxt';    
   		 		/* defined in unix.mac.h --- sets the type */
   		 		_ftype = 'PICT';       
				wmf2pict (picture.name, strcat (dummyBuf, ".pict"), &width, &height);
				picture.width = width;
				picture.height = height;
				err = FSpLocationFromFullPath (strlen (picture.name), picture.name, &fileSpec);
				strcpy (picture.name, dummyBuf);
				IncludeGraphics ("pict");
				/* convert to EPS */
				if ((int)preferenceValue[GetPreferenceNum("pict2eps_translate")])
				{	
					strcpy (dummyBuf, picture.name);
					ConvertPICT2EPS (dummyBuf);
					sprintf (outputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "eps");
					if(!ScanBoundingBox(outputPicture))
					{
						IncludeGraphics ("eps");
						convertedPict = true;
					}
					else
					{
						RTFMsg("* PICT->EPS conversion using Laserwriter failed...\n");
						RTFMsg("* Attempting PICT->EPS using ImageMagick...\n");
					}		
				}
				/* If the LaserWriter driver is not available, use ImageMagick */
				if(!convertedPict)
				{
					sprintf (inputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "pict");
					sprintf (outputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "eps");
					if(Figure2eps(inputPicture, outputPicture))
            		{
                		RTFMsg("* Figure conversion to EPS failed! Is ImageMagick installed?\n");
                		IncludeGraphics ("pict");
            		}
		    		else if(!ScanBoundingBox(outputPicture))
						IncludeGraphics ("eps");
					else
						IncludeGraphics ("pict");
				}			
			#else
				IncludeGraphics ("wmf");
			#endif
			break;
		case png:
			RTFMsg ("* Warning: PNG format image encountered.\n");
			#if __POWERPC__ || __CFM68K__ || __MC68K__
   		 		_fcreator = 'ogle';    /* defined in unix.mac.h --- sets the creator */
   		 		_ftype = 'PNGf';       /* defined in unix.mac.h --- sets the type */
   			#endif
			ConvertHexPicture ("png");
			#if __POWERPC__ || __CFM68K__ || __MC68K__
   		 		_fcreator = 'vgrd';    /* defined in unix.mac.h --- sets the creator */
   		 		_ftype = 'EPSF';       /* defined in unix.mac.h --- sets the type */
   			#endif			
			sprintf (inputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "png");
			sprintf (outputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "eps");
			if(Figure2eps(inputPicture, outputPicture))
            {
                printf("Figure conversion to EPS failed! Is ImageMagick installed?\n");
                IncludeGraphics ("png");
            }
		    else if(!ScanBoundingBox(outputPicture))
				IncludeGraphics ("eps");
			else
				IncludeGraphics ("png");	
			suppressLineBreak = true;
			break;
		case jpeg:
			#if __POWERPC__ || __CFM68K__ || __MC68K__
   		 		_fcreator = 'R*ch';    /* defined in unix.mac.h --- sets the creator */
   		 		_ftype = 'TEXT';       /* defined in unix.mac.h --- sets the type */
   			#endif
   			if((int)preferenceValue[GetPreferenceNum("swpMode")])
   			{
   			ConvertHexPicture ("jpg");
   			IncludeGraphics("jpg");
   			}
			else
			{
            JPEGtoEPS (&picture);
			sprintf (outputPicture, "%sFig%d.%s", RTFGetInputName (), picture.count, "eps");            
            ScanBoundingBox(outputPicture);            
            IncludeGraphics ("eps");
            }
			suppressLineBreak = true;
			break;
		default:
			ConvertHexPicture ("unknown");
			printf ("* Warning: unknown picture type encountered.\n");
			IncludeGraphics ("unknown");
			break;
	}

	/* feed "}" back to router */
	RTFRouteToken ();
	
	
	/* reset picture type */
	picture.type = unknownPict;
	picture.width = 0;
	picture.height = 0;
	picture.goalWidth = 0;
	picture.goalHeight = 0;
	picture.scaleX = 100;
	picture.scaleY = 100;
	strcpy (picture.name, "");

/* reset type and creator */
#if __POWERPC__ || __CFM68K__ || __MC68K__
   	_fcreator = Str2OSType(fileCreator);    /* defined in unix.mac.h --- sets the creator */
   	_ftype = 'TEXT';       /* defined in unix.mac.h --- sets the type */
#endif

	
}


/* This function looks for the beginning of the hex data in a \pict structure */
static int HexData (void)
{
	/* get the next token */
	RTFGetToken ();
	
	/* if we fall into a group, skip the whole she-bang */
	if (RTFCheckCM(rtfGroup, rtfBeginGroup) != 0)
	{
		RTFPeekToken ();
		if ((int)(rtfTextBuf[0]) == 10)
		{
			RTFGetToken (); /* skip any carriage returns */
			RTFPeekToken (); /* look at the next token to 
			                  * check if there is another row */
		}

		/* 
		 * there are some groups within the header that contain text data that should not
		 * be confused with hex data
		 */
		if (RTFCheckMM(rtfDestination, rtfSp) != 0 || strcmp(rtfTextBuf, "\\*") == 0)
		{
			RTFSkipGroup ();
			return (0);
		}
	}
	
	/* paydirt, hex data starts */
	if (rtfClass == rtfText) return (1);
	
	/* no such luck, but set picture attributes when encountered */
	if (RTFCheckCM(rtfControl, rtfPictAttr) != 0)
	{
		switch (rtfMinor)
		{
			case rtfMacQD:
				picture.type = pict;
				break;
			case rtfWinMetafile:
				picture.type = wmf;
				break;
			case rtfPng:
				picture.type = png;
				break;
			case rtfJpeg:
				picture.type = jpeg;
				break;
			case rtfPicGoalWid:
				picture.goalWidth = rtfParam;
				break;
			case rtfPicGoalHt:
				picture.goalHeight = rtfParam;
				break;
			case rtfPicScaleX:
				picture.scaleX = rtfParam;
				break;
			case rtfPicWid:
				picture.width = rtfParam;
				break;
			case rtfPicHt:
				picture.height = rtfParam;
				break;
			case rtfPicScaleY:
				picture.scaleY = rtfParam;
				break;
		}
	}
	return (0);

}

/* start reading hex encoded picture */
static void
ConvertHexPicture (char *pictureType)
{
FILE *pictureFile;
char dummyBuf[rtfBufSiz];
char pictByte;
int  groupEnd = false;
short hexNumber;
short hexEvenOdd = 0;  /* check if we read in even number of hex characters */

	strcpy (picture.name, "");
	
	

	
	/* increment the picture counter and clear dummy buffer*/
	(picture.count)++;
	strcpy (dummyBuf, "");

	/* get input file name and create corresponding picture file name */
	if (pictureType == (char *)NULL) strcpy (pictureType, "unknown");
	strcpy (picture.name, RTFGetInputName ());
	sprintf (dummyBuf, "Fig%d.%s", picture.count, pictureType);
	strcat (picture.name, dummyBuf);

	/* open picture file */
	if ((pictureFile = fopen(picture.name, "wb")) == NULL) 
		RTFPanic("Cannot open input file %s\n", picture.name);
		
	/* write appropriate header */
	WritePictureHeader (pictureFile);
	
	/* now we have to read the hex code in pairs of two 
	 * (1 byte total) such as ff, a1, 4c, etc...*/
	while (!groupEnd)
	{
		RTFGetToken ();
		/* the next line is just to make sure that any 
		 * CR or LF in the hex code are skipped */
		if ((int)(rtfTextBuf[0]) == 10 || (int)(rtfTextBuf[0]) == 13) 
			RTFGetToken ();
		if (rtfClass == rtfGroup) 
		{
			groupEnd = true;
			break;
		}
		if (!groupEnd) 
		{
			hexNumber = 16 * RTFCharToHex(rtfTextBuf[0]);
			hexEvenOdd++;
		}
		
		RTFGetToken ();
		if ((int)(rtfTextBuf[0]) == 10 || (int)(rtfTextBuf[0]) == 13) 
			RTFGetToken ();
		if (rtfClass == rtfGroup) {
			groupEnd = true;
			break;
		}
		if (!groupEnd) 
		{
			hexNumber += RTFCharToHex(rtfTextBuf[0]); /* this is the the number */
			hexEvenOdd--;
			/* shove that number into a character of 1 byte */
			pictByte = hexNumber;
			fputc(pictByte, pictureFile);

		}
	
	}
	
	if (fclose(pictureFile) != 0) 
		printf("* error closing picture file %s\n", picture.name);
	if (hexEvenOdd)
		 printf ("* Warning! Odd number of hex characters read for picture %s\n", 
		         picture.name);
	
	
}


/* some picture formats like PICT and WMF require headers that
 * the RTF file does not include.
 */
static void
WritePictureHeader (FILE *pictureFile)
{
unsigned char wmfhead[22]={
/* key = */ 0xd7,0xcd,0xc6,0x9a,
/* hmf = */ 0,0,
/* bbox = */ 0xfc,0xff,0xfc,0xff, /*width*/0,0,/*height*/0,0,
/* inch = */ 0x60,0x00,
/* reserved = */ 0,0,0,0,
/* checksum = */ 0,0};
int i;
int height, width;

	if(picture.goalHeight == 0)
	{
		height=((float)picture.height*picture.scaleY*96)/(rtfTpi*100);
			    width=((float)picture.width*picture.scaleX*96)/(rtfTpi*100);
	}
	else 
	{
		height=((float)picture.goalHeight*picture.scaleY*96)/(rtfTpi*100);

		width=((float)picture.goalWidth*picture.scaleX*96)/(rtfTpi*100);
	}
	wmfhead[10]=(width)%256;
	wmfhead[11]=(width)/256;
	wmfhead[12]=(height)%256;
	wmfhead[13]=(height)/256;

	/* compute Checksum */
	wmfhead[20]=0;
	wmfhead[21]=0;
	for(i=0;i<20;i+=2)
	{
		wmfhead[20]^=wmfhead[i];
		wmfhead[21]^=wmfhead[i+1];
	}


	switch (picture.type)
	{
		case pict: /* write 512 byte empty header */
			for (i = 0; i < 512; i++) fputc(' ', pictureFile);
			break;
		case wmf:
			fwrite(wmfhead,22,1,pictureFile);
			break;
		case png:
			break;
		
	}

}


/* This function writes the appropriate LaTeX2e commands to include the picture 
   into the LaTeX file */
static void
IncludeGraphics (char *pictureType)
{
char *figPtr, *suffix;
char dummyBuf[rtfBufSiz], specialBuf[rtfBufSiz];
double scaleX, scaleY, scale;
double width, height;
int llx, lly, urx, ury;

	
	suffix = strrchr (picture.name, '.');
	if (suffix != (char *)NULL && strcmp (pictureType, "eps") == 0)
		strcpy (suffix, ".eps");

	if (picture.scaleX == 0)
		scaleX = 1;
	else scaleX = (double)(picture.scaleX)/100;
	if (picture.scaleY == 0)
		scaleY = 1;
	else scaleY = (double)(picture.scaleY)/100;
	
	
	if (picture.goalHeight == 0)
	{
		width = (double)((double)picture.width*scaleX);
		height = (double)((double)picture.height*scaleY);
	}
	else 
	{
		width = (double)((double)picture.goalWidth*scaleX/20);
		height = (double)((double)picture.goalHeight*scaleY/20);
	}
		
	DoParagraphCleanUp ();
	
	figPtr = strrchr(picture.name, PATH_SEP);
	if (!figPtr) figPtr = picture.name; 
	else figPtr++;   

  if(!(int)preferenceValue[GetPreferenceNum("swpMode")])
  {
  if (strcmp(pictureType, "eps") == 0)
	{
				
		llx = picture.llx;
		lly = picture.lly;
		urx = picture.urx;
		ury = picture.ury;
		scale = (double)((double)(height)/((double)(ury - lly)));
		sprintf (dummyBuf, "\\includegraphics[bb = %d %d %d %d, scale=%2.2f]{%s}",
					llx, lly, urx, ury, scale, figPtr);
				
	}
	else 
		sprintf (dummyBuf, "\\includegraphics[width=%2.3fin, height=%2.3fin]{%s}", 
				width/72, height/72, figPtr);

	
	if (!(table.inside) && !insideFootnote)
	{
		if (height > 50) 
		{
			PutLitStr ("\\begin{figure}[htbp]");
			wrapCount = 0;
		}
		if (height > 20)
		{
			PutLitStr ("\n\\begin{center}");
			InsertNewLine ();
			wrapCount = 0;
		}
		PutLitStr (dummyBuf);
		wrapCount += strlen (dummyBuf);
		if (height > 50)
		{
			sprintf (dummyBuf, "\n\\caption{%s about here.}", figPtr);
			PutLitStr (dummyBuf);
		}
		if (height > 20)
		{
			PutLitStr ("\n\\end{center}");
			InsertNewLine ();
			if (height <= 50){
				InsertNewLine ();
				InsertNewLine ();
			}	
			blankLineCount++;
		}
		if (height > 50)
		{
			PutLitStr ("\\end{figure}");
			InsertNewLine ();
			InsertNewLine ();			
			wrapCount = 0;
		}
		suppressLineBreak = true;
	}
	}
  else /* this is for compatibility with Scientific Word */
  {
  sprintf (dummyBuf, "\\FRAME{ftbpxFU}{%2.3fpt}{%2.3fpt}{0pt}{}{}{Figure %s}", 
		width, height, figPtr);
  PutLitStr (dummyBuf);
  PutLitStr ("{");
  InsertNewLine ();
  sprintf (specialBuf,"\\special{language \"Scientific Word\";type \"GRAPHIC\";maintain-aspect-ratio TRUE; display \"USEDEF\";valid_file \"T\";");
  PutLitStr (specialBuf);
  InsertNewLine ();
  sprintf (specialBuf,"height %2.3fpt;width %2.3fpt;depth 0pt;", width, height);
  PutLitStr (specialBuf);
  InsertNewLine ();
  sprintf (specialBuf,"cropleft \"0\";croptop \"1\";cropright \"1\";cropbottom \"0\";");
  PutLitStr (specialBuf);
  InsertNewLine ();
  sprintf (specialBuf,"tempfilename '%s';tempfile-properties \"XPNEU\";", figPtr);
  PutLitStr (specialBuf);
  PutLitStr ("}}");
  InsertNewLine ();
  suppressLineBreak = true;
  }
}

/* This function scans the bounding box information for an EPS file */
static int
ScanBoundingBox (char *epsFile)
{
FILE *fp;
char	buf[50];


	if ((fp = fopen (epsFile, "r")) == (FILE *) NULL)
			{
				RTFMsg ("* can't open eps file %s\n", epsFile);
				return (1);
			}

	while (fscanf (fp, "%s", buf) != EOF)
	{
		if (strcmp (buf, "%%BoundingBox:") == 0)
		{
			fscanf (fp, "%d", &(picture.llx));
			fscanf (fp, "%d", &(picture.lly));
			fscanf (fp, "%d", &(picture.urx));
			fscanf (fp, "%d", &(picture.ury));
			if (fclose(fp) != 0) 
            {
                printf("* error closing eps file %s\n", epsFile);
			    return (1);
            }
            return (0);
		}
	}
	
	if (fclose(fp) != 0) 
    {
        printf("* error closing eps file %s\n", epsFile);
        return (1);
    }
	return (0);
}

/* a really useful diagnostic function that lets me examine
 * the token just read.
 */
void ExamineToken (void)
{
	printf ("* Token is %s\n", rtfTextBuf);
	printf ("* Class is %d\n", rtfClass);
	printf ("* Major is %d\n", rtfMajor);
	printf ("* Minor is %d\n", rtfMinor);
	printf ("* Param is %ld\n\n", rtfParam);
}


/* This function reads objects that may contain pictures or equations */
static void
ReadObject (void)
{
int i;
int groupCounter = 1; /* one opening brace has been counted */
int temp;
boolean res;
	
	GetObjectClass (&groupCounter);

	switch (object.class)
	{
		case unknownObj: default:
			printf ("* unsupported object \"%s\" encountered, skipping...\n", 
			        object.className);
			RTFSkipGroup ();
			break;			
		case Equation: case equation:
			if((int)preferenceValue[GetPreferenceNum("convertEquations")]) 
				res = ReadEquation (&groupCounter);
			else
				res = false;	
			/* if unsuccessful, read the equation as a picture */
			if(!res)
			{
				temp = groupCounter;
				while (!RTFCheckMM(rtfDestination, rtfPict))
				{
					RTFGetToken();
					if(RTFCheckCM(rtfGroup, rtfBeginGroup))
						groupCounter++;
					if(RTFCheckCM(rtfGroup, rtfEndGroup))
						groupCounter--;
					if (groupCounter < temp)
						break;
				}				
				if(groupCounter > temp)
				{
					ReadPicture ();
					if(groupCounter - 1 - temp > 0)
					{
						for(i = 0; i < groupCounter - 1 - temp; i++)
							RTFSkipGroup();
					}
				}
				groupCounter = temp;			
			}
			break;
		case WordPicture: case MSGraphChart:
			while (!ReachedResult (&groupCounter));
			ReadPicture ();
			break;
	}
	object.class = 0;
	strcpy (object.className, "");

	/* if there are open groups left, close them */
	if (groupCounter != 0)
	{
		for (i = 0; i < groupCounter; i++)
			RTFSkipGroup ();
	}		
	
	/* send the last closing brace back into the router */
	RTFRouteToken ();

}


/* 
 * gets object class. right now it recognizes only
 * equations. everything else is unknown.
 */
static void
GetObjectClass (int *groupCounter)
{
int reachedObjectClass = 0;
int reachedEndGroup = 0;
int i;

/* keep scanning until \objectclass is found */
	while (!reachedObjectClass)
	{
		RTFGetToken ();
		if (RTFCheckCM (rtfGroup, rtfBeginGroup) != 0) (*groupCounter)++;
		else if (RTFCheckCM (rtfGroup, rtfEndGroup) != 0) (*groupCounter)--;
		if (RTFCheckMM (rtfDestination, rtfObjClass) != 0)
			reachedObjectClass = 1;
		if (*groupCounter == 0)
		{
			object.class = unknownObj;
			return;
		}
	}

/* read the object class */	
	strcpy (object.className, "");
	while (!reachedEndGroup)
	{
		RTFGetToken ();
		if (RTFCheckCM (rtfGroup, rtfBeginGroup) != 0) 
			RTFSkipGroup ();
		if (RTFCheckCM (rtfGroup, rtfEndGroup) == 0)
			strcat (object.className, rtfTextBuf);
		else 
		{
			reachedEndGroup = 1;
			(*groupCounter)--;
		}
	}

/* do we recognize this object class? */
	for (i = 0; objectClassList[i] != (char *) NULL; i++)
	{
		if (strstr(object.className, objectClassList[i]) != (char *)NULL)
		{
			object.class = i;
			break;
		}
		object.class = 0;
	}
	
}


/* We want to look at the result section of the object because 
 * that's what tells us what the object looks like. This usually
 * contains a picture */
static int 
ReachedResult (int *groupCount)
{
char *fn = "ReachedResult";


	/* get the next token */
	RTFGetToken ();
	
	if (RTFCheckCM (rtfGroup, rtfBeginGroup) != 0)
	{
		(*groupCount)++;
		return (0);
	}
	
	else if (RTFCheckCM (rtfGroup, rtfEndGroup) != 0)
	{
		(*groupCount)--;
		return (0);
	}
	
	else if (RTFCheckMM (rtfDestination, rtfObjResult) != 0 || 
			 RTFCheckMM (rtfWord97ObjAttr, rtfWord97ObjResult) != 0 || 
			 RTFCheckMM (rtfDestination, rtfPict) != 0 ||
			 RTFCheckMM (rtfWord97ObjAttr, rtfWord97ObjText) != 0)
	{
		if (RTFCheckMM (rtfDestination, rtfPict) != 0)
			word97ObjectType = word97Picture;
		else if (RTFCheckMM (rtfDestination, rtfObjResult) != 0) 
			word97ObjectType = standardObject;
		else if (RTFCheckMM (rtfWord97ObjAttr, rtfWord97ObjResult) != 0)
			word97ObjectType = word97Object;
		else if (RTFCheckMM (rtfWord97ObjAttr, rtfWord97ObjText) != 0)
			word97ObjectType = word97ObjText;
		(*groupCount)--; /* account for opening brace just 
		                  * before result control word */
		return (1);
	}
	
		
	else if (rtfClass == rtfEOF)
	{
		RTFPanic ("* EOF reached!\n");
		exit (1);
	}
	
	return (0);
		
}

/* This function reads a MathType Equation embedded as a an OLE
   object */
static boolean 
ReadEquation (int *groupCount)
{
char objectFileName[rtfBufSiz];
unsigned char *nativeEquationStream = NULL;
int err;
size_t equationSize;
MTEquation* theEquation = NULL;
boolean res;

	DoParagraphCleanUp();
	DoSectionCleanUp();

	/* look for start of \objdata  group */
	while (!RTFCheckMM (rtfDestination, rtfObjData))
	{
		/* get the next token */
		RTFGetToken ();
	
		if (RTFCheckCM (rtfGroup, rtfBeginGroup) != 0)
			(*groupCount)++;
	
		else if (RTFCheckCM (rtfGroup, rtfEndGroup) != 0)
		{
			(*groupCount)--;
			if (*groupCount == 0)
			{
				RTFMsg ("* ReadEquation: objdata group not found!\n");
				return (false);
			}
		}
			
		else if (rtfClass == rtfEOF)
		{
			RTFPanic ("* ReadEquation: EOF reached!\n");
			exit (1);
		}
	}
	
	/* convert hex-encoded object data group to binary and read the OLE object
	   into file objectFileName */
	ReadObjectData (objectFileName, equation, EQUATION_OFFSET);
	(*groupCount)--;


	
	/* Decode the OLE and extract the "Equation Native" stream into buffer 
	   nativeEquationStream */
	err = DecodeOLE (objectFileName, "/Equation Native", &nativeEquationStream, &equationSize);
	if (err != 0)
	{
		RTFMsg ("* error decoding OLE equation object!\n");
		if (nativeEquationStream != NULL)
			free (nativeEquationStream);
		return (false);
	}
  else
  {
    remove(objectFileName);
  }
/*	
	printf ("* pointer is %d, size is %d\n", nativeEquationStream, equationSize);
*/		
	theEquation = (MTEquation *)malloc (sizeof(MTEquation));
	if (theEquation == NULL)
	{
		RTFMsg ("* error allocating memory for equation!\n");
		if (nativeEquationStream != NULL)
			free (nativeEquationStream);
		return (false);
	}
	else
		memset(theEquation, 0, sizeof(MTEquation));
	
  	res = Eqn_Create(theEquation, nativeEquationStream+UNKNOWN_EQUATION_OFFSET, 
  					 equationSize-UNKNOWN_EQUATION_OFFSET);
  	if(!res)
  	{
		RTFMsg ("* could not create equation structure!\n");
		if (nativeEquationStream != NULL)
			free (nativeEquationStream);
		if (theEquation != NULL)
    		free(theEquation);
		return (false);
  	}
  	Eqn_TranslateObjectList(theEquation, ostream, 0);
  	Eqn_Destroy(theEquation);
		
	if (theEquation != NULL)
    	free(theEquation);

		
	if (nativeEquationStream != NULL)
		free (nativeEquationStream);
		
	requireAmsMathPackage = true;			
	return (true);		
}



/* This function reads in the hex-encoded object data 
   group in the file objectFileName */
static void
ReadObjectData (char *objectFileName, int type, int offset)
{
char dummyBuf[20];
FILE *objFile;
int i;
int	hexOffset;
int  groupEnd = false;
char objByte;
short hexNumber;
short hexEvenOdd = 0;  /* check if we read in even number of hex characters */

/*	RTFMsg ("* Reading object data...\n"); */
	
	if (type == equation)
	{
		(oleEquation.count)++;
		sprintf (dummyBuf, "Eq%d.eqn", oleEquation.count);
	}	
	else
		sprintf (dummyBuf, ".obj");
	
	
	/* construct full path of file name */
	strcpy (objectFileName, RTFGetInputName ());
	strcat (objectFileName, dummyBuf);

	#if __POWERPC__ || __CFM68K__ || __MC68K__
   		 _fcreator = 'MSWD';    /* defined in unix.mac.h --- sets the creator */
   		 _ftype = 'W8BN';       /* defined in unix.mac.h --- sets the type */
   	#endif


	/* open object file */
	if ((objFile = fopen(objectFileName, "wb")) == NULL) 
		RTFPanic("Cannot open input file %s\n", objectFileName);
		
	/* skip offset header (2 hex characters for each byte) */
	hexOffset = offset*2;
	for (i = 0; i < hexOffset; i++)
		RTFGetToken ();
		
	/* now we have to read the hex code in pairs of two 
	 * (1 byte total) such as ff, a1, 4c, etc...*/
	while (!groupEnd)
	{
		RTFGetToken ();
		/* the next line is just to make sure that any 
		 * CR or LF in the hex code are skipped */
		if ((int)(rtfTextBuf[0]) == 10 || (int)(rtfTextBuf[0]) == 13) 
			RTFGetToken ();
		if (rtfClass == rtfGroup) 
		{
			groupEnd = true;
			break;
		}
		if (!groupEnd) 
		{
			hexNumber = 16 * RTFCharToHex(rtfTextBuf[0]);
			hexEvenOdd++;
		}
		
		RTFGetToken ();
		if ((int)(rtfTextBuf[0]) == 10 || (int)(rtfTextBuf[0]) == 13) 
			RTFGetToken ();
		if (rtfClass == rtfGroup) {
			groupEnd = true;
			break;
		}
		if (!groupEnd) 
		{
			hexNumber += RTFCharToHex(rtfTextBuf[0]); /* this is the the number */
			hexEvenOdd--;
			/* shove that number into a character of 1 byte */
			objByte = hexNumber;
			fputc(objByte, objFile);

		}
	
	}	

	if (fclose(objFile) != 0) 
		printf("* error closing object file %s\n", objectFileName);
	if (hexEvenOdd)
		 printf ("* Warning! Odd number of hex characters read for object!\n");

}


/* Of course, Word97 has to do everything differently. */
static void
ReadWord97Object (void)
{
int i;
long objectStart;
int groupCount = 1; /* one opening brace has been counted */
int	tempGroupCount = groupLevel;
int word97ObjTextGL = 1;
short prevChar;
	
	word97ObjectType = unknownWord97Object;
	
	/* look for a standard embedded object first: may have an equation */
	/* mark the current cursor position */
	prevChar = RTFPushedChar ();
	RTFStoreStack ();
	objectStart = ftell (ifp);
	

	while (!RTFCheckMM (rtfDestination, rtfObject))
	{
		RTFGetToken ();

		if (RTFCheckCM (rtfGroup, rtfBeginGroup) != 0)
			groupCount++;
	
		else if (RTFCheckCM (rtfGroup, rtfEndGroup) != 0)
		{
			groupCount--;
			/* did not find a standard object */
			if (groupCount == 0) 
			{
				fseek (ifp, objectStart, 0);
				RTFSimpleInit ();
				RTFSetPushedChar (prevChar);
				RTFRestoreStack ();
				break;
			}
		}
		
	}
	
	/* if we found a standard object, read it and get out */
	if (RTFCheckMM (rtfDestination, rtfObject))
	{
		ReadObject ();
		groupCount--;
		/* if there are open groups left, close them */
		if (groupCount != 0)
			for (i = 0; i < groupCount; i++)
				RTFSkipGroup ();
		/* send the last closing brace back into the router */
		RTFRouteToken ();
		return;
	}
	
	groupCount = 1;
	
	
	while (!ReachedResult (&groupCount))
	{
		if (groupCount == 0)
		{
			RTFMsg ("*unknown Word97 object...\n");
			RTFRouteToken ();
			return;
		}
	}
	
		
	switch (word97ObjectType)
	{
		case word97Picture:
			ReadPicture ();
			break;
		case standardObject: case word97Object:
			ReadWord97Result ();
			break;
		case word97ObjText:
			if (!table.inside)
			{
				StoreGroupLevels ();
				for (i = 0; i < charAttrCount; i++)
					PutLitStr ("}");
				charAttrCount = 0;
				if (blankLineCount < MAX_BLANK_LINES)
				{
					InsertNewLine ();
					InsertNewLine ();
					blankLineCount++;
				}
				wrapCount = 0;
				word97ObjTextGL = groupLevel;
				while (groupLevel >= word97ObjTextGL)
				{
					suppressLineBreak = true;
					RTFGetToken ();
					RTFRouteToken ();
				}
				RestoreGroupLevels ();
				if (blankLineCount < MAX_BLANK_LINES)
				{
					InsertNewLine ();
					InsertNewLine ();
					blankLineCount++;
				}
				wrapCount = 0;
			}
			break;
	}
	
	if (groupCount == 0)
		return;
	
	
	/* if there are open groups left, close them */
	if (groupCount != 0)
		for (i = 0; i < groupCount; i++)
			RTFSkipGroup ();

	object.word97 = 0;
	
	/* send the last closing brace back into the router */
	RTFRouteToken ();

}

/* This is the result field of the Word97 object */
static void 
ReadWord97Result (void)
{
int i;
int groupCount = 1; /* one opening brace has been counted */

	/* scan until object or picture is reached */
	while (groupCount != 0)
	{
		RTFGetToken ();
		if (RTFCheckMM (rtfDestination, rtfObject) != 0)
		{
			ReadObject ();
			groupCount--;
			break;
		}
		else if (RTFCheckMM (rtfDestination, rtfPict) != 0)
		{
			ReadPicture ();
			groupCount--;
			break;
		}
		else if (RTFCheckCM (rtfGroup, rtfBeginGroup) != 0)
			groupCount++;
		else if (RTFCheckCM (rtfGroup, rtfEndGroup) != 0)
			groupCount--;
	}
	
	if (groupCount == 0) 
		printf ("* Warning: no supported structure in Word97 object found.\n");
	
	
	/* if there are open groups left, skip to the end */
	if (groupCount > 0)
		for (i = 0; i < groupCount; i++)
			RTFSkipGroup ();	
			
	RTFRouteToken ();
	
}

/* Decodes the OLE and extract the specified stream type into a buffer.
   This function uses the cole library */
static int
DecodeOLE (char *objectFileName, char *streamType,
           unsigned char **nativeStream, size_t *size)
{
COLEFS * cfs;
COLERRNO colerrno;
COLEFILE *coleFile;

	cfs = cole_mount (objectFileName, &colerrno);
	if (cfs == NULL) {
		cole_perror ("DecodeOLE cole_mount", colerrno);
		return (1);
	}
/*
	if (cole_print_tree (cfs, &colerrno)) {
		cole_perror ("DecodeOLE cole_print_tree", colerrno);
		cole_umount (cfs, NULL);
		return (1);
	}
*/	
	if ((coleFile = cole_fopen (cfs, streamType, &colerrno)) == NULL)
	{
		cole_perror ("DecodeOLE cole_fopen", colerrno);
		cole_umount (cfs, NULL);
		return 1;
	}

	*size = cole_fsize(coleFile);
	

	*nativeStream = (unsigned char *)malloc (*size);

	

	if (*nativeStream == (unsigned char *)NULL)
	{
		RTFMsg ("* DecodeOLE: memory allocation failed for native stream!\n");
		cole_fclose (coleFile, &colerrno);
		cole_umount (cfs, NULL);
		return 1;
	}
		
	if (cole_fread (coleFile, *nativeStream, *size, &colerrno) == 0)
	{
		cole_perror ("DecodeOLE cole_fread", colerrno);
		cole_fclose (coleFile, &colerrno);
		cole_umount (cfs, NULL);
		return 1;
	}
	
	if (cole_fclose (coleFile, &colerrno) != 0)
	{
		cole_perror ("DecodeOLE cole_fclose", colerrno);
		cole_umount (cfs, NULL);
		return 1;
	}

	if (cole_umount (cfs, &colerrno)) {
		cole_perror ("DecodeOLE cole_umount", colerrno);
		return (1);
	}

	return 0;

}


/* All right, the big monster. When we reach a table, 
 * we don't know anything about it, ie. number of rows
 * and columns, whether any rows or columns are merged,
 * etc. We have to prescan the table first, where we
 * get vital table parameters, and then come back to
 * read the table in actual.
 */
static void 
DoTable (void)
{
boolean newRow = true;
int i, rowNum;
int cols = 0;
cell *cellPtr;
char buf[100];

	requireTablePackage = true;
	table.previousColumnValue = -1000000;
	
	/* throw away old cell information lists */
	while (table.cellInfo != (cell *) NULL)
	{
		cellPtr = (table.cellInfo)->nextCell;
		RTFFree ((char *) table.cellInfo);
		table.cellInfo = cellPtr;
	}
	
	/* Prescan table */
	table.cellCount = 0;
	PrescanTable ();
/*	printf ("* done prescanning table...\n"); */
	table.cellCount = 0;

	table.inside = true;

	DoParagraphCleanUp ();

	if (blankLineCount < MAX_BLANK_LINES)
	{
		InsertNewLine ();
		InsertNewLine ();
		blankLineCount++;
	}

	PutLitStr ("\\begin{");
	PutLitStr (tableString);
	PutLitStr ("}{");
	if (table.multiCol)
	{	
		for (i = 0; i < table.cols; i++)
			PutLitChar ('l');
		PutLitStr ("}");
	}
	else
	{
		PutLitStr("|");
		for (i = 0; i < table.cols; i++)
		{
			cellPtr = GetCellInfo(i);
			if (cellPtr->x > 0)
				break;
			sprintf(buf, "p{%1.3fin}|", cellPtr->width);
			PutLitStr(buf);  
		}
		PutLitStr("}"); 
	}	
	PutLitStr ("\n\\hline");
	InsertNewLine ();
	
	wrapCount = 0;
	
	paragraph.alignment = left; /* default justification */
	
	rowNum = 0;
	
/*	printf ("* processing table rows...\n");	*/
	for (i = 0; i < table.rows; i++)
	{
		sprintf(buf, "%% ROW %d", i+1);
		PutLitStr(buf);
		InsertNewLine();
		
		ProcessTableRow (rowNum);
			
		fseek (ofp, -4, 2); /* erase the "& \n" at the end of the last cell of the row */
		PutLitStr ("\\\\");
		InsertNewLine ();
		if(i < (table.rows - 1))
			DrawTableRowLine (rowNum);
		wrapCount = 0;
		paragraph.alignment = left;
		rowNum++;
		
		InitializeTextStyle ();

	}
	
	PutLitStr ("\\hline\n");
	sprintf (buf, "\\end{%s}\n", tableString);
	PutLitStr (buf);
	InsertNewLine ();
	lastCharWasLineBreak = true;
	suppressLineBreak = true;
	
	wrapCount = 0;
	cols = 0;		
	

	RTFFree ((char *) table.columnBorders);
	table.inside = false; /* end of table */
	table.previousColumnValue = -1000000;
	table.multiCol = false;
	table.multiRow = false;

}

/* This is where we actually process each table row. */
static void
ProcessTableRow (int rowNum)
{
boolean endOfRow = false;
boolean startCellText = false;
cell *cellPtr;
int	cellsInThisRow = 0;
int i;

	wroteCellHeader = false;
	
	
	while (!endOfRow)
	{
		RTFGetToken ();
		if (RTFCheckCM (rtfControl, rtfCharAttr) != 0)
		{
			switch (rtfMinor)
			{
				case rtfFontNum:
					RTFRouteToken ();
					break;
				case rtfPlain:
					CheckForCharAttr (); /* suggested by Jens Ricky */
					InitializeTextStyle ();
					break;
				case rtfBold: case rtfItalic: case rtfUnderline: case rtfAllCaps: 
				case rtfSubScrShrink: case rtfSuperScrShrink: case rtfForeColor: 
				case rtfBackColor: case rtfFontSize:
					CharAttr ();
					if (startCellText && (cellsInThisRow < (table.rowInfo)[rowNum]))
						RTFRouteToken ();
					break;
			}
		}
		
		else if (RTFCheckCM (rtfControl, rtfDestination))
		{
			WriteCellHeader	(table.cellCount);
			cellPtr = GetCellInfo (table.cellCount);		
			wrapCount = 1;
			startCellText = true;
			RTFRouteToken ();
		}
		
		/* ignore unknown destination groups */
		else if (strcmp (rtfTextBuf, "\\*") == 0) 
		{
			RTFSkipGroup ();
			continue;
		}
		
		else if (RTFCheckCM (rtfControl, rtfSpecialChar) != 0)
		{
			switch (rtfMinor)
			{
				case rtfRow:
					endOfRow = true;
					return;
					
				case rtfCell:
					(table.cellCount)++;
					cellsInThisRow++;
					if (!startCellText) /* this is in case the cell was blank. 
					                     * We still have to tell latex about it. */
					{
						cellPtr = GetCellInfo (table.cellCount - 1);
						WriteCellHeader (table.cellCount - 1);
					}
					StoreGroupLevels ();
					CheckForCharAttr ();
					for (i = 0; i < charAttrCount; i++)
						PutLitChar ('}');
					charAttrCount = 0;
					RestoreGroupLevels ();
					if (cellPtr->mergePar == first) 
						PutLitChar ('}');
					PutLitStr ("} & ");
					InsertNewLine ();
					suppressLineBreak = true;
					startCellText = false; /* we are not in the text 
					                        * field of a cell any more */
					paragraph.alignment = left;
					wroteCellHeader = false;
					break;
				
				default:
					if (!startCellText) /* this is in case the cell was blank. 
					                     * We still have to tell latex about it. */
					{
						cellPtr = GetCellInfo (table.cellCount);
						WriteCellHeader (table.cellCount);
						startCellText = true;
					}
					RTFRouteToken ();
					break;
			}
		}
		
		else if (RTFCheckCM (rtfControl, rtfTblAttr) == 0)
		{
			/* if we see text, we are already in the text field of a cell */
			if (rtfClass == rtfText && !startCellText)
			{
				WriteCellHeader	(table.cellCount);
				cellPtr = GetCellInfo (table.cellCount);		
				wrapCount = 1;
				startCellText = true;
				textStyle.newStyle = false;
			}
			RTFRouteToken ();
		}
		
		else if (RTFCheckCM (rtfControl, rtfGroup) != 0)
			RTFRouteToken ();
		
	}

}

/* writes cell information for each cell.
   Each cell is defined in a multicolumn
   environment for maximum flexibility.
   Useful when we have merged rows and
   columns.
 */
static void
WriteCellHeader (int cellNum)
{
char buf[rtfBufSiz];
cell *cellPtr;
char *fn = "WriteCellHeader";

	if (wroteCellHeader)
		return;
	
	cellPtr = GetCellInfo (cellNum);
	if (cellPtr == (cell *)NULL)
		RTFPanic ("%s: Attempting to access invalid cell at index %d\n", fn, cellNum);
		
	if(table.multiCol)
	{	
		sprintf (buf, "\\multicolumn{%d}{", cellPtr->columnSpan);
		PutLitStr (buf);
		if (cellPtr->columnSpan < 1)
			RTFMsg ("* Warning: nonsensical table encountered...cell %d spans %d columns.\n \
					Proceed with caution!\n", cellPtr->index, cellPtr->columnSpan);
		/* this check is to draw the left vertical boundary of the table */
		if (cellPtr->y == 0) 
			PutLitChar ('|');
		if (cellPtr->mergePar == first)
		{
			sprintf (buf, "p{%1.3fin}|}\n{", cellPtr->width);
			PutLitStr (buf);
		}
		else
		{ 
			sprintf (buf, "p{%1.3fin}|}\n{%s", cellPtr->width, 
		         	justificationList[paragraph.alignment]);
			PutLitStr (buf);
			InsertNewLine ();
		}
	}
	else if (cellPtr->mergePar != first)
	{
		sprintf (buf, "{%s ", justificationList[paragraph.alignment]);
		PutLitStr (buf);
	}	
	else
	{
		PutLitStr ("{");
	}	

	if (rtfClass == rtfText)
		WriteTextStyle ();
				
	if (cellPtr->mergePar == first) 
		DoMergedCells (cellPtr);
	
	suppressLineBreak = true;
	wroteCellHeader = true;

}


/*
 This function draws horizontal lines within a table. It looks 
 for vertically merged rows that  do not have any bottom border
 */
static void
DrawTableRowLine (int rowNum)
{
int i, cellPosition;
cell *cellInfo1, *cellInfo2;
char buf[rtfBufSiz];

	/* if we are at the last row of the table, just draw a straight \hline. */
	if (rowNum == (table.rows) - 1 || !table.multiRow)
	{
		PutLitStr ("\\hline");
		InsertNewLine ();
		return;
	}
	
	/* otherwise use \cline for every cell */
	/* this is to count cell positions as if the table is a matrix. */
	cellPosition = 0; 
	for (i = 0; i < (table.rowInfo)[rowNum]; i++)
	{

		cellInfo1 = GetCellByPos (rowNum, cellPosition);
		cellPosition += cellInfo1->columnSpan;

		if (cellInfo1->mergePar == none)
		{
			sprintf(buf,"\\cline{%d-%d}", cellInfo1->y + 1, 
			        cellInfo1->y + cellInfo1->columnSpan);
			PutLitStr (buf);
		}
		
		else if (cellInfo1->mergePar == previous)
		{
			cellInfo2 = GetCellByPos (rowNum + 1, i);
			if (cellInfo2->mergePar != previous)
			{
				sprintf(buf,"\\cline{%d-%d}", cellInfo1->y + 1,
				        cellInfo1->y + cellInfo1->columnSpan);
				PutLitStr (buf);
			}
		}
			
	}
	
	InsertNewLine ();
}

static void
DoTableAttr (void)
{
	switch (rtfMinor)
	{
		case rtfRowLeftEdge:
			table.leftEdge = rtfParam;
			break;
		case rtfCellPos:
			ReadCell ();
			table.cellCount++;
			((table.rowInfo)[table.rows])++;
			(table.cols)++;
			break;
		case rtfVertMergeRngFirst:
			table.cellMergePar = first;
			break;
		case rtfVertMergeRngPrevious:
		 	table.cellMergePar = previous;
		 	break;
	
	}

}

/* 
 This routine prescans the table. It figures out how many rows there are in the table
 and the number of cells in each row. It also calculates the cell widths. Finally, it
 buils an array of column borders that is useful in figuring out whether a cell spans
 multiple columns. It has lots of diagnostic printf statements. Shows how much I
 struggled, and how much I trust it.
*/
static void
PrescanTable (void)
{
long tableStart, tempGroupLevel;
boolean foundRow = true;
boolean foundColumn = true;
boolean recordedCellBorder = false;
int i, j;
cell *cellPtr, *cellPtr1;
char *fn = "PrescanTable";
boolean stop = false;
short prevChar;
int dummyCount = 0;
int maxCols = 0;
int rowWithMaxCols = 0;
long tableLeft, tableRight, tableWidth;

	RTFStoreStack ();
	prevChar = RTFPushedChar ();
	tempGroupLevel = groupLevel;
	/* mark the current cursor position */
	tableStart = ftell (ifp);
	
	RTFGetToken ();
	
	table.rows = 0;
	table.cols = 0;
	table.inside = true;
	table.cellInfo = (cell *)NULL;
	
	i = 0;
	
	/* Prescan each row until end of the table. */
	while (foundRow)
	{
		table.cols = 0;
		(table.rowInfo)[table.rows] = 0;
		
		while (foundColumn)
		{
			RTFGetToken ();
			if (RTFCheckMM (rtfSpecialChar, rtfOptDest) != 0)
				RTFSkipGroup ();
			
			else if (RTFCheckCM (rtfControl, rtfTblAttr) != 0)
				RTFRouteToken ();
				
			else if (RTFCheckMM (rtfSpecialChar, rtfRow) != 0 || rtfClass == rtfEOF)
				foundColumn = false;
				
		}
		
/*		RTFMsg ("* reached end of row %d\n", table.rows); */
		
		table.inside = false;
		table.newRowDef = false;
		stop = false;

		while (!stop)
		{
		
			RTFGetToken ();
			
			if (RTFCheckMM (rtfSpecialChar, rtfOptDest) != 0)
			{
				RTFSkipGroup ();
			}
			
			else if (RTFCheckMM (rtfTblAttr, rtfRowDef) != 0)
			{
				stop = true;
				table.inside = true;
				table.newRowDef = true;
			}

			
			else if (rtfClass == rtfText || RTFCheckCM (rtfControl, rtfSpecialChar) != 0)
			{
				stop = true;
				break;
			}
				
			else if (RTFCheckMM (rtfParAttr, rtfInTable) != 0)
			{
				stop = true;
				table.inside = true;
				break;
			}
			
			else if (rtfClass == rtfEOF)
			{
/*				printf ("* end of file!\n"); */
				stop = true;
				table.inside = false;
				break;
			}
			
		
		}
		
		
		if (!(table.inside))
			foundRow = false;
		else
			foundRow = true;


		if ((table.rowInfo)[table.rows] == 0)
			(table.rowInfo)[table.rows] = (table.rowInfo)[table.rows - 1];
		
		if (table.cols > maxCols)
		{
			maxCols = table.cols;
			rowWithMaxCols = table.rows;
		}
		
/*		RTFMsg ("* read row %d with %d cells\n", table.rows, (table.rowInfo)[table.rows]); */
		(table.rows)++;
		table.previousColumnValue = -1000000;
		
/*		RTFMsg ("* inside table is %d, new row def is %d\n", table.inside, table.newRowDef); */
		
		if (table.inside && !(table.newRowDef))
		{
			(table.rows)--;
			InheritTableRowDef ();
/*			RTFMsg ("* Inherited: read row %d with %d cells\n", table.rows, (table.rowInfo)[table.rows]); */
			(table.rows)++;
		}
			
		
		foundColumn = true;

	}
	
	
	table.cols = maxCols;
	
	{
	long *rightBorders;
	boolean	enteredValue;
	
		rightBorders = (long *) RTFAlloc ((table.cellCount) * sizeof(long));
		if (rightBorders == (long *)NULL)
			RTFPanic ("%s: cannot allocate array for cell borders\n", fn);
		
		table.cols = 0;
		for (cellPtr = table.cellInfo; cellPtr != (cell *)NULL; cellPtr = cellPtr->nextCell)
		{
			enteredValue = false;
			for (j = 0; j < table.cols; j++)
				if (rightBorders[j] == cellPtr->right)
					enteredValue = true;
			if (!enteredValue)
				rightBorders[(table.cols)++] = cellPtr->right;
			if (cellPtr->y == 0)
				cellPtr->left = table.leftEdge;		
		}
		
		/* allocate array for coulumn border entries. */
		table.columnBorders = (long *) RTFAlloc (((table.cols) + 1)*sizeof(long));
		if ((table.columnBorders) == (long *)NULL)
			RTFPanic ("%s: cannot allocate array for cell borders\n", fn);

		for (i = 0; i < table.cols; i++)
			(table.columnBorders)[i+1] = rightBorders[i];
			
		RTFFree (rightBorders);
	
	}
	
/*	
 *	Table parsing can be messy, and it is still buggy. 
 *  Here are a few diagnostic messages
 *	that I print out when I am having trouble. 
 *  Comment out for normal operation.
*/


/*	RTFMsg ("* table has %d rows and %d cols \n", table.rows, table.cols); */
		
	(table.columnBorders)[0] = table.leftEdge;
	
	
	/*
	sort the column border array in ascending order. 
	This is important for figuring out 
	whether a cell spans multiple columns. 
	This is calculated in the function GetColumnSpan.
	*/
/*	RTFMsg ("* sorting borders...\n"); */
	for (i = 0; i < (table.cols); i++)
		for (j = i+1; j < (table.cols + 1); j++)
			if ((table.columnBorders)[i] > (table.columnBorders)[j])
				Swap ((table.columnBorders)[i], (table.columnBorders)[j]);
	
/*	RTFMsg ("* table left border is at %ld\n", (table.columnBorders)[0]); */
	for (i = 1; i < (table.cols + 1); i++)
	{
		/* Sometimes RTF does something stupid and assigns two adjoining cell
		   the same right border value. Dunno why. If this happens, I move the
		   second cell's right border by 10 twips. Microsoft really has some 
		   morons!
		*/
		if (table.columnBorders[i] == table.columnBorders[i-1])
			table.columnBorders[i] += 10;
/*
		RTFMsg ("* cell right border is at %ld\n", table.columnBorders[i]); 
*/
	}


	/* fill in column spans for each cell */
	for (i = 0; i < table.cellCount; i++)
	{
		cellPtr = GetCellInfo (i);
		if (cellPtr == (cell *)NULL)
			RTFPanic ("%s: Attempting to access invalid cell at index %d\n", fn, i);
		cellPtr->columnSpan = GetColumnSpan (cellPtr);
        if (cellPtr->columnSpan > 1)
            table.multiCol = true;
/*		RTFMsg ("* cell %d spans %d columns\n", cellPtr->index, cellPtr->columnSpan); */

		/* correct the vertical cell position for any multicolumn cells */
		if ((cellPtr->y) != 0)
		{
			cellPtr1 = GetCellInfo (i-1);
			if (cellPtr == (cell *)NULL)
				RTFPanic ("%s: Attempting to access invalid cell at index %d\n", fn, i);
			cellPtr->y = cellPtr1->y + cellPtr1->columnSpan;

		}
	
		tableLeft = (table.columnBorders)[0];
		tableRight = (table.columnBorders)[table.cols];
		tableWidth = tableRight - tableLeft;

		cellPtr->width = (double)((double)(cellPtr->right) - (double)(cellPtr->left))/rtfTpi;
		if (tableWidth > 4.5*rtfTpi)
			cellPtr->width *= (double)(4.5*(double)rtfTpi/(double)tableWidth);
		
	}

/*
	for (cellPtr = table.cellInfo; cellPtr != (cell *)NULL; cellPtr = cellPtr->nextCell)
		RTFMsg ("* now cell %d (%d, %d) has left and right borders at %ld, %ld and spans %d columns\n", 
                cellPtr->index, cellPtr->x, cellPtr->y, cellPtr->left, cellPtr->right, cellPtr->columnSpan);
*/	
	/* go back to beginning of the table */
	fseek (ifp, tableStart, 0);
	RTFSimpleInit ();
	RTFSetPushedChar (prevChar);
	RTFRestoreStack ();

}




/* In RTF, each table row need not start with a table row definition.
 * The next row may decide to use the row definition of the previous
 * row. In that case, I need to call this InheritTableRowDef function
 */
static void
InheritTableRowDef (void)
{
int prevRow;
int cellsInPrevRow;
cell *cellPtr, *newCellPtr;
int i;
char *fn = "InheritTableRowDef";

	prevRow = table.rows;
	cellsInPrevRow = (table.rowInfo)[prevRow];
	
	(table.rowInfo)[prevRow + 1] = (table.rowInfo)[prevRow];
	
	for (i = 0; i < cellsInPrevRow; i++)
	{
		
		cellPtr = GetCellByPos (prevRow, i);
		
		if ((newCellPtr = New (cell)) == (cell *) NULL)
			RTFPanic ("%s: cannot allocate inheriting cell entry", fn);
			
		newCellPtr->nextCell = table.cellInfo;
		newCellPtr->x = prevRow + 1;
		newCellPtr->y = cellPtr->y;
		newCellPtr->left = cellPtr->left;
		newCellPtr->right = cellPtr->right;
		newCellPtr->width = cellPtr->width;
		newCellPtr->index = table.cellCount;
		newCellPtr->mergePar = cellPtr->mergePar;
		table.cellMergePar = none; /* reset */
		table.cellInfo = newCellPtr;
		table.cellCount++;
	
	
	}


}


/*
 This routine figures out how many cells are merged vertically and writes the
 corresponding \multirow statement.
 */
static void
DoMergedCells (cell *cellPtr)
{
int i;
long x, y;
cell *localCellPtr = cellPtr;
char buf[rtfBufSiz];


	x = cellPtr->x;
	y = cellPtr->y;
	
	i = 1;
	localCellPtr = GetCellByPos (x+i, y);
	for (i = 1; localCellPtr->mergePar == previous; i++)
		if (x + i > table.rows -1 ) break;
		else localCellPtr = GetCellByPos (x+i, y);

	sprintf (buf, "\\multirow{%d}{%1.3fin}{%s ", 
			i-1, cellPtr->width, justificationList[paragraph.alignment]);
	PutLitStr (buf);
	
	table.multiRow = true;
	requireMultirowPackage = true;

}

/* This routine sets attributes for the detected cell and 
 * adds it to the table.cellInfo list. Memory for cells is 
 * allocated dynamically as each cell is encountered. 
 */
static void
ReadCell (void)
{
cell *cellPtr;
char *fn = "ReadCell";
	
	
	if ((cellPtr = New (cell)) == (cell *) NULL)
		RTFPanic ("%s: cannot allocate cell entry", fn);

	
	cellPtr->nextCell = table.cellInfo;
	cellPtr->x = table.rows;
	cellPtr->y = (table.rowInfo)[table.rows];
	if (table.cols == 0)
		cellPtr->left = table.leftEdge;
	else 
		cellPtr->left = (table.cellInfo)->right;
	cellPtr->right = rtfParam;
	cellPtr->width = (double)((double)(cellPtr->right) - (double)(cellPtr->left))/rtfTpi;
	cellPtr->index = table.cellCount;
	cellPtr->mergePar = table.cellMergePar;
	table.cellMergePar = none; /* reset */
	table.cellInfo = cellPtr;
	
	if (cellPtr->right <= table.previousColumnValue)
		cellPtr->right = table.previousColumnValue + 100;
	
	table.previousColumnValue = cellPtr->right;
	
}

/* This function searches the cell list by the cell index */
static cell *
GetCellInfo (int cellNum)
{
cell *cellPtr;
char *fn = "GetCellInfo";

	if (cellNum == -1)
		return (table.cellInfo);
	for (cellPtr = (table.cellInfo); cellPtr != (cell *) NULL; cellPtr = cellPtr->nextCell)
	{
		if (cellPtr->index == cellNum)
			break;
	}
	if (cellPtr == (cell *)NULL)
		RTFPanic ("%s: Attempting to access invalid cell at index %d\n", fn, cellNum);
	return (cellPtr);		/* NULL if not found */
}


/* This function searches the cell list by the cell coordinates */
static cell *
GetCellByPos (int x, int y)
{
cell *cellPtr;
char *fn = "GetCellByPos";

	for (cellPtr = (table.cellInfo); cellPtr != (cell *) NULL; cellPtr = cellPtr->nextCell)
	{
		if (cellPtr->x == x && cellPtr->y == y)
			break;
	}
	if (cellPtr == (cell *)NULL)
		RTFPanic ("%s: Attempting to access invalid cell at %d, %d\n", fn, x, y);
	return (cellPtr);		/* NULL if not found */
}


/*
 This function figures out how many columns the cell spans. 
 This is done by comparing the cell's left and right edges 
 to the sorted column border array. If the left and right 
 edges of the cell are not consecutive entries in the array, 
 the cell spans multiple columns.
 */
static int
GetColumnSpan (cell *cellPtr)
{
int i, j;

	/* if this is the last cell in the row, make its right edge
	   flush with the table right edge */
	if (cellPtr->y == ((table.rowInfo)[cellPtr->x]) - 1)
		cellPtr->right = (table.columnBorders)[table.cols];
	
	for (i = 0; i < table.cols; i++)
		if ((table.columnBorders)[i] == cellPtr->left)
			break;
	for (j = i; j < table.cols + 1; j++)
		if ((table.columnBorders)[j] == cellPtr->right)
			break;


return (j-i);
}

/* set paragraph attributes that might be useful */
static void
ParAttr (void)
{
int i;
RTFStyle *stylePtr = NULL;

	if (insideFootnote)
		return;
	
	if (!(table.inside))
	{
		CheckForBeginDocument ();

	switch (rtfMinor)
	{
		case rtfSpaceBetween:
			if (rtfParam == 360)
				paragraph.lineSpacing = oneAndAHalfSpace;
			else if (rtfParam == 480)
				paragraph.lineSpacing = doubleSpace;
			break;
		case rtfQuadCenter:
			paragraph.alignment = center;
			break;
		case rtfQuadJust: case rtfQuadLeft:
			InsertNewLine ();
			wrapCount = 0;
			paragraph.alignment = left;
			break;
		case rtfQuadRight:
			paragraph.alignment = right;
			break;
		case rtfParDef:
			DoParagraphCleanUp ();			
			InitializeTextStyle ();
			for (i = 0; i < charAttrCount; i++)
				PutLitChar ('}');
			charAttrCount = 0;
			paragraph.alignment = left;
			paragraph.newStyle = true;
			break;
		case rtfStyleNum:
			if ((stylePtr = RTFGetStyle (rtfParam)) == (RTFStyle *)NULL)
				break;			
			if (strcmp(stylePtr->rtfSName, "heading 1") == 0)
			{
				InsertNewLine ();
				InsertNewLine ();
				PutLitStr (heading1String);
				writingHeading1 = true;
				wrapCount = strlen (heading1String);
				paragraph.newStyle = false;
			}
			else if (strcmp(stylePtr->rtfSName, "heading 2") == 0)
			{
				InsertNewLine ();
				InsertNewLine ();
				PutLitStr (heading2String);
				writingHeading2 = true;
				wrapCount = strlen (heading2String);
				paragraph.newStyle = false;
			}
			else if (strcmp(stylePtr->rtfSName, "heading 3") == 0)
			{
				InsertNewLine ();
				InsertNewLine ();
				PutLitStr (heading3String);
				writingHeading3 = true;
				wrapCount = strlen (heading3String);
				paragraph.newStyle = false;
			}
			break;
		case rtfFirstIndent:
			paragraph.firstIndent = rtfParam;
			break;
		case rtfLeftIndent:
			paragraph.leftIndent = rtfParam;
			break;
		case rtfRightIndent:
			paragraph.rightIndent = rtfParam;
			break;
	}
	}
	else
	{
		switch (rtfMinor)
		{
			case rtfQuadCenter:
				paragraph.alignment = center;
				break;
			case rtfQuadRight:
				paragraph.alignment = right;
				break;
		}
	}
			
}


static void
WriteParagraphStyle (void)
{
char buf[rtfBufSiz];
double	spacing;
long temp1, temp2, temp3;
double textWidth = page.width - page.leftMargin - page.rightMargin;

	temp1 = paragraph.firstIndent;
	temp2 = paragraph.leftIndent;
	temp3 = paragraph.rightIndent;

	DoParagraphCleanUp ();
	
	paragraph.firstIndent = temp1;
	paragraph.leftIndent = temp2;
	paragraph.rightIndent = temp3;

	if (paragraph.lineSpacing == oneAndAHalfSpace)
		spacing = 1.5;
	else if (paragraph.lineSpacing == doubleSpace)
		spacing	= 2;


	if (!preferenceValue[GetPreferenceNum("ignoreRulerSettings")])
	{
	
		if (paragraph.leftIndent != 0 || paragraph.rightIndent != 0)
		{
			if (paragraph.leftIndent > 0)
			{
				sprintf (buf, "\n\\makebox[%2.3fin]{}",
						 ((double)(paragraph.leftIndent)/(double)rtfTpi));
				PutLitStr (buf);
			}
			sprintf (buf, "\n\\parbox{%2.3fin}\n{", 
					textWidth - 
					((double)(paragraph.rightIndent)/(double)rtfTpi) -
					((double)(paragraph.leftIndent)/(double)rtfTpi));
			PutLitStr (buf);
			paragraph.parbox = true;
		}
	}		
	
	if (paragraph.alignment != left && 
		!preferenceValue[GetPreferenceNum("ignoreParagraphAlignment")])
	{
		sprintf (buf, "\n\\begin{%s}", environmentList[paragraph.alignment]);
		PutLitStr (buf);
		InsertNewLine ();
		paragraph.wroteAlignment = true;
	}
	
	if (paragraph.lineSpacing != singleSpace &&
		!preferenceValue[GetPreferenceNum("ignoreSpacing")])
	{
		sprintf (buf, "\n\\begin{spacing}{%1.1f}", spacing);
		PutLitStr (buf);
		InsertNewLine ();
		paragraph.wroteSpacing = true;
		requireSetspacePackage = true;
	}	
	
	if (!preferenceValue[GetPreferenceNum("ignoreRulerSettings")])
		IndentParagraph ();	
		
	paragraph.newStyle = false;
	wrapCount = 0;
	suppressLineBreak = true;
	

}

static void 
IndentParagraph (void)
{
char buf[100];

	if (paragraph.firstIndent == 0) {
		PutLitStr ("\n\\noindent");
		InsertNewLine ();
	}
	else 
	{
		sprintf (buf, "\n\\setlength{\\parindent}{%2.3fin}",
				(double)((double)(paragraph.leftIndent)/(double)rtfTpi));
		PutLitStr (buf);
		InsertNewLine ();
	}


}


static void 
DoParagraphCleanUp (void)
{
char buf[rtfBufSiz];
int i;

	CheckForBeginDocument ();
//	ExamineToken();
	
	if (charAttrCount > 0)
		CheckForCharAttr ();
	for (i = 0; i < charAttrCount; i++)
		PutLitChar ('}');
	charAttrCount = 0;

	if (writingHeading1)
	{	
		for (i = 0; i < CountCharInString (heading1String, '{'); i++)
			PutLitStr ("}");
		InsertNewLine ();
		wrapCount = 0;
		writingHeading1 = false;
		suppressLineBreak = true;
	}
	if (writingHeading2)
	{	
		for (i = 0; i < CountCharInString (heading2String, '{'); i++)
			PutLitStr ("}");
		InsertNewLine ();
		wrapCount = 0;
		writingHeading2 = false;
		suppressLineBreak = true;
	}
	if (writingHeading3)
	{	
		for (i = 0; i < CountCharInString (heading3String, '{'); i++)
			PutLitStr ("}");
		InsertNewLine ();
		wrapCount = 0;
		writingHeading3 = false;
		suppressLineBreak = true;
	}

	if (insideFootnote)
	{	
		PutLitStr ("}");
		InsertNewLine ();
		wrapCount = 0;
		insideFootnote = false;
		suppressLineBreak = true;
	}

		
	if (paragraph.lineSpacing != singleSpace && paragraph.wroteSpacing)
	{
		PutLitStr ("\n\\end{spacing}");
		InsertNewLine ();
		paragraph.lineSpacing = singleSpace;
		paragraph.wroteSpacing = false;
		suppressLineBreak = true;
	}
	
	/* close previous environment */
	if (paragraph.alignment != left && paragraph.wroteAlignment) 
	{
		sprintf (buf, "\n\\end{%s}", environmentList[paragraph.alignment]);
		PutLitStr (buf);
		InsertNewLine ();
		paragraph.alignment = left;
		paragraph.wroteAlignment = false;
		suppressLineBreak = true;
	}
	
	if (paragraph.parbox)
	{
		PutLitStr ("}");
		InsertNewLine ();
		wrapCount = 0;
		paragraph.parbox = false;
	}

	
	paragraph.firstIndent = 0;
	paragraph.leftIndent = 0;
	paragraph.rightIndent = 0;

	
}

static void
SectAttr (void)
{
	switch (rtfMinor)
	{
		case rtfColumns:
			section.cols = rtfParam;
			section.newStyle = true;
			break;
		case rtfSectDef:
			DoParagraphCleanUp();
			DoSectionCleanUp ();
			section.cols = 1;
//			section.newStyle = true;
			break;
	}
			
}

static void
WriteSectionStyle (void)
{
char buf[rtfBufSiz];
int i;
			
	DoParagraphCleanUp ();
	if (charAttrCount > 0)
	{
		for (i = 0; i < charAttrCount; i++)
			PutLitStr ("}");
		charAttrCount = 0;
	}
	
	if (section.cols > 1)
	{
		sprintf (buf, "\n\\begin{multicols}{%d}", section.cols);
		PutLitStr (buf);
		InsertNewLine ();
		wrapCount = 0;
		requireMultiColPackage = true;
	}
	
	section.newStyle = false;
	suppressLineBreak = true;
	
}

static void
DoSectionCleanUp (void)
{
	if (section.cols > 1)
	{
		PutLitStr ("\n\\end{multicols}");
		InsertNewLine ();
		section.cols = 1;
	}
	
	wrapCount = 0;

}

static void
ReadFieldInst (void)
{
char	buf[100];
int i = 0;
short tokenClass;
short currentCharSet = RTFGetCharSet ();
short major, minor;
int groupCount = 1;

	strcpy (buf, "");

	while (rtfClass != rtfText || rtfMinor == rtfSC_space){
		RTFGetToken ();
		if(RTFCheckCM(rtfGroup, rtfBeginGroup))
			groupCount++;
		if(RTFCheckCM(rtfGroup, rtfEndGroup))
			groupCount--;
		if(groupCount == 0)
		{
			RTFRouteToken();
			return;
		}		
	}			
	
	/* figure out wht the field is */
	/* HYPERLINK and SYMBOL are supported */
	strcat (buf, rtfTextBuf);
	while (strcmp(rtfTextBuf, " ") != 0 && rtfClass != rtfGroup)
	{
		RTFGetToken ();
		if (strcmp(rtfTextBuf, " ") != 0 && rtfClass != rtfGroup)
			strcat (buf, rtfTextBuf);
	}
	
	/* if this is a hyperlink, call the ReadHyperlink() function */
	if (strcmp(buf, "HYPERLINK") == 0 \
		&& !(int)preferenceValue[GetPreferenceNum("ignoreHypertext")])
	{	
		ReadHyperlink ();
		return;
	}
	
	/* We have a symbol; switch our character set */
	if (strcmp(buf, "SYMBOL") == 0)
		RTFSetCharSet (rtfCSSymbol);
	else 
	{
		for (i = 0; i < groupCount; i++)
			RTFSkipGroup ();
		RTFRouteToken ();
		return;
	}
	
	/* go to the start of the symbol representation */
	strcpy (buf, "");
	if ((tokenClass = RTFGetToken ()) != rtfText)
	{
		if (RTFCheckCM(rtfGroup, rtfBeginGroup) != 0)
			RTFSkipGroup ();
		RTFSkipGroup ();
		RTFRouteToken ();
		return;
	}
	
	/* read in the symbol token */
	strcat (buf, rtfTextBuf);
	while (strcmp(rtfTextBuf, " ") != 0)
	{
		RTFGetToken ();
		if (strcmp(rtfTextBuf, " ") != 0)
			strcat (buf, rtfTextBuf);
	}
	
	/* convert the text symbol token to an int */
	major = atoi(buf);
	/* do the mapping */
	minor = RTFMapChar (major);
	
	/* set the rtf token to the new value */
	RTFSetToken (rtfText, major, minor, rtfNoParam, buf);

	if (minor >= rtfSC_therefore && minor < rtfSC_currency)
		requireAmsSymbPackage = true;
	
	/* call the handler for text */
	TextClass();
	
	/* reset our character set */
	RTFSetCharSet (currentCharSet);
	RTFSkipGroup ();
	RTFRouteToken ();
}


static void
ReadHyperlink (void)
{
int localGL;
	
	
	PutLitStr ("\\R2Lurl{");
	wrapCount += 5;
	
	localGL = groupLevel;
	
	insideHyperlink = true;
	
	
	while (groupLevel >= localGL)
	{
		RTFGetToken ();
		if (rtfClass == rtfText && rtfTextBuf[0] != '"' \
			&& !RTFCheckMM(rtfSpecialChar, rtfLDblQuote) \
			&& !RTFCheckMM(rtfSpecialChar, rtfRDblQuote))
			RTFRouteToken ();
	}
	
	PutLitStr ("}{");
	wrapCount += 2;
	
	/* skip over to the result group */
	while (!RTFCheckCMM (rtfControl, rtfDestination, rtfFieldResult))
		RTFGetToken ();
		
	localGL = groupLevel;
	/* switch off hyperlink flag */
	insideHyperlink = false;
	
	
	while (groupLevel >= localGL)
	{
		RTFGetToken ();
		if(RTFCheckCMM(rtfControl, rtfSpecialChar, rtfOptDest))
			RTFSkipGroup();
		textStyle.newStyle = false;
		if (rtfClass == rtfText)
			RTFRouteToken ();
	}
	
	PutLitStr ("}");
	wrapCount++;
	requireHyperrefPackage = true;
	
	
}


static int
CountCharInString (char *theString, char theChar)
{
int i;
int count = 0;
	
	for (i = 0; i < strlen (theString); i++)
		if (theString[i] == theChar)
			count++;
			
	return (count);
	
}

static void
InsertNewLine (void)
{
	PutLitStr ("\n");
	lineIsBlank = true;
}	

