/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file hideout_revealing.cpp
 * \brief Implementation of the hideout_revealing class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/hideout_revealing.hpp"
#include "generic_items/decorative_item.hpp"

BASE_ITEM_EXPORT( hideout_revealing, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::hideout_revealing::hideout_revealing()
  : m_player_in_zone(false), m_definitive_disclosure(false),
    m_revealing_opacity(0), m_hideout_opacity(1),
    m_current_revealed(false), m_last_revealed(false),
    m_hideout_found(false),
    m_first_player(NULL), m_second_player(NULL)
{
  set_phantom(true);
  set_can_move_items(false);
  m_items.clear();
} // hideout_revealing::hideout_revealing()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void
ptb::hideout_revealing::progress( bear::universe::time_type elapsed_time )
{
  super::progress(elapsed_time);

  if ( !m_current_revealed && !m_hideout_found )
    {
      bool player_in_zone = false;
      search_players();
 
      if ( m_first_player.get() != NULL )
        player_in_zone =
          m_first_player->get_bounding_box().intersects(get_bounding_box());

      if ( !player_in_zone )
        {
          if ( m_second_player.get() != NULL )
            player_in_zone =
              m_second_player->get_bounding_box().intersects
              (get_bounding_box());
        }

      if ( player_in_zone )
        {
          reveal();
          m_hideout_found = m_definitive_disclosure;
        } 
    }

  if ( !m_hideout_found )
    {
      if ( !m_current_revealed && m_last_revealed )
        hide();

      m_last_revealed = m_current_revealed;
      m_current_revealed = false;
    }
} // hideout_revealing::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void ptb::hideout_revealing::build()
{
  super::build();

  search_players();
  hide();
} // hideout_revealing::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type list of <*base_item>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::hideout_revealing::set_item_list_field
( const std::string& name, const std::vector<base_item*>& value )
{
  bool result = false;

  if ( name == "hideout_revealing.hiding_items" )
    {
      m_items.clear();
      unsigned int index;
      for ( index = 0; index != value.size(); index++ )
        m_items.push_back(value[index]);
      result = true;
    }
  else
    result = super::set_item_list_field( name, value );

  return result;
} // hideout_revealing::set_item_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c bool.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
ptb::hideout_revealing::set_bool_field( const std::string& name, bool value )
{
  bool result = false;

  if ( name == "hideout_revealing.definitive_disclosure" )
    {
      m_definitive_disclosure = value;
      result = true;
    }
  else
    result = super::set_bool_field( name, value );

  return result;
} // hideout_revealing::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c double.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::hideout_revealing::set_real_field
( const std::string& name, double value )
{
  bool result = false;

  if ( name == "hideout_revealing.revealing_opacity" )
    m_revealing_opacity = value;
  else if ( name == "hideout_revealing.hideout_opacity" )
    m_hideout_opacity = value;
  else
    result = super::set_real_field( name, value );

  return result;
} // hideout_revealing::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Search players.
 */
void ptb::hideout_revealing::search_players()
{
  plee::get_instance_message msg;
  
  if ( m_first_player.get() == NULL )
    {
      get_level_globals().send_message(plee::player_name(1),msg);
      m_first_player = msg.get_instance();
    }
  
  if ( ( game_variables::get_players_count() == 2 ) && 
       ( m_second_player.get() == NULL ) )
    {
      get_level_globals().send_message(plee::player_name(2),msg);
      m_second_player = msg.get_instance();
    }
} // hideout_revealing::search_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Reveal the items.
 */
void ptb::hideout_revealing::reveal()
{
  m_current_revealed = true;

  std::vector<bear::universe::item_handle>::iterator it;
  for (it = m_items.begin(); it != m_items.end(); ++it)
    {
      if ( *it != NULL )
	{
	  bear::decorative_item*
	    deco = dynamic_cast<bear::decorative_item*>(it->get());
	  if ( deco != NULL )
	    deco->get_rendering_attributes().set_opacity(m_revealing_opacity);
	}
    }
} // hideout_revealing::reveal()

/*----------------------------------------------------------------------------*/
/**
 * \brief Hide the items.
 */
void ptb::hideout_revealing::hide()
{
  std::vector<bear::universe::item_handle>::iterator it;
  for (it = m_items.begin(); it != m_items.end(); ++it)
    {
      if ( *it != NULL )
        {
          bear::decorative_item*
            deco = dynamic_cast<bear::decorative_item*>(it->get());
          if ( deco != NULL )
            deco->get_rendering_attributes().set_opacity(m_hideout_opacity);
        }
    }
} // hideout_revealing::hide()
