/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame_start_menu.cpp
 * \brief Implementation of the ptb::frame_start_menu class.
 * \author Julien Jorge
 */
#include "ptb/frame/frame_start_menu.hpp"

#include "ptb/defines.hpp"
#include "ptb/game_variables.hpp"
#include "ptb/frame/message_box.hpp"
#include "engine/game.hpp"

#include <libintl.h>

/*----------------------------------------------------------------------------*/
const unsigned int ptb::frame_start_menu::s_pos_one_player = 0;
const unsigned int ptb::frame_start_menu::s_pos_two_players_local = 1;
const unsigned int ptb::frame_start_menu::s_pos_load = 4; //3;
const unsigned int ptb::frame_start_menu::s_pos_back = 2; //4;

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer onwing the window.
 * \param use_loading Tell if we use the "loading" level.
 */
ptb::frame_start_menu::frame_start_menu
( windows_layer* owning_layer, bool use_loading )
  : menu_frame(owning_layer), m_use_loading(use_loading)
{
  create_controls();
} // frame_start_menu::frame_start_menu()

/*----------------------------------------------------------------------------*/
/**
 * \brief Validate the selected action.
 */
void ptb::frame_start_menu::validate() const
{
  switch ( get_menu().cursor_position() )
    {
    case s_pos_one_player : start_game(1); break;
    case s_pos_two_players_local : start_game(2); break;
    case s_pos_load :
      show_window( new message_box(m_owning_layer, gettext("Not available")) );
      break;
    case s_pos_back : close_window(); break;
    }
} // frame_start_menu::validate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the static text components.
 */
void ptb::frame_start_menu::create_controls()
{
  std::vector<std::string> strings;

  strings.reserve(5);

  strings.push_back( gettext("One player") );
  strings.push_back( gettext("Two players (local)") );
  //strings.push_back( gettext("Load savestate") );
  strings.push_back( gettext("Back") );

  get_menu().make( strings );
  set_size( get_menu().get_size() );
} // frame_start_menu::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the game.
 * \param p The number of players.
 */
void ptb::frame_start_menu::start_game( unsigned int p ) const
{
  game_variables::set_players_count(p);

  if ( m_use_loading )
    bear::engine::game::get_instance().set_waiting_level
      ( PTB_LOADING_LEVEL_NAME );
  else
    bear::engine::game::get_instance().set_waiting_level
      ( game_variables::get_next_level_name() );
} // frame_start_menu::start_game()
