/*
  Bear Engine - Level editor

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/properties_frame.cpp
 * \brief Implementation of the bf::properties_frame class.
 * \author Julien Jorge
 */
#include "bf/properties_frame.hpp"

#include "bf/gui_level.hpp"
#include "bf/item_class_selection_dialog.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/history/action_set_item_class.hpp"
#include "bf/history/action_set_item_field.hpp"
#include "bf/history/action_set_item_fixed_attribute.hpp"
#include "bf/history/action_set_item_id.hpp"

#include <list>

/**
 * \brief Implement the item_field_edit::set_field_value() methods.
 * \param type The type for which the method is implemented.
 * \remark The method for std::list<type> v is also declared.
 *
 * Those methods just call the template method
 * properties_frame::do_set_field_value().
 */
#define IMPLEMENT_PROXY_SET_FIELD_VALUE(type)                           \
  void bf::properties_frame::set_field_value                            \
  ( item_instance& item, const std::string& name, const type& v )       \
  {                                                                     \
    do_set_field_value(item, name, v);                                  \
  }                                                                     \
                                                                        \
  void bf::properties_frame::set_field_value                            \
  ( item_instance& item, const std::string& name, const std::list<type>& v ) \
  {                                                                     \
    do_set_field_value(item, name, v);                                  \
  }

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent Pointer to the owner.
 */
bf::properties_frame::properties_frame( wxWindow* parent )
  : wxPanel( parent ), m_windows_layout(NULL)
{
  create_controls();

  Fit();
} // properties_frame::properties_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the windows layout of the program.
 * \param layout The layout.
 */
void bf::properties_frame::set_window_layout( windows_layout& layout )
{
  m_windows_layout = &layout;
} // properties_frame::set_window_layout()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the item for which we want the properties.
 * \param item The item instance concerned by this window.
 */
void bf::properties_frame::set_item( item_instance* item )
{
  m_prop->set_item( item );

  if ( m_prop->has_item() )
    {
      m_fixed_box->Enable();
      m_id_text->Enable();
      m_item_class->SetLabel
        ( std_to_wx_string( m_prop->get_item().get_class().get_class_name()) );
      FindWindow( IDC_CHANGE_ITEM_CLASS )->Enable();

      update_controls();
    }
  else
    {
      m_fixed_box->Disable();
      m_id_text->Disable();
      m_item_class->SetLabel(wxEmptyString);
      FindWindow( IDC_CHANGE_ITEM_CLASS )->Disable();
    }
} // properties_frame::set_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the controls.
 */
void bf::properties_frame::update_controls()
{
  const item_class& current_class(m_prop->get_item().get_class());

  if ( current_class.get_fixable() )
    m_fixed_box->SetValue(m_prop->get_item().get_fixed());
  else
    m_fixed_box->Disable();

  m_id_text->SetValue( std_to_wx_string(m_prop->get_item().get_id()) );
} // properties_frame::update_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls of the window.
 */
void bf::properties_frame::create_controls()
{
  m_item_class =
    new wxStaticText(this, wxID_ANY, wxEmptyString, wxDefaultPosition,
                     wxDefaultSize, wxST_NO_AUTORESIZE | wxALIGN_CENTRE);

  m_prop = new item_field_edit( *this, this, IDC_ITEM_PROPERTIES );
  m_fixed_box = new wxCheckBox( this, IDC_FIXED_STATE, _("Fixed") );
  m_id_text =
    new wxTextCtrl( this, IDC_TEXT_IDENTIFIER, wxEmptyString, wxDefaultPosition,
                    wxDefaultSize, wxTE_PROCESS_ENTER );
  m_description = new wxStaticText(this, wxID_ANY, wxEmptyString);

  wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );
  wxBoxSizer* s_sizer = new wxBoxSizer( wxHORIZONTAL );
  wxButton* change_item_class =
    new wxButton( this, IDC_CHANGE_ITEM_CLASS, wxT("..."), wxDefaultPosition,
                  wxSize(30, -1) );

  s_sizer->Add( m_item_class, 1, wxEXPAND );
  s_sizer->Add( change_item_class, 0, wxEXPAND );

  sizer->Add( s_sizer, 0, wxEXPAND );
  sizer->Add( m_prop, 1, wxEXPAND );
  sizer->AddSpacer(5);

  s_sizer = new wxBoxSizer( wxHORIZONTAL );
  s_sizer->Add( new wxStaticText(this, wxID_ANY, _("Id: ")), 0,
                wxALIGN_CENTRE_VERTICAL | wxALL );
  s_sizer->Add( m_id_text, 1, wxEXPAND );
  s_sizer->AddSpacer(5);
  s_sizer->Add( m_fixed_box, 0, wxEXPAND );

  sizer->Add( s_sizer, 0, wxEXPAND );

  sizer->AddSpacer(5);
  sizer->Add( m_description, 0, wxEXPAND );

  m_id_text->Disable();
  m_fixed_box->Disable();
  change_item_class->Disable();

  SetSizer(sizer);
} // properties_frame::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do the action of deleting a field.
 * \param item The item in which we delete the field.
 * \param name The name of the field.
 */
void bf::properties_frame::delete_field
( item_instance& item, const std::string& name )
{
  const item_class& item_class(m_prop->get_item().get_class());
  const type_field& f = item_class.get_field(name);

  ingame_view* view =
    m_windows_layout->get_current_level_view()->get_ingame_view();

  level_action* action;

  if( f.is_list() )
    switch ( f.get_field_type() )
      {
      case type_field::integer_field_type:
        action =
          new action_set_item_field< std::list<integer_type> >(&item, name);
        break;
      case type_field::u_integer_field_type:
        action =
          new action_set_item_field< std::list<u_integer_type> >(&item, name);
        break;
      case type_field::real_field_type:
        action =
          new action_set_item_field< std::list<real_type> >(&item, name);
        break;
      case type_field::boolean_field_type:
        action =
          new action_set_item_field< std::list<bool_type> >(&item, name);
        break;
      case type_field::string_field_type:
        action =
          new action_set_item_field< std::list<string_type> >(&item, name);
        break;
      case type_field::sprite_field_type:
        action = new action_set_item_field< std::list<sprite> >(&item, name);
        break;
      case type_field::animation_field_type:
        action =
          new action_set_item_field< std::list<animation_file_type> >
          (&item, name);
        break;
      case type_field::item_reference_field_type:
        action =
          new action_set_item_field< std::list<item_reference_type> >
          (&item, name);
        break;
      case type_field::font_field_type:
        action =
          new action_set_item_field< std::list<font_file_type> >
          (&item, name);
        break;
      case type_field::sample_field_type:
        action =
          new action_set_item_field< std::list<sample_file_type> >
          (&item, name);
        break;
      }
  else switch ( f.get_field_type() )
    {
    case type_field::integer_field_type:
      action = new action_set_item_field<integer_type>(&item, name);
      break;
    case type_field::u_integer_field_type:
      action = new action_set_item_field<u_integer_type>(&item, name);
      break;
    case type_field::real_field_type:
      action = new action_set_item_field<real_type>(&item, name);
      break;
    case type_field::boolean_field_type:
      action = new action_set_item_field<bool_type>(&item, name);
      break;
    case type_field::string_field_type:
      action = new action_set_item_field<string_type>(&item, name);
      break;
    case type_field::sprite_field_type:
      action = new action_set_item_field<sprite>(&item, name);
      break;
    case type_field::animation_field_type:
      action = new action_set_item_field<animation_file_type>(&item, name);
      break;
    case type_field::item_reference_field_type:
      action = new action_set_item_field<item_reference_type>(&item, name);
      break;
    case type_field::font_field_type:
      action = new action_set_item_field<font_file_type>(&item, name);
      break;
    case type_field::sample_field_type:
      action = new action_set_item_field<sample_file_type>(&item, name);
      break;
    }

  view->do_action( action );
} // properties_frame::delete_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the identifiers that can be given to a field of type
 *        item_reference_tpe of the edited item.
 * \param id (out) The identifiers.
 * \param f The type of the field currently edited.
 */
void bf::properties_frame::get_item_identifiers
( wxArrayString& id, const type_field& f )
{
  const ingame_view* view =
    m_windows_layout->get_current_level_view()->get_ingame_view();
  layer::const_item_iterator it;
  const layer::const_item_iterator eit(view->get_active_layer().item_end());
  std::list<std::string> valid_classes;
  f.get_set(valid_classes);

  for (it=view->get_active_layer().item_begin(); it!=eit; ++it)
    if ( !it->get_id().empty() )
      if ( it->get_id() != m_prop->get_item().get_id() )
        {
          bool ok(false);
          std::list<std::string>::iterator itv;
          for (itv=valid_classes.begin(); itv!=valid_classes.end(); ++itv)
            ok = it->get_class().inherits_from(*itv);

          if ( ok || valid_classes.empty() )
            id.Add( std_to_wx_string(it->get_id()) );
        }

  id.Sort();
} // properties_frame::get_item_identifiers()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when closing the window.
 * \param event This event occured.
 */
void bf::properties_frame::on_close(wxCloseEvent& event)
{
  if ( event.CanVeto() )
    {
      Hide();
      event.Veto();
    }
} // properties_frame::on_close()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when an item of a list get the focus.
 * \param event The event.
 */
void bf::properties_frame::on_item_focused(wxListEvent& event)
{
  if ( m_prop->has_item() && (event.GetIndex() != -1) )
    {
      const item_class& item(m_prop->get_item().get_class());
      std::string name;

      if( m_prop->get_field_name(event.GetIndex(), name) )
        {
          const type_field& f = item.get_field(name);

          m_description->SetLabel
            ( wxGetTranslation( std_to_wx_string(f.get_description()) ) );
        }
    }
  else
    m_description->SetLabel( wxEmptyString );

  m_description->SetToolTip( m_description->GetLabel() );
} // properties_frame::on_item_focused()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent when the user chek or unchek the fixed box.
 * \param event The event.
 */
void bf::properties_frame::on_change_fixed(wxCommandEvent& event)
{
  ingame_view* view =
    m_windows_layout->get_current_level_view()->get_ingame_view();

  view->do_action
    ( new action_set_item_fixed_attribute
      (&m_prop->get_item(), m_fixed_box->IsChecked()) );
} // properties_frame::on_change_fixed()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user pressed the "Enter" key in the identifier text field.
 * \param event The event.
 */
void bf::properties_frame::on_validate_id(wxCommandEvent& event)
{
  CLAW_PRECOND( m_prop->has_item() );

  const std::string id = wx_to_std_string( m_id_text->GetValue() );
  ingame_view* view =
    m_windows_layout->get_current_level_view()->get_ingame_view();

  if ( id.empty() )
    view->do_action( new action_set_item_id(&m_prop->get_item(), id) );
  else if ( id != m_prop->get_item().get_id() )
    {
      std::pair<bool, layer::const_item_iterator> it =
        view->get_level().find_item_by_id(id);

      if (it.first)
        {
          wxMessageDialog dlg
            ( this, _("This identifier is already in use."),
              _("Bad identifier"), wxID_OK );

          dlg.ShowModal();
        }
      else
        view->do_action( new action_set_item_id(&m_prop->get_item(), id) );
    }
} // properties_frame::on_validate_id()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user wants to change the class of the item.
 * \param event The event.
 */
void bf::properties_frame::on_change_item_class(wxCommandEvent& event)
{
  const std::string class_name(m_prop->get_item().get_class().get_class_name());
  item_class_selection_dialog dlg
    (m_windows_layout->get_item_class_pool(), this, class_name);

  if ( dlg.ShowModal() == wxID_OK )
    if ( dlg.get_class_name() != class_name )
      {
        ingame_view* view =
          m_windows_layout->get_current_level_view()->get_ingame_view();

        view->do_action
          ( new action_set_item_class
            (&m_prop->get_item(),
             m_windows_layout->get_item_class_pool().get_item_class_ptr
             (dlg.get_class_name())) );

        // force a full refresh
        item_instance* item(&m_prop->get_item());
        set_item(NULL);
        set_item(item);
      }
} // properties_frame::on_change_item_class()

IMPLEMENT_PROXY_SET_FIELD_VALUE(integer_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(u_integer_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(real_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(bool_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(string_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(sprite)
IMPLEMENT_PROXY_SET_FIELD_VALUE(animation_file_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(item_reference_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(font_file_type)
IMPLEMENT_PROXY_SET_FIELD_VALUE(sample_file_type)

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::properties_frame, wxPanel)
  EVT_CLOSE( bf::properties_frame::on_close )
  EVT_LIST_ITEM_FOCUSED( bf::properties_frame::IDC_ITEM_PROPERTIES,
                         bf::properties_frame::on_item_focused )
  EVT_CHECKBOX( bf::properties_frame::IDC_FIXED_STATE,
                bf::properties_frame::on_change_fixed )
  EVT_TEXT_ENTER( bf::properties_frame::IDC_TEXT_IDENTIFIER,
                  bf::properties_frame::on_validate_id )
  EVT_BUTTON( bf::properties_frame::IDC_CHANGE_ITEM_CLASS,
              bf::properties_frame::on_change_item_class )
END_EVENT_TABLE()
