/*
    Bear Engine - Editor library

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file xml/value_to_xml.cpp
 * \brief Implementation of the bf::xml::value_to_xml class.
 * \author Julien Jorge
 */
#include "bf/xml/value_to_xml.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Write the XML attributes of a bitmap_rendering_attributes.
 * \param os The stream in which we save the value.
 * \param att The attributes to write.
 */
void
bf::xml::base_sprite_animation_value_to_xml::bitmap_rendering_attributes_xml
( std::ostream& os, const bitmap_rendering_attributes& att )
{
  os << "width='" << att.width() << "' height='" << att.height()
     << "' mirror='";

  if ( att.is_mirrored() )
    os << "true";
  else
    os << "false";

  os << "' flip='" ;

  if ( att.is_flipped() )
    os << "true";
  else
    os << "false";

  os << "' angle='" << att.get_angle();

  os << "' opacity='" << att.get_opacity() << "' red_intensity='"
     << att.get_red_intensity() << "' green_intensity='"
     << att.get_green_intensity() << "' blue_intensity='"
     << att.get_blue_intensity() << "'";
} // base_sprite_animation_value_to_xml::bitmap_rendering_attributes_xml()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write a XML representation of an animation.
 * \param os The stream in which we write.
 * \param anim The animation to write.
 */
void bf::xml::value_to_xml<bf::animation>::write
( std::ostream& os, const animation& anim )
{
  os << "<animation loops='" << anim.get_loops() << "' first_index='"
     << anim.get_first_index() << "' last_index='" << anim.get_last_index()
     << "' loop_back='";

  if ( anim.get_loop_back() )
    os << "true' ";
  else
    os << "false' ";

  bitmap_rendering_attributes_xml(os, anim);
  os << ">\n";

  animation::const_iterator it;

  for ( it=anim.begin(); it!=anim.end(); ++it )
    {
      os << "  <frame duration='" << it->get_duration() << "'>\n";
      os << "    ";
      value_to_xml<sprite>::write(os, it->get_sprite());
      os << "  </frame>\n";
    }

  os << "</animation>\n";
} // value_to_xml::write() [animation]





/*----------------------------------------------------------------------------*/
/**
 * \brief Write a XML representation of a sprite.
 * \param os The stream in which we write.
 * \param spr The sprite to write.
 */
void bf::xml::value_to_xml<bf::sprite>::write( std::ostream& os, const sprite& spr )
{
  os << "<sprite image='" << spr.get_image_name() << "' x='"
     << spr.get_left() << "' y='" << spr.get_top() << "' clip_width='"
     << spr.get_clip_width() << "' clip_height='"
     << spr.get_clip_height() << "' ";

  bitmap_rendering_attributes_xml(os, spr);
  os << "/>\n";
} // value_to_xml::write() [sprite]
