/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file block.cpp
 * \brief Implementation of the bear::block class.
 * \author Sebastien Angibaud
 */
#include "generic_items/block.hpp"

#include "universe/collision_info.hpp"
#include "engine/export.hpp"

BASE_ITEM_EXPORT( block, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
bear::block::block()
{

} // block::block()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the collision is on a solid side and align the other item.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::block::collision_check_and_align
( engine::base_item& that, universe::collision_info& info )
{
  bool align(false);
  double f;

  switch( info.get_collision_side() )
    {
    case universe::zone::bottom_zone:
      align = bottom_side_is_active();
      f = get_bottom_friction();
      break;
    case universe::zone::top_zone:
      align = top_side_is_active();
      f = get_top_friction();
      break;
    case universe::zone::middle_left_zone:
      align = left_side_is_active();
      f = get_left_friction();
      break;
    case universe::zone::middle_right_zone:
      align = right_side_is_active();
      f = get_right_friction();
      break;
    case universe::zone::middle_zone:
      align = true;
      break;
    default: { CLAW_ASSERT( false, "Invalid collision side." ); }
    }

  if ( align && satisfy_collision_condition(that) )
    if ( default_collision(info) )
      {
        that.set_contact_friction(f);
        that.set_system_angle(0);
        z_shift(that);
      }
} // block::collision_check_and_align()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call collision_check_and_align().
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::block::collision
( engine::base_item& that, universe::collision_info& info )
{
  collision_check_and_align(that, info);
} // block::collision()
