/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file visual/code/scene_element.cpp
 * \brief Implementation of the visual::scene_element.
 * \author Julien Jorge
 */
#include "visual/scene_element.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param e The effective element of the scene.
 */
bear::visual::scene_element::scene_element( const base_scene_element& e )
  : m_elem(e.clone())
{

} // scene_element::scene_element()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The instance to copy from.
 */
bear::visual::scene_element::scene_element( const scene_element& that )
  : m_elem(that.m_elem->clone())
{

} // scene_element::scene_element()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::visual::scene_element::~scene_element()
{
  delete m_elem;
} // scene_element::~scene_element()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignment
 * \param that The instance to copy from.
 */
bear::visual::scene_element&
bear::visual::scene_element::operator=( const scene_element& that )
{
  base_scene_element* tmp( that.m_elem->clone() );

  delete m_elem;
  m_elem = tmp;

  return *this;
} // scene_element::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a rectangle where the element is completely opaque.
 */
bear::visual::rectangle_type bear::visual::scene_element::get_opaque_box() const
{
  return m_elem->get_opaque_box();
} // scene_element::get_opaque_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a rectangle bounding the element.
 */
bear::visual::rectangle_type
bear::visual::scene_element::get_bounding_box() const
{
  return m_elem->get_bounding_box();
} // scene_element::get_bounding_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Split the element in sub elements
 * \param boxes The boxes describing how to split the element.
 * \param output The resulting elements (they are inserted to the front.
 */
void bear::visual::scene_element::burst
( const rectangle_list& boxes, scene_element_list& output ) const
{
  m_elem->burst(boxes, output);
} // scene_element::burst()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the element on a screen.
 * \param scr The screen on which we render the element.
 */
void bear::visual::scene_element::render( base_screen& scr ) const
{
  m_elem->render(scr);
} // scene_element::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position on the element on the screen.
 */
const bear::visual::position_type&
bear::visual::scene_element::get_position() const
{
  return m_elem->get_position();
} // scene_element::get_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position on the element on the screen.
 */
bear::visual::position_type& bear::visual::scene_element::get_position()
{
  return m_elem->get_position();
} // scene_element::get_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position on the element on the screen.
 * \param p The new position.
 */
void bear::visual::scene_element::set_position( const position_type& p )
{
  m_elem->set_position(p.x, p.y);
} // scene_element::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position on the element on the screen.
 * \param x The new x-position.
 * \param y The new y-position.
 */
void bear::visual::scene_element::set_position
( coordinate_type x, coordinate_type y )
{
  m_elem->set_position(x, y);
} // scene_element::set_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the factor applied to the element when rendered.
 * \param x The factor on the x-axis.
 * \param y The factor on the y-axis.
 */
void bear::visual::scene_element::set_scale_factor( double x, double y )
{
  m_elem->set_scale_factor(x, y);
} // scene_element::set_scale_factor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the factor applied on the x-axis to the element when rendered.
 */
double bear::visual::scene_element::get_scale_factor_x() const
{
  return m_elem->get_scale_factor_x();
} // scene_element::get_scale_factor_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the factor applied on the y-axis to the element when rendered.
 */
double bear::visual::scene_element::get_scale_factor_y() const
{
  return m_elem->get_scale_factor_y();
} // scene_element::get_scale_factor_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the attributes applied to the element when rendering.
 * \param a The attributes.
 */
void bear::visual::scene_element::set_rendering_attributes
( const bitmap_rendering_attributes& a )
{
  m_elem->set_rendering_attributes(a);
} // scene_element::get_rendering_attributes()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the attributes applied to the element when rendering.
 */
const bear::visual::bitmap_rendering_attributes&
bear::visual::scene_element::get_rendering_attributes() const
{
  return m_elem->get_rendering_attributes();
} // scene_element::get_rendering_attributes()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the attributes applied to the element when rendering.
 */
bear::visual::bitmap_rendering_attributes&
bear::visual::scene_element::get_rendering_attributes()
{
  return m_elem->get_rendering_attributes();
} // scene_element::get_rendering_attributes()
