/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file gl_screen.cpp
 * \brief Implementation of the bear::visual::gl_screen class.
 * \author Julien Jorge
 */
#include "visual/gl_screen.hpp"

#include "visual/gl_image.hpp"

#include <claw/exception.hpp>

#include <SDL/SDL.h>
#include <GL/gl.h>
#include <GL/glext.h>

#include <limits>

/*----------------------------------------------------------------------------*/
/**
 * \brief Global initializations common to all gl_screens. Must be called at the
 *        begining of your program.
 */
void bear::visual::gl_screen::initialize()
{
  if ( !SDL_WasInit(SDL_INIT_VIDEO) )
    if ( SDL_InitSubSystem(SDL_INIT_VIDEO) != 0 )
      throw CLAW_EXCEPTION( SDL_GetError() );

  if ( SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 ) != 0 )
    {
      SDL_QuitSubSystem(SDL_INIT_VIDEO);
      throw CLAW_EXCEPTION( SDL_GetError() );
    }
} // gl_screen::initialize()

/*----------------------------------------------------------------------------*/
/**
 * \brief Global uninitializations common to all gl_screens. Must be called at
 *        the end of your program.
 */
void bear::visual::gl_screen::release()
{
  SDL_QuitSubSystem(SDL_INIT_VIDEO);
} // gl_screen::release()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param size Size of the gl_screen.
 * \param title The title of the window created.
 * \param full Tell if the window is full gl_screen or not.
 */
bear::visual::gl_screen::gl_screen
( const claw::math::coordinate_2d<unsigned int>& size,
  const std::string& title, bool full )
  : m_size(size),
    m_screenshot_buffer( new claw::graphic::rgba_pixel_8[m_size.x * m_size.y] )
{
  fullscreen(full);
  m_need_restoration = false;

  SDL_WM_SetCaption( title.c_str(), NULL );
  glEnable(GL_TEXTURE_2D);

  resize_view(m_size.x, m_size.y);
} // gl_screen::gl_screen() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::visual::gl_screen::~gl_screen()
{
  delete[] m_screenshot_buffer;
} // gl_screen::~gl_screen)

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a new dimension for the resulting projection (doesn't change
 *        the gl_screen's size.
 * \param width Target's width.
 * \param height Target's height.
 */
void
bear::visual::gl_screen::resize_view(unsigned int width, unsigned int height)
{
  glViewport(0, 0, width, height );
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0, m_size.x, m_size.y, 0, 0, 1);
  glMatrixMode(GL_MODELVIEW);

  failure_check( __FUNCTION__ );
} // gl_screen::resize_view()

/*----------------------------------------------------------------------------*/
/**
 * \brief Turn fullscreen mode on/off.
 * \param b Tell if we want a fullgl_screen mode.
 */
void bear::visual::gl_screen::fullscreen( bool b )
{
#ifdef _WIN32
  release();
  initialize();
#endif

  Uint32 flags = SDL_OPENGL | SDL_RESIZABLE;

  if (b)
    flags |= SDL_FULLSCREEN;

  SDL_Surface* s = SDL_SetVideoMode( m_size.x, m_size.y, 32, flags );

  if (!s)
    throw CLAW_EXCEPTION( SDL_GetError() );

  SDL_ShowCursor(0);

  glClearColor(0.0, 0.0, 0.0, 0.0);
  glClearDepth(1.0);

#ifdef _WIN32
  glEnable(GL_TEXTURE_2D);
  resize_view(m_size.x, m_size.y);
  m_need_restoration = true;
#endif
} // gl_screen::fullscreen()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the size of the gl_screen.
 */
claw::math::coordinate_2d<unsigned int>
bear::visual::gl_screen::get_size() const
{
  return m_size;
} // gl_screen::get_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the environment needs to be restored.
 */
bool bear::visual::gl_screen::need_restoration() const
{
  return m_need_restoration;
} // gl_screen::need_restoration()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform that the environment had been restored.
 */
void bear::visual::gl_screen::set_restored()
{
  m_need_restoration = false;
} // gl_screen::set_restored()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the rendering process.
 */
void bear::visual::gl_screen::begin_render()
{
  glClear( GL_COLOR_BUFFER_BIT );
} // gl_screen::begin_render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw a sprite on the gl_screen.
 * \param pos On gl_screen position of the sprite.
 * \param s The sprite to draw.
 */
void bear::visual::gl_screen::render
( const position_type& pos, const sprite& s )
{
  if ( s.has_transparency() )
    {
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
    }

  glColor4f( s.get_red_intensity(), s.get_green_intensity(),
             s.get_blue_intensity(), s.get_opacity() );

  const gl_image* impl = static_cast<const gl_image*>(s.get_image().get_impl());
  glBindTexture( GL_TEXTURE_2D, impl->texture_id() );

  if ( s.get_angle() == 0 )
    {
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    }
  else
    {
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    }

  render_sprite( pos, s );

  if ( s.has_transparency() )
    glDisable(GL_BLEND);

  failure_check( __FUNCTION__ );
} // gl_screen::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop the rendering process.
 */
void bear::visual::gl_screen::end_render()
{
  glFlush();
  SDL_GL_SwapBuffers();
  failure_check( __FUNCTION__ );
} // gl_screen::end_render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw a line.
 * \param color The color of the line.
 * \param p The points of the line.
 * \param w The width of the line.
 */
void bear::visual::gl_screen::draw_line
( const claw::graphic::rgba_pixel& color, const std::vector<position_type>& p,
  double w )
{
  glBindTexture( GL_TEXTURE_2D, 0 );
  glLineWidth(w);

  const GLfloat max =
    std::numeric_limits<claw::graphic::rgba_pixel::component_type>::max();

  if (color.components.alpha != max)
    glEnable(GL_BLEND);

  glBegin(GL_LINE_STRIP);
  {
    glColor4f( (GLfloat)color.components.red / max,
         (GLfloat)color.components.green / max,
         (GLfloat)color.components.blue / max,
         (GLfloat)color.components.alpha / max );

    for ( unsigned int i=0; i!=p.size(); ++i )
      glVertex2i( p[i].x, p[i].y );
  }
  glEnd();

  if (color.components.alpha != max)
    glDisable(GL_BLEND);

  failure_check( __FUNCTION__ );
} // gl_screen::draw_line()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw a filled polygon.
 * \param color The color of the polygon.
 * \param p The points of the polygon.
 */
void bear::visual::gl_screen::draw_polygon
( const claw::graphic::rgba_pixel& color, const std::vector<position_type>& p )
{
  glBindTexture( GL_TEXTURE_2D, 0 );

  const GLfloat max =
    std::numeric_limits<claw::graphic::rgba_pixel::component_type>::max();

  if (color.components.alpha != max)
    glEnable(GL_BLEND);

  glBegin(GL_QUADS);
  {
    glColor4f( (GLfloat)color.components.red / max,
         (GLfloat)color.components.green / max,
         (GLfloat)color.components.blue / max,
         (GLfloat)color.components.alpha / max );

    for ( unsigned int i=0; i!=p.size(); ++i )
      glVertex2i( p[i].x, p[i].y );
  }
  glEnd();

  if (color.components.alpha != max)
    glDisable(GL_BLEND);

  failure_check( __FUNCTION__ );
} // gl_screen::draw_polygon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do a gl_screen shot.
 * \param img The image in which we save the content of the gl_screen.
 */
void bear::visual::gl_screen::shot( claw::graphic::image& img ) const
{
  img.set_size( m_size.x, m_size.y );
  const std::size_t pixels_count(m_size.x * m_size.y);

  glReadPixels
    ( 0, 0, m_size.x, m_size.y, GL_RGBA, GL_UNSIGNED_BYTE,
      m_screenshot_buffer );

  for ( claw::graphic::rgba_pixel_8* it=m_screenshot_buffer;
        it!=m_screenshot_buffer + pixels_count;
        ++it )
    it->components.alpha = 255;

  for (unsigned int y=0; y!=m_size.y; ++y)
    std::copy( m_screenshot_buffer + y * m_size.x,
               m_screenshot_buffer + (y+1) * m_size.x,
               img[m_size.y - y - 1].begin() );

  failure_check( __FUNCTION__ );
} // gl_screen::shot()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render a sprite with transformation (flip or mirror).
 * \param pos On gl_screen position of the sprite.
 * \param s The sprite to draw.
 */
void bear::visual::gl_screen::render_sprite
( const position_type& pos, const sprite& s )
{
  claw::math::box_2d<GLdouble> clip_vertices;
  const claw::math::rectangle<GLdouble> clip_rectangle(s.clip_rectangle());
  const claw::math::coordinate_2d<GLdouble> tex_size(s.get_image().size());
  const GLdouble pixel_width( 1.0 / tex_size.x );
  const GLdouble pixel_height( 1.0 / tex_size.y );

  if ( s.is_mirrored() )
    {
      clip_vertices.second_point.x = clip_rectangle.position.x / tex_size.x;
      clip_vertices.first_point.x =
        (clip_rectangle.right() - clip_rectangle.width * pixel_width)
        / tex_size.x;
    }
  else
    {
      clip_vertices.first_point.x = clip_rectangle.position.x / tex_size.x;
      clip_vertices.second_point.x =
        (clip_rectangle.right() - clip_rectangle.width * pixel_width)
        / tex_size.x;
    }

  if ( s.is_flipped() )
    {
      clip_vertices.second_point.y  = clip_rectangle.position.y / tex_size.y;
      clip_vertices.first_point.y =
        (clip_rectangle.bottom() - clip_rectangle.height * pixel_height)
        / tex_size.y;
    }
  else
    {
      clip_vertices.first_point.y  = clip_rectangle.position.y / tex_size.y;
      clip_vertices.second_point.y =
        (clip_rectangle.bottom() - clip_rectangle.height * pixel_height)
        / tex_size.y;
    }

  typedef claw::math::coordinate_2d<GLdouble> coord_double;
  coord_double render_coord[4];

  const coord_double center = pos + s.get_size() / 2;

  coord_double top_right( pos );
  coord_double bottom_left( pos );
  top_right.x += s.width();
  bottom_left.y += s.height();

  render_coord[0] = rotate( pos, s.get_angle(), center );
  render_coord[1] = rotate( top_right, s.get_angle(), center );
  render_coord[2] = rotate( pos + s.get_size(), s.get_angle(), center );
  render_coord[3] = rotate( bottom_left, s.get_angle(), center );

  render_image( render_coord, clip_vertices );
} // gl_screen::render_sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Draw current texture.
 * \param render_box On gl_screen position and size of the texture.
 * \param clip Part of the texture to draw.
 */
void bear::visual::gl_screen::render_image
( const claw::math::coordinate_2d<GLdouble> render_coord[],
  const claw::math::box_2d<GLdouble>& clip )
{
  glBegin(GL_QUADS);
  {
    // Top-left corner
    glTexCoord2d( clip.first_point.x, clip.first_point.y );
    glVertex2d(render_coord[0].x, render_coord[0].y);

    // Top-right corner
    glTexCoord2d( clip.second_point.x, clip.first_point.y );
    glVertex2d(render_coord[1].x, render_coord[1].y);

    // Bottom-right corner
    glTexCoord2d( clip.second_point.x, clip.second_point.y );
    glVertex2d(render_coord[2].x, render_coord[2].y);

    // Bottom-left corner
    glTexCoord2d( clip.first_point.x, clip.second_point.y );
    glVertex2d(render_coord[3].x, render_coord[3].y);
  }
  glEnd();

  failure_check( __FUNCTION__ );
} // gl_screen::render_image()

/*----------------------------------------------------------------------------*/
claw::math::coordinate_2d<GLdouble> bear::visual::gl_screen::rotate
( const claw::math::coordinate_2d<GLdouble>& pos, GLdouble a,
  const claw::math::coordinate_2d<GLdouble>& center ) const
{
  claw::math::coordinate_2d<GLdouble> result(center);

  result.x +=  (pos.x - center.x) * cos(a) + (pos.y - center.y) * sin(a);
  result.y += -(pos.x - center.x) * sin(a) + (pos.y - center.y) * cos(a);

  return result;
} // gl_screen::rotate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if no error has occured.
 * \param where Information on where this function is called.
 */
void bear::visual::gl_screen::failure_check( const std::string& where ) const
{
  GLenum err = glGetError();

  if ( err != GL_NO_ERROR)
    {
      std::string err_string(where + ": ");

      switch (err)
        {
        case GL_NO_ERROR:
          err_string += "no error (?).";
          break;
        case GL_INVALID_ENUM:
          err_string +=
            "unacceptable value is specified for an enumerated argument.";
          break;
        case GL_INVALID_VALUE:
          err_string += "numeric argument is out of range.";
          break;
        case GL_INVALID_OPERATION:
          err_string += "operation is not allowed in the current state.";
          break;
        case GL_STACK_OVERFLOW: err_string += "stack overflow.";
          break;
        case GL_STACK_UNDERFLOW: err_string += "stack underflow.";
          break;
        case GL_OUT_OF_MEMORY:
          err_string += "not enough memory to execute the command.";
          break;
        case GL_TABLE_TOO_LARGE:
          err_string +=
            "table exceeds the implementation's maximum supported table size";
          break;
        default:
          err_string += "unknow error code.";
        }

      throw claw::exception( err_string );
    }
} // gl_screen::failure_check()
