/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file gui/slider.cpp
 * \brief Implementation of the gui::slider class.
 * \author Julien Jorge
 */
#include <claw/assert.hpp>

#include "visual/scene_sprite.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The control owning this one.
 * \param bar The sprite of the bar on which the slider slides..
 * \param slider The sprite of the slide.
 * \param min The minimum value.
 * \param max The maximum value.
 * \param value The initial value.
 * \param value_changed The callback called when the valued has changed. Will
 *        be deleted in the destructor.
 * \pre min <= max
 */
template<typename T>
bear::gui::slider<T>::slider
( visual_component* owner, const visual::sprite& bar,
  const visual::sprite& slider, T min, T max, T value, callback* value_changed )
  : visual_component(owner), m_bar(bar), m_slider(slider), m_min(min),
    m_max(max), m_value_changed_callback(value_changed)
{
  CLAW_PRECOND(min <= max);

  if ( value <= m_min )
    m_value = m_min;
  else if ( value >= m_max )
    m_value = m_max;
  else
    m_value = value;
} // slider::slider()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
template<typename T>
bear::gui::slider<T>::~slider()
{
  if (m_value_changed_callback != NULL)
    delete m_value_changed_callback;
} // slider::~slider()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the position of the slider.
 * \param v The new value.
 */
template<typename T>
void bear::gui::slider<T>::set_value( T v )
{
  if ( v < m_min )
    m_value = m_min;
  else if ( v > m_max )
    m_value = m_max;
  else
    m_value = v;

  if (m_value_changed_callback != NULL)
    m_value_changed_callback->execute(*this);
} // slider::set_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the position of the slider.
 * \param text The new text.
 */
template<typename T>
T bear::gui::slider<T>::get_value() const
{
  return m_value;
} // slider::get_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the scene elements of the component.
 * \param e (out) The scene elements.
 */
template<typename T>
void bear::gui::slider<T>::display
( std::list<visual::scene_element>& e ) const
{
  claw::math::coordinate_2d<unsigned int> p(top_left());

  if ( height() > m_bar.height() )
    p.y += (height() - m_bar.height()) / 2;

  e.push_back( visual::scene_sprite(p.x, p.y, m_bar) );

  p = top_left();

  if ( height() > m_slider.height() )
    p.y += (height() - m_slider.height()) / 2;

  if (m_min != m_max)
    p.x += (m_slider.width() / 2)
      + (int)( ((m_value - m_min)
                * (width() - 2 * m_slider.width())) / (m_max - m_min) );

  e.push_back( visual::scene_sprite(p.x, p.y, m_slider) );
} // slider::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief The control was resized.
 */
template<typename T>
void bear::gui::slider<T>::on_resized()
{
  m_bar.set_width( width() );
} // slider::on_resized()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the component that a mouse button had been pressed.
 * \param key The value of the pressed button.
 * \param pos The current position of the cursor.
 */
template<typename T>
bool bear::gui::slider<T>::on_mouse_press
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  if ( pos.x <= (m_slider.width() / 2) )
    set_value(m_min);
  else if ( pos.x >= (right() - m_slider.width() / 2) )
    set_value(m_max);
  else
    set_value( m_min + (T)( (pos.x - m_slider.width()) * (m_max - m_min))
               / (T)(width() - m_slider.width()) );

  return true;
} // slider::on_mouse_pressed()
