/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file static_text.cpp
 * \brief Implementation of the bear::gui::static_text class.
 * \author Julien Jorge
 */
#include "gui/static_text.hpp"

#include "visual/scene_writing.hpp"
#include "visual/text_metric.hpp"
#include "visual/text_layout.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param result (out) The length of the longest text.
 *
 * The instance keep the references on each parameter. So they must live longen
 * than \a this.
 */
bear::gui::static_text::arrange_longest_text::arrange_longest_text
( std::size_t& result )
  : m_result(result)
{

} // static_text::arrange_longest_text::arrange_longest_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the index of the last character displayed.
 * \param x The x-position where the word starts.
 * \param y The y-position where the word starts.
 * \param first The first character of the word to display.
 * \param last The character just past the last character to display.
 */
void bear::gui::static_text::arrange_longest_text::operator()
  ( visual::coordinate_type x, visual::coordinate_type y, std::size_t first,
    std::size_t last )
{
  m_result = last;
} // static_text::arrange_longest_text::operator()()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param f The font used to display the text.
 * \param result (out) The size of the control.
 *
 * The instance keep the references on each parameter. So they must live longen
 * than \a this.
 */
bear::gui::static_text::arrange_max_size::arrange_max_size
( const visual::font& f, claw::math::coordinate_2d<unsigned int>& result )
  : m_result(result), m_font(f)
{

} // static_text::arrange_max_size::arrange_max_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compute the maximum size.
 * \param x The x-position where the word starts.
 * \param y The y-position where the word starts.
 * \param first The first character of the word to display.
 * \param last The character just past the last character to display.
 */
void bear::gui::static_text::arrange_max_size::operator()
  ( visual::coordinate_type x, visual::coordinate_type y, std::size_t first,
    std::size_t last )
{
  const unsigned int w( x + (last - first) * m_font->get_size().x + 0.5 );
  const unsigned int h( y + m_font->get_size().y + 0.5 );

  m_result.x = std::max( m_result.x, w );
  m_result.y = std::max( m_result.y, h );
} // static_text::arrange_max_size::operator()()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component containing this component.
 */
bear::gui::static_text::static_text( visual_component* owner )
  : visual_component(owner), m_auto_size(false)
{

} // static_text::static_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The component containing this component.
 * \param f The font used to draw the text.
 */
bear::gui::static_text::static_text( visual_component* owner, font_type f )
  : visual_component(owner), m_font(f), m_auto_size(false)
{
  CLAW_PRECOND( f != NULL );
} // static_text::static_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the Font used to draw the text.
 * \param b The new value.
 */
void bear::gui::static_text::set_font( font_type f )
{
  m_font = f;
  
  if (m_auto_size)
    adjust_size_to_text();

  m_writing.create( m_font, m_text, get_size() );
} // static_text::set_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the control must adjust its size to fit the text.
 * \param b The new value.
 */
void bear::gui::static_text::set_auto_size( bool b )
{
  m_auto_size = b;

  if (m_auto_size)
    adjust_size_to_text();
} // static_text::set_auto_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the text to draw.
 * \param text The text to draw.
 */
void bear::gui::static_text::set_text( const std::string& text )
{
  m_text = text;

  if (m_auto_size)
    adjust_size_to_text();

  m_writing.create( m_font, m_text, get_size() );
} // static_text::set_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the text.
 */
const std::string& bear::gui::static_text::get_text() const
{
  return m_text;
} // static_text::get_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the font used for displaying the text.
 */
bear::gui::static_text::font_type bear::gui::static_text::get_font() const
{
  return m_font;
} // static_text::get_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the height to fit the text.
 */
void bear::gui::static_text::expand_vertically()
{
  if ( m_font != NULL )
    {
      claw::math::coordinate_2d<unsigned int> result(0, 0);
      const visual::size_box_type s
        ( width(), m_font->get_size().y * m_text.length() );

      arrange_max_size func(m_font, result);
      visual::text_layout layout( m_font, m_text, s );
      
      layout.arrange_text( func );

      set_size( result );
    }
} // static_text::expand_vertically()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the length of the longest text that can be displayed in this
 *        component.
 * \param text The text on which we work.
 * \param i The position in \a text
 * \return the position of the first character out of the component.
 */
std::size_t bear::gui::static_text::get_longest_text
( const std::string& text, std::size_t i ) const
{
  std::size_t result;

  if ( m_font == NULL )
    result = text.size() - i;
  else
    {
      arrange_longest_text func(result);
      visual::text_layout layout( m_font, text, get_size() );
      
      layout.arrange_text( func );
    }

  return result;
} // static_text::get_longest_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the scene elements of the component.
 * \param e (out) The scene elements.
 */
void bear::gui::static_text::display
( std::list<visual::scene_element>& e ) const
{
  e.push_back( visual::scene_writing(left(), top(), m_writing) );
} // static_text::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the size of the component to fit the text.
 */
void bear::gui::static_text::adjust_size_to_text()
{
  if ( m_font == NULL )
    set_size(0, 0);
  else
    {
      visual::text_metric tm( m_text, m_font );

      claw::math::coordinate_2d<unsigned int> size( tm.width(), tm.height() );

      set_size( size );
    }
} // static_text::adjust_size_to_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when the size of the component has changed.
 */
void bear::gui::static_text::on_resized()
{
  m_writing.create( m_font, m_text, get_size() );
} // static_text::on_resized()
