/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file gui/checkbox.cpp
 * \brief Implementation of the gui::checkbox class.
 * \author Julien Jorge
 */
#include "gui/checkbox.hpp"

#include "visual/scene_sprite.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The control owning this one.
 * \param off The sprite displayed when the box is not checked.
 * \param on The sprite displayed when the box is checked.
 */
bear::gui::checkbox::checkbox
( visual_component* owner, const visual::sprite& off, const visual::sprite& on )
  : visual_component(owner), m_text(NULL), m_checked(false), m_off(off),
    m_on(on)
{
  create();
} // checkbox::checkbox()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The control owning this one.
 * \param off The sprite displayed when the box is not checked.
 * \param on The sprite displayed when the box is checked.
 * \param f The font used to display the text.
 */
bear::gui::checkbox::checkbox
( visual_component* owner, const visual::sprite& off, const visual::sprite& on,
  font_type f )
  : visual_component(owner), m_text(NULL), m_checked(false), m_off(off),
    m_on(on)
{
  create();
  m_text->set_font(f);
} // checkbox::checkbox()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the font of the text.
 * \param f The new font.
 */
void bear::gui::checkbox::set_font( font_type f )
{
  set_size_maximum();
  m_text->set_font(f);
  fit();
} // checkbox::set_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the text of the control.
 * \param text The new text.
 */
void bear::gui::checkbox::set_text( const std::string& text )
{
  set_size_maximum();
  m_text->set_text(text);
  fit();
} // checkbox::set_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check/uncheck the box.
 * \param b Tell if the box is checked or not.
 */
void bear::gui::checkbox::check( bool b )
{
  m_checked = b;
} // checkbox::check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the value of the box.
 */
void bear::gui::checkbox::toggle_value()
{
  check( !checked() );
} // checkbox::toggle_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the text of the control.
 */
const std::string& bear::gui::checkbox::get_text() const
{
  return m_text->get_text();
} // checkbox::get_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the box is checked or not.
 */
bool bear::gui::checkbox::checked() const
{
  return m_checked;
} // checkbox::checked()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the scene elements of the component.
 * \param e (out) The scene elements.
 */
void bear::gui::checkbox::display( std::list<visual::scene_element>& e ) const
{
  claw::math::coordinate_2d<unsigned int> p(top_left());

  if (m_checked)
    {
      p.y += (height() - m_on.height()) / 2;
      e.push_back( visual::scene_sprite(p.x, p.y, m_on) );
    }
  else
    {
      p.y += (height() - m_off.height()) / 2;
      e.push_back( visual::scene_sprite(p.x, p.y, m_off) );
    }
} // checkbox::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the control.
 */
void bear::gui::checkbox::create()
{
  set_size_maximum();

  m_text = new static_text(this);
  m_text->set_auto_size(true);
  m_text->set_position( std::max(m_off.width(), m_on.width() ) + 5, 0 );

  fit();
} // checkbox::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the size of the control to the size of its sub controls.
 */
void bear::gui::checkbox::fit()
{
  set_size( std::max(m_on.width(), m_off.width()) + m_text->width() + 5,
            std::max( (unsigned int)std::max(m_on.height(), m_off.height()),
                      m_text->height() ) );
} // checkbox::fit()
