/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file level.hpp
 * \brief One level in the game.
 * \author Julien Jorge
 */
#ifndef __ENGINE_LEVEL_HPP__
#define __ENGINE_LEVEL_HPP__

#include <vector>

#include "engine/layer/gui_layer_stack.hpp"
#include "engine/layer/layer.hpp"
#include "engine/level_globals.hpp"
#include "visual/screen.hpp"

#include "engine/class_export.hpp"

#include <claw/multi_type_map.hpp>

namespace bear
{
  namespace engine
  {
    class level_loader;

    /**
     * \brief One level in the game.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT level
    {
    public:
      typedef claw::meta::type_list
      < int,
        claw::meta::type_list
        < unsigned int,
          claw::meta::type_list
          < bool,
            claw::meta::type_list
            < double,
              claw::meta::type_list
              < std::string,
                claw::meta::no_type > > > > > var_types;

      typedef claw::multi_type_map<std::string, var_types> var_map;

    private:
      typedef layer::region_type region_type;
      typedef std::map<universe::const_item_handle, universe::size_box_type>
      activity_map_type;

    public:
      level
      ( const std::string& name, const universe::size_box_type& level_size,
        const std::string& level_music );
      ~level();

      void start();
      void stop();

      void progress( universe::time_type elapsed_time );
      void render( visual::screen& screen ) const;
      visual::scene_element
      element_to_screen_coordinates( const visual::scene_element& e ) const;

      const universe::size_box_type& get_size() const;
      unsigned int get_depth() const;
      const std::string& get_name() const;
      level_globals& get_globals();

      void set_camera( const base_item& cam );
      void add_interest_around( const base_item* item );
      void add_interest_around
      ( const base_item* item, const universe::size_box_type& s );

      void push_layer( layer* the_layer );
      void push_layer( gui_layer* the_layer );

      void set_pause();
      void unset_pause();
      bool is_paused() const;

      void play_music();

      universe::size_box_type get_camera_size() const;
      universe::rectangle_type get_camera_focus() const;
      universe::position_type get_camera_center() const;

      var_map& get_level_variables();
      const var_map& get_level_variables() const;

    private:
      void render
      ( const std::list<scene_visual>& visuals,
        const universe::position_type& cam_pos, visual::screen& screen,
        double r_w, double r_h ) const;
      visual::scene_element element_to_screen_coordinates
      ( const visual::scene_element& e, const universe::position_type& cam_pos,
        double r_w, double r_h ) const;

      void clear();

      void get_layer_region
      ( unsigned int layer_index, region_type& the_region ) const;
      void get_layer_area
      ( unsigned int layer_index, universe::rectangle_type& area ) const;

      void get_active_regions( region_type& active_regions );

      void add_region
      ( region_type& active_regions, const universe::rectangle_type& r,
        const universe::size_box_type& s ) const;

    private:
      /** \brief The name of the level. */
      const std::string m_name;

      /** \brief Visible/active part of the level. */
      universe::const_item_handle m_camera;

      /** \brief A set of items around which the activity is. */
      activity_map_type m_activity;

      /** \brief The layers of the level, from the back to the front. */
      std::vector<layer*> m_layers;

      /** \brief The size of the level. */
      universe::size_box_type m_level_size;

      /** \brief Resources of the level. */
      level_globals* m_level_globals;

      /** \brief The default music to play in the level. */
      std::string m_music;

      /** \brief The identifier of the music when played. */
      std::size_t m_music_id;

      /** \brief The layers of the interface. */
      gui_layer_stack m_gui;

      /** \brief How many times this level is currently paused. */
      unsigned int m_paused;

      /** \brief Variables global to the level. */
      var_map m_level_variables;

    }; // class level
  } // namespace engine
} // namespace bear

#endif // __ENGINE_LEVEL_HPP__
