/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file game.hpp
 * \brief The class managing the levels and the development of the game.
 * \author Julien Jorge
 */
#ifndef __ENGINE_GAME_HPP__
#define __ENGINE_GAME_HPP__

#include <fstream>
#include <queue>

#include "communication/messageable.hpp"
#include "engine/class_export.hpp"
#include "engine/game_description.hpp"
#include "engine/level_globals.hpp"
#include "engine/libraries_pool.hpp"
#include "engine/game_action/game_action.hpp"
#include "engine/game_action/game_action_load_level.hpp"
#include "engine/game_action/game_action_pop_level.hpp"
#include "engine/game_action/game_action_push_level.hpp"
#include "engine/game_action/game_action_set_current_level.hpp"
#include "engine/layer/gui_layer.hpp"
#include "time/time.hpp"
#include "visual/screen.hpp"

#include <claw/arguments_table.hpp>
#include <claw/multi_type_map.hpp>

namespace bear
{
  namespace engine
  {
    class level;

    /**
     * \brief The class managing the levels and the development of the game.
     * \author Julien Jorge
     */
    class ENGINE_EXPORT game
    {
      friend class game_action;
      friend class game_action_load_level;
      friend class game_action_pop_level;
      friend class game_action_push_level;
      friend class game_action_set_current_level;

    public:
      typedef claw::meta::type_list
      < int,
        claw::meta::type_list
        < unsigned int,
          claw::meta::type_list
          < bool,
            claw::meta::type_list
            < double,
              claw::meta::type_list
              < std::string,
                claw::meta::no_type > > > > > var_types;

      typedef claw::multi_type_map<std::string, var_types> var_map;

    private:
      /** \brief Type of the game specific initialisation procedure. */
      typedef void (*init_game_function_type)();

      /** \brief Type of the game specific ending procedure. */
      typedef void (*end_game_function_type)();

      /**
       * \brief Game status.
       */
      enum status
        {
          /** \brief The game is under initialization. */
          status_init,

          /** \brief The game is running. */
          status_run,

          /** \brief We're quiting. */
          status_quit

        }; // enum status

    public:
      static game& get_instance();
      static void print_help();

      game( int& argc, char** &argv );
      ~game();

      void show_fps();

      void run();

      void set_fullscreen( bool full );
      bool get_fullscreen() const;
      void toggle_fullscreen();

      void set_sound_muted( bool m );
      bool get_sound_muted() const;
      void toggle_sound_muted();
      void set_sound_volume( double v );
      double get_sound_volume() const;

      void set_music_muted( bool m );
      bool get_music_muted() const;
      void toggle_music_muted();
      void set_music_volume( double v );
      double get_music_volume() const;

      void toggle_time_step();
      systime::milliseconds_type get_time_step() const;

      void screenshot( claw::graphic::image& img ) const;

      void end();
      void set_waiting_level( const std::string& level_name );
      void set_waiting_level( level* the_level );
      void push_level( const std::string& level_name );
      void pop_level();

      const claw::math::coordinate_2d<unsigned int>& get_screen_size() const;
      double get_active_area_margin() const;
      std::string get_custom_game_file( const std::string& name ) const;

      var_map& get_game_variables();

      const std::string& get_level_file( const std::string& level_name ) const;
      bool level_exists( const std::string& level_name ) const;

      const std::string& get_name() const;

      libraries_pool& get_symbols();

    private:
      void init_game() const;
      void end_game() const;

      std::string get_game_name_as_filename() const;

      std::string get_game_directory() const;
      bool create_game_directory( const std::string& dir ) const;

      void run_level();
      void one_step_beyond();

      void progress( universe::time_type elapsed_time );
      void render();

      void update_inputs();

      void init_environment() const;
      void close_environment() const;

      void load_libraries( const std::list<std::string>& p );
      void init_resource_pool( const std::list<std::string>& p ) const;
      void load_level_files();

      bool do_post_actions();

      void set_current_level( level* the_level );
      void load_level( const std::string& name );
      void close_level();
      void do_push_level( const std::string& name );
      void do_pop_level();

      void start_current_level();

      void clear();

      bool check_arguments( int& argc, char** &argv );

      static claw::arguments_table get_arguments_table();

    protected:
      /** \brief The instance of the game. */
      static game* s_instance;

    private:
      // must be declared before m_game_description
      /** \brief The libraries in which we take custom functions. */
      libraries_pool m_symbols;

      /** \brief The current status of the game. */
      status m_status;

      /** \brief Description of the game. */
      game_description m_game_description;

      /** \brief Global variables of the game. */
      var_map m_game_variables;

      /** \brief The screen. */
      visual::screen* m_screen;

      /** \brief Tell if we are fullscreen or not. */
      bool m_fullscreen;

      /** \brief The current level. */
      level* m_current_level;

      /** \brief A level in abeyance. */
      level* m_level_in_abeyance;

      /** \brief The layer containing all windows. */
      gui_layer* m_gui;

      /** \brief The name of the next level to load, if any. */
      std::string m_waiting_level;

      /** \brief Actions to do once an iteration is done. */
      std::queue<game_action*> m_post_actions;

      /** \brief Number of milliseconds between two iterations. */
      systime::milliseconds_type m_time_step;

      /** \brief The date of the last call to progress. */
      systime::milliseconds_type m_last_progress;

      /** \brief The prefix of the name of the game specific initialization
          procedure. */
      static std::string s_init_game_function_prefix;

      /** \brief The prefix of the name of the game specific ending
          procedure. */
      static std::string s_end_game_function_prefix;

    }; // class game
  } // namespace engine
} // namespace bear

#endif // __ENGINE_GAME_HPP__
