/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file export.hpp
 * \brief Macros and function useful for class exportation.
 * \author Julien Jorge
 */
#ifndef __ENGINE_EXPORT_HPP__
#define __ENGINE_EXPORT_HPP__

#include "engine/base_item.hpp"
#include "engine/layer/layer.hpp"

#ifndef DLL_EXPORT
#  ifdef _WIN32
#    define DLL_EXPORT __declspec(dllexport)
#  else // def _WIN32
#    define DLL_EXPORT
#  endif // def _WIN32
#endif // ndef EXPORT

/*----------------------------------------------------------------------------*/
/**
 * \brief Prepare a class inheriting from engine::base_item.
 *
 * \param class_name The name of the class.
 * \param env The full namespace of this class.
 *
 * Add this macro at the beginning of your class, like:
 *
 *   <tt>namespace my_game
 *       {
 *         class my_class: public bear::engine::base_item*
 *         {
 *           DECLARE_BASE_ITEM( my_class, my_game );
 *
 *           // some stuff...
 *
 *         }; // class my_class
 *       } // namespace my_game</tt>
 *
 * This macro adds the methods and members to get the class name of your class:
 *  - <tt>const char* get_class_name() const;</tt>
 *  - <tt>static const char* static_class_name();</tt>
 *  - <tt>static const char* s_my_class_class_name;</tt>
 *
 * Also, this macro redefines the virtual method
 *  - <tt>base_item* clone() const</tt>
 * of base_item with
 *  - <tt>my_class* clone() const</tt>
 */
#define DECLARE_BASE_ITEM( class_name, env )                            \
  public:                                                               \
  class_name* clone() const { return new class_name(*this); }           \
                                                                        \
  const char* get_class_name() const { return static_class_name(); }    \
                                                                        \
  static const char* static_class_name()                                \
  {                                                                     \
    return s_ ## class_name ## _class_name;                             \
  }                                                                     \
                                                                        \
  private:                                                              \
  static const char* s_ ## class_name ## _class_name

/*----------------------------------------------------------------------------*/
/**
 * \brief Implement common members of aclass inheriting from engine::base_item.
 *
 * \param class_name The name of the class to instanciate.
 * \param env The full namespace of this class.
 *
 * Call this macro in the .cpp file, <b>only if you don't call
 * BASE_ITEM_EXPORT</b>.
 */
#define BASE_ITEM_IMPLEMENT( class_name, env )                          \
  const char* env::class_name::s_ ## class_name ## _class_name =        \
    #env "::" #class_name;

/*----------------------------------------------------------------------------*/
/**
 * \brief Export a function instanciating a class inheriting from
 *        engine::base_item.
 *
 * \param class_name The name of the class to instanciate.
 * \param env The full namespace of this class.
 *
 * The env::class_name class must have a constructor without parameters.
 *
 * The signature of the function declared by this macro is
 *
 *   <tt>engine::base_item* create_<i>class_name</i>()</tt>
 *
 * The name of the function can be obtained by a call to
 * engine::export_function_name( class_name ).
 */
#define BASE_ITEM_EXPORT( class_name, env )                             \
  BASE_ITEM_IMPLEMENT( class_name, env )                                \
                                                                        \
  extern "C"                                                            \
  bear::engine::base_item* DLL_EXPORT                                   \
  create_ ## class_name()                                               \
  {                                                                     \
    return new env::class_name;                                         \
  }

/*----------------------------------------------------------------------------*/
/**
 * \brief Export a function instanciating a class inheriting from
 *        engine::layer.
 *
 * \param class_name The name of the class to instanciate.
 * \param env The full namespace of this class.
 *
 * The env::class_name class must have a constructor taking a const
 * universe::size_box_type& as parameter.
 *
 * The signature of the function declared by this macro is
 *
 *   <tt>engine::base_item*
 *       create_layer_<i>class_name</i>(const universe::size_box_type&)</tt>
 *
 * The name of the function can be obtained by a call to
 * engine::layer_export_function_name( class_name ).
 */
#define LAYER_EXPORT( class_name, env )                                 \
  extern "C"                                                            \
  bear::engine::layer* DLL_EXPORT                                       \
  create_layer_ ## class_name( const bear::universe::size_box_type& s ) \
  {                                                                     \
    return new env::class_name(s);                                      \
  }

/*----------------------------------------------------------------------------*/
namespace bear
{
  namespace engine
  {
    /**
     * \brief Get the name of the exported function made for a class.
     * \param class_name The name of the class that has been exported.
     */
    inline std::string export_function_name( const std::string& class_name )
    {
      // prefix must be the string used for prefix in BASE_ITEM_EXPORT
      return "create_" + class_name;
    } // export_function_name()

    typedef base_item* (*base_export_function_type)();

    /**
     * \brief Get the name of the exported function made for a class.
     * \param class_name The name of the class that has been exported.
     */
    inline std::string
    layer_export_function_name( const std::string& class_name )
    {
      // prefix must be the string used for prefix in LAYER_EXPORT
      return "create_layer_" + class_name;
    } // layer_export_function_name()

    typedef
    layer* (*layer_export_function_type)(const universe::size_box_type& s);

  } // namespace engine
} // namespace bear

#endif // __ENGINE_EXPORT_HPP__
