// --------------------------------------------------------------------
// ipetoipe
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeq.h"
#include "ipemodel.h"
#include "ipeprefs.h"

#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qtextcodec.h>

#include <cstdio>
#include <cstdlib>

using namespace std;

#ifdef WIN32
#include <windows.h>
#endif

#ifdef __BORLANDC__
#ifdef main
#undef main
#endif
#endif

#ifdef WIN32
static void ipeDebugImpl(const char *msg)
{
  OutputDebugStringA(msg);
}
#else
static void ipeDebugImpl(const char *)
{
  // fprintf(stderr, "%s\n", msg);
  // nothing
}
#endif

static int toxml(const char *src, const char *xml)
{
  int reason;
  IpeDocument *doc = IpeDocument::New(src, reason);

  if (!doc) {
    fprintf(stderr, "Could not read Ipe file '%s' (reason %d)\n",
	    src, reason);
    return 1;
  }

  doc->Save(xml, "ipetoipe", IpeDocument::EXml, IpeDocument::ESaveNormal);
  delete doc;
  return 0;
}

static int runlatex(IpeDocument *doc)
{
  QString logFile;
  switch (IpeModel::RunLatex(doc, true, QDir::currentDirPath(), logFile)) {
  case IpeModel::ErrNoText:
    fprintf(stderr, "No text objects in document, no need to run Pdflatex\n");
    return 0;
  case IpeModel::ErrNoDir:
    fprintf(stderr, "Directory '%s' does not exist and cannot be created\n",
	    ((const char *) IpePreferences::Static()->iLatexDir.local8Bit()));
    return 1;
  case IpeModel::ErrWritingSource:
    fprintf(stderr, "Error writing Latex source.\n");
    return 1;
  case IpeModel::ErrOldPdfLatex:
    fprintf(stderr, "Your installed version of Pdflatex is too old.\n");
    return 1;
  case IpeModel::ErrLatex:
    fprintf(stderr, "There were errors trying to run Pdflatex\n");
    return 1;
  case IpeModel::ErrNone:
  default:
    fprintf(stderr, "Pdflatex was run sucessfully.\n");
    return 0;
  }
}

static int topdforps(const char *src, const char *dst,
		     IpeDocument::TFormat fm, uint flags)
{
  int reason;
  IpeDocument *doc = IpeDocument::New(src, reason);

  if (!doc) {
    fprintf(stderr, "Could not read Ipe file '%s' (reason %d)\n",
	    src, reason);
    return 1;
  }
  if (fm == IpeDocument::EEps && doc->TotalViews() > 1) {
    fprintf(stderr,
	    "The document contains %d pages, cannot convert to EPS format.\n",
	    doc->size());
    delete doc;
    return 1;
  }

  fprintf(stderr, "Document %s has %d pages (views)\n",
	  src, doc->TotalViews());

  int exitCode = runlatex(doc);
  if (!exitCode) {

    if ((fm == IpeDocument::EEps || fm == IpeDocument::EPs) &&
	doc->HasTrueTypeFonts()) {
      fprintf(stderr,
	      "The document contains TrueType fonts.\n"
	      "Ipe cannot currently save these in Postscript format.\n"
	      "You can save as PDF, and use pdftops to create a "
	      "Postscript file.\n");
    } else if (!doc->Save(dst, "ipetoipe", fm, flags))
      fprintf(stderr, "Failed to save document!\n");
    else if (flags & IpeDocument::EExport)
      fprintf(stderr, "Warning: the exported file contains no Ipe markup!\n");
  }
  delete doc;
  return exitCode;
}

static void usage()
{
  fprintf(stderr,
	  "Usage: ipetoipe ( -xml | -eps | -pdf | -ps ) [ -export ] "
	  "[ -noshading ] [ -lastview ]"
	  "infile outfile\n"
	  "ipetoipe converts between the different Ipe file formats.\n"
	  "With -export output contains no Ipe markup.\n"
	  "With -lastview only the last view of each Ipe page is saved.\n");
  exit(1);
}

int main(int argc, char *argv[])
{
  // Check Ipelib version mismatch
  if (IpelibVersion() != IPELIB_VERSION) {
    fprintf(stderr,
	    "Ipe has been compiled with header files for Ipelib %d\n"
	    "but is dynamically linked against libipe %d.\n"
	    "Check with 'ldd ipe' which libipe is being loaded, and either "
	    "replace it by the correct version or set LD_LIBRARY_PATH.\n",
	    IPELIB_VERSION, IpelibVersion());
    exit(99);
  }

  // set ipeDebug handler
  ipeDebugHandler = ipeDebugImpl;

  // ensure at least three arguments (handles -help as well :-)
  if (argc < 4)
    usage();

  QString s(argv[1]);

  uint flags = IpeDocument::ESaveNormal;
  int i = 2;
  if (!qstrcmp(argv[i], "-export")) {
    flags |= IpeDocument::EExport;
    ++i;
  }
  if (!qstrcmp(argv[i], "-noshading")) {
    flags |= IpeDocument::ENoShading;
    ++i;
  }

  if (!qstrcmp(argv[i], "-lastview")) {
    flags |= IpeDocument::ELastView;
    ++i;
  }

  // remaining arguments must be two filenames
  if (argc != i + 2)
    usage();

  if (s == "-xml") {
    if (flags & IpeDocument::EExport)
      fprintf(stderr, "An XML file without Ipe markup would be empty, "
	      "ignoring -export flag.\n");
    return toxml(argv[i], argv[i+1]);
  } else if (s == "-pdf") {
    return topdforps(argv[i], argv[i+1], IpeDocument::EPdf, flags);
  } else if (s == "-ps") {
    return topdforps(argv[i], argv[i+1], IpeDocument::EPs, flags);
  } else if (s == "-eps") {
    return topdforps(argv[i], argv[i+1], IpeDocument::EEps, flags);
  } else
    usage();
}

// --------------------------------------------------------------------
