/* GTS - Library for the manipulation of triangulated surfaces
 * Copyright (C) 1999 Stphane Popinet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include "gts.h"

static void prepend_triangle_bbox (GtsTriangle * t, GSList ** bboxes)
{
  *bboxes = g_slist_prepend (*bboxes, 
			     gts_bbox_triangle (gts_bbox_class (), t));
}

int main (int argc, char * argv[])
{
  GtsSurface * s1, * s2;
  GtsSurface * s1out2, * s1in2, * s2out1, * s2in1;
  GtsSurfaceInter * si;
  GNode * tree1, * tree2;
  GSList * bboxes;
  FILE * fp;
  guint line;

  if (argc != 3) {
    fprintf (stderr, 
	     "%s: test boolean operations between surfaces\n"
	     "usage: %s FILE1 FILE2\n",
	     argv[0], argv[0]);
    return 1;
  }

  /* open first file */
  if ((fp = fopen (argv[1], "rt")) == NULL) {
    fprintf (stderr, "%s: can not open file `%s'\n", argv[0], argv[2]);
    return 1;
  }
  /* reads in first surface file */
  s1 = gts_surface_new (gts_surface_class (),
			GTS_FACE_CLASS (gts_nface_class ()),
			GTS_EDGE_CLASS (gts_nedge_class ()),
			GTS_VERTEX_CLASS (gts_nvertex_class ()));
  if ((line = gts_surface_read (s1, fp))) {
    fprintf (stderr, "%s: `%s' is not a valid GTS surface file\n", 
	     argv[0], argv[1]);
    fprintf (stderr, "Format error at line %d\n", line);
    return 1;
  }
  fclose (fp);

 /* open second file */
  if ((fp = fopen (argv[2], "rt")) == NULL) {
    fprintf (stderr, "%s: can not open file `%s'\n", argv[0], argv[3]);
    return 1;
  }
  /* reads in second surface file */
  s2 = gts_surface_new (gts_surface_class (),
			GTS_FACE_CLASS (gts_nface_class ()),
			GTS_EDGE_CLASS (gts_nedge_class ()),
			GTS_VERTEX_CLASS (gts_nvertex_class ()));
  if ((line = gts_surface_read (s2, fp))) {
    fprintf (stderr, "%s: `%s' is not a valid GTS surface file\n", 
	     argv[0], argv[2]);
    fprintf (stderr, "Format error at line %d\n", line);
    return 1;
  }
  fclose (fp);

  /* display summary information about both surfaces */
#if 0
  gts_surface_print_stats (s1, stderr);
  gts_surface_print_stats (s2, stderr);
#endif

  /* build bounding boxes for first surface */
  bboxes = NULL;
  gts_surface_foreach_face (s1, (GtsFunc) prepend_triangle_bbox, &bboxes);
  /* build bounding box tree for first surface */
  tree1 = gts_bb_tree_new (bboxes);
  /* free list of bboxes */
  g_slist_free (bboxes);

  /* build bounding boxes for second surface */
  bboxes = NULL;
  gts_surface_foreach_face (s2, (GtsFunc) prepend_triangle_bbox, &bboxes);
  /* build bounding box tree for second surface */
  tree2 = gts_bb_tree_new (bboxes);
  /* free list of bboxes */
  g_slist_free (bboxes);

  si = gts_surface_inter_new (gts_surface_inter_class (), 
			      s1, s2, tree1, tree2);

  s1out2 = gts_surface_new (gts_surface_class (),
			    gts_face_class (),
			    gts_edge_class (),
			    gts_vertex_class ());
  gts_surface_inter_boolean (si, s1out2, GTS_1_OUT_2);
  gts_surface_print_stats (s1out2, stderr);

  s1in2 = gts_surface_new (gts_surface_class (),
			   gts_face_class (),
			   gts_edge_class (),
			   gts_vertex_class ());
  gts_surface_inter_boolean (si, s1in2, GTS_1_IN_2);
  gts_surface_print_stats (s1in2, stderr);

  s2out1 = gts_surface_new (gts_surface_class (),
			    gts_face_class (),
			    gts_edge_class (),
			    gts_vertex_class ());
  gts_surface_inter_boolean (si, s2out1, GTS_2_OUT_1);
  gts_surface_print_stats (s2out1, stderr);

  s2in1 = gts_surface_new (gts_surface_class (),
			   gts_face_class (),
			   gts_edge_class (),
			   gts_vertex_class ());
  gts_surface_inter_boolean (si, s2in1, GTS_2_IN_1);
  gts_surface_print_stats (s2in1, stderr);

  printf ("(geometry \"1out2\" { =\n");
  gts_surface_write_oogl (s1out2, stdout);
  printf ("})\n"
	  "(normalization \"1out2\" none)\n");

  printf ("(geometry \"1in2\" { =\n");
  gts_surface_write_oogl (s1in2, stdout);
  printf ("})\n"
	  "(normalization \"1in2\" none)\n");

  printf ("(geometry \"2out1\" { =\n");
  gts_surface_write_oogl (s2out1, stdout);
  printf ("})\n"
	  "(normalization \"2out1\" none)\n");

  printf ("(geometry \"2in1\" { =\n");
  gts_surface_write_oogl (s2in1, stdout);
  printf ("})\n"
	  "(normalization \"2in1\" none)\n");

  g_assert (gts_surface_inter_check (si));

  /* destroy surfaces and intersection */
  gts_object_destroy (GTS_OBJECT (s1));
  gts_object_destroy (GTS_OBJECT (s2));
  gts_object_destroy (GTS_OBJECT (si));

  /* destroy bounding box trees (including bounding boxes) */
  gts_bb_tree_destroy (tree1, TRUE);
  gts_bb_tree_destroy (tree2, TRUE);  

  return 0;
}
