/*
 * 2006-11-11 -	Conversion utilities, for creating filenames from GTK+'s UTF-8 strings. Tries to
 *		establish a simplistic "scope", in which filesystem versions of a set of strings
 *		is made available. Don't forget to call conv_end() when going out of the scope
 *		though, as failure to do so will cause memory leaks. Optimized for a single string,
 *		but will handle any number.
*/

#include <errno.h>
#include <stdio.h>

#include "convutil.h"

/* ----------------------------------------------------------------------------------------- */

static void item_set(CItem *ci, const gchar *display, gchar *disk)
{
	ci->in  = display;
	ci->out = disk;
}

static void item_free(CItem *ci)
{
	if(ci == NULL || ci->in == ci->out)	/* Conveniently captures the case when display and disk are NULL. */
		return;
	g_free(ci->out);			/* The output is always the dynamic one. */
}

/* Don't call this on a non-freed item, or memory will be lost. */
static void item_clear(CItem *ci)
{
	ci->in = ci->out = NULL;
}

void conv_init(Conv *c)
{
	if(c == NULL)
		return;
	item_clear(&c->item0);
	c->items = NULL;
	c->size = 0;
}

gchar * conv_begin(Conv *c, const gchar *display)
{
	c->item0.in  = display;
	c->item0.out = g_filename_from_utf8(display, -1, NULL, NULL, NULL);
	c->items = NULL;
	c->size = 1;
	return c->item0.out;
}

gchar * conv_begin_reverse(Conv *c, const gchar *disk)
{
	c->item0.in = disk;
	c->item0.out = g_filename_to_utf8(disk, -1, NULL, NULL, NULL);
	c->items = NULL;
	c->size = 1;
	return c->item0.out;
}

static gchar * item_add(Conv *c, const CItem *item)
{
	if(c->size == 0)
		c->item0 = *item;
	else
	{
		CItem	*n = g_malloc(sizeof *n);
		*n = *item;
		c->items = g_slist_append(c->items, n);
	}
	c->size++;
	return item->out;
}

gchar * conv_add(Conv *c, const gchar *display)
{
	gchar	*disk;
	CItem	item;

	if(c == NULL || display == NULL)
		return NULL;
	if((disk = g_filename_from_utf8(display, -1, NULL, NULL, NULL)) == NULL)
		return NULL;
	item.in = display;
	item.out = disk;
	return item_add(c, &item);
}

gchar * conv_add_reverse(Conv *c, const gchar *disk)
{
	gchar	*display;
	CItem	item;
	int	errno_cache;

	if(c == NULL || disk == NULL)
		return NULL;
	errno_cache = errno;	/* FIXME: g_filename_to_utf() seems to trash errno to 7 even on success. */
	if((display = g_filename_to_utf8(disk, -1, NULL, NULL, NULL)) == NULL)
		return NULL;
	errno = errno_cache;
	item.in = disk;
	item.out = display;
	return item_add(c, &item);
}

gchar * conv_set(Conv *c, guint index, const gchar *display)
{
	gchar	*disk;

	if(c == NULL || index >= c->size || display == NULL)
		return NULL;
	if((disk = g_filename_from_utf8(display, -1, NULL, NULL, NULL)) == NULL)
		return NULL;
	if(index == 0)
	{
		item_free(&c->item0);
		c->item0.in = display;
		c->item0.out = disk;
	}
	else
	{
		GSList	*iter = g_slist_nth(c->items, --index);
		item_free(iter->data);
		item_set(iter->data, display, disk);
	}
	return disk;
}

gchar * conv_get(const Conv *c, guint index)
{
	GSList	*iter;

	if(c == NULL || index >= c->size)
		return NULL;
	if(index == 0)
		return c->item0.out;
	iter = g_slist_nth(c->items, --index);
	return ((CItem *) iter->data)->out;
}

void conv_end(Conv *c)
{
	GSList	*iter;

	if(c == NULL)
		return;
	item_free(&c->item0);
	item_clear(&c->item0);
	if(c->items != NULL)
	{
		for(iter = c->items; iter != NULL; iter = g_slist_next(iter))
			item_free(iter->data);
		g_slist_free(c->items);
		c->items = NULL;
	}
	c->size = 0;
}
