/********************************************************************************
*                                                                               *
*                          P r o f i l e r    C l a s s                         *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: FXProfiler.cpp,v 1.7 2001/01/25 19:48:34 jeroen Exp $                    *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXString.h"
#include "FXProfiler.h"


// Shared dictionary of times
FXDict FXProfiler::dict;


// Shared sorting flags
FXuint FXProfiler::flags=0;


// Record start time
FXProfiler::FXProfiler(const FXchar* func) {
  ps=(FXProfilerStruct*)dict.find(func);
  if(!ps){
    ps=new FXProfilerStruct;
    ps->name=func;
    ps->ncalls=0;
    ps->totaltime=0.0;
    ps->avgtime=0.0;
    dict.insert(func,ps);
    }
  ps->ncalls++;
  startClock();
  }


// Start clock
void FXProfiler::startClock(){
#ifdef WIN32
  QueryPerformanceCounter(&start);
#else
  gettimeofday(&start,NULL);
#endif
  }


// Stop clock
void FXProfiler::stopClock(){
#ifdef WIN32
  LARGE_INTEGER frequency,stop;
  QueryPerformanceFrequency(&frequency);
  QueryPerformanceCounter(&stop);
  FXdouble elapsed=((FXdouble)(stop.QuadPart-start.QuadPart))/(FXdouble)frequency.QuadPart;
#else
  struct timeval stop;
  gettimeofday(&stop,NULL);
  FXdouble elapsed=1E-6*(FXdouble)(stop.tv_usec-start.tv_usec)+(FXdouble)(stop.tv_sec-start.tv_sec);
#endif
  ps->totaltime+=elapsed;
  }


// Erase all of the table entries and start over
void FXProfiler::reset(){
  FXint s=dict.first();
  while(s<dict.size()){
    FXProfilerStruct *ps=(FXProfilerStruct*)dict.data(s);
    delete ps;
    s=dict.next(s);
    }
  dict.clear();
  }


// Comparison function for qsort()
int FXProfiler::compare(const void* elem1,const void* elem2){
  FXProfilerStruct *ps1=(FXProfilerStruct*)elem1;
  FXProfilerStruct *ps2=(FXProfilerStruct*)elem2;
  if(flags&PROFILER_SORT_BY_NAME){
    if(ps1->name<ps2->name) return -1;
    else if(ps1->name==ps2->name) return 0;
    else return 1;
    }
  else if(flags&PROFILER_SORT_BY_HITS){
    if(ps1->ncalls<ps2->ncalls) return -1;
    else if(ps1->ncalls==ps2->ncalls) return 0;
    else return 1;
    }
  else if(flags&PROFILER_SORT_BY_TOTAL_TIME){
    if(ps1->totaltime<ps2->totaltime) return -1;
    else if(ps1->totaltime==ps2->totaltime) return 0;
    else return 1;
    }
  else{
    if(ps1->avgtime<ps2->avgtime) return -1;
    else if(ps1->avgtime==ps2->avgtime) return 0;
    else return 1;
    }
  }


// Report interesting stuff
void FXProfiler::speak(FXProfilerOptions opts) {
  if(dict.no()>0){
    const FXchar* fmt1="%-30s  %9s  %15s  %15s\n";
    const FXchar* fmt2="%-30s  %9d  %15e  %15e\n";
    fxmessage(fmt1,"        Function Name","  Hits   ","Total Time (s)","Avg. Time (s)");
    fxmessage(fmt1,"------------------------------","---------","---------------","---------------");

    // Yes, this is inefficient, but who cares?
    FXProfilerStruct *entries=new FXProfilerStruct[dict.no()];
    flags=opts;
    FXint s=dict.first(),c=0;
    while(s<dict.size()){
      FXProfilerStruct *ps=(FXProfilerStruct*)dict.data(s);
      entries[c].name=ps->name;
      entries[c].ncalls=ps->ncalls;
      entries[c].totaltime=ps->totaltime;
      entries[c].avgtime=ps->totaltime/ps->ncalls;
      s=dict.next(s);
      c++;
      }
    
    // Sort this list
    qsort(entries,dict.no(),sizeof(FXProfilerStruct),FXProfiler::compare);
    
    // Dump it to whereever fxmessage() goes
    for(FXint i=0; i<dict.no(); i++)
      fxmessage(fmt2,entries[i].name.text(),entries[i].ncalls,entries[i].totaltime,entries[i].avgtime);
    
    // Free the list
    delete[] entries;
    }
  }


// Record end time
FXProfiler::~FXProfiler() {
  stopClock();
  }
