/********************************************************************************
*                                                                               *
*      F i l e   I n f o r m a t i o n   a n d   M a n i p u l a t i o n        *
*                                                                               *
*********************************************************************************
* Copyright (C) 2000,2001 by Jeroen van der Zijp.   All Rights Reserved.        *
*********************************************************************************
* Contributed by: Sean Hubbell                                                  *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: FXFile.cpp,v 1.97 2001/08/29 04:12:48 jeroen Exp $                       *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXFile.h"



/*
  Notes:
  - Windows flavors of some of these functions are not perfect yet.

  - Windows 95 and NT:

      -  1 to 255 character name.
      -  Complete path for a file or project name cannot exceed 259
         characters, including the separators.
      -  May not begin or end with a space.
      -  May not begin with a $
      -  May contain 1 or more file extensions (eg. MyFile.Ext1.Ext2.Ext3.Txt).
      -  Legal characters in the range of 32 - 255 but not ?"/\<>*|:
      -  Filenames may be mixed case.
      -  Filename comparisons are case insensitive (eg. ThIs.TXT = this.txt).

  - MS-DOS and Windows 3.1:

      -  1 to 11 characters in the 8.3 naming convention.
      -  Legal characters are A-Z, 0-9, Double Byte Character Set (DBCS)
         characters (128 - 255), and _^$~!#%&-{}@'()
      -  May not contain spaces, 0 - 31, and "/\[]:;|=,
      -  Must not begin with $
      -  Uppercase only filename.

  - Perhaps use GetEnvironmentVariable instead of getenv?
*/


#ifndef TIMEFORMAT
#define TIMEFORMAT "%m/%d/%Y %H:%M:%S"
#endif

/*******************************************************************************/

  
// Return value of environment variable name
FXString FXFile::getEnvironment(const FXString& name){
  return FXString(getenv(name.text()));
  }


// Get current working directory
FXString FXFile::getCurrentDirectory(){
  FXchar buffer[MAXPATHLEN];
#ifndef WIN32
  if(getcwd(buffer,MAXPATHLEN)) return FXString(buffer);
#else
  if(GetCurrentDirectory(MAXPATHLEN,buffer)) return FXString(buffer);
#endif
  return FXString::null;
  }


// Change current directory
FXbool FXFile::setCurrentDirectory(const FXString& path){
#ifdef WIN32
  return !path.empty() && SetCurrentDirectory(path.text());
#else
  return !path.empty() && chdir(path.text())==0;
#endif
  }


// Get current drive prefix "a:", if any
// This is the same method as used in VC++ CRT.
FXString FXFile::getCurrentDrive(){
#ifdef WIN32
  FXchar buffer[MAXPATHLEN];
  if(GetCurrentDirectory(MAXPATHLEN,buffer) && isalpha((FXuchar)buffer[0]) && buffer[1]==':'){
    buffer[0]=tolower((FXuchar)buffer[0]);
    return FXString(buffer,2);
    }
#endif
  return FXString::null;
  }


// Change current drive prefix "a:"
// This is the same method as used in VC++ CRT.
FXbool FXFile::setCurrentDrive(const FXString& prefix){
#ifdef WIN32
  FXchar buffer[3];
  if(!prefix.empty() && isalpha((FXuchar)prefix[0]) && prefix[1]==':'){
    buffer[0]=prefix[0];
    buffer[1]=':';
    buffer[2]='\0';
    return SetCurrentDirectory(buffer);
    }
  return FALSE;
#else
  return TRUE;
#endif
  }


// Get home directory for a given user
FXString FXFile::getUserDirectory(const FXString& user){
#ifndef WIN32
  register struct passwd *pwd;
  if(user.empty()){
    register const FXchar* str;
    if((str=getenv("HOME"))!=NULL) return str;
    if((str=getenv("USER"))!=NULL || (str=getenv("LOGNAME"))!=NULL){
      if((pwd=getpwnam(str))!=NULL) return pwd->pw_dir;
      }
    if((pwd=getpwuid(getuid()))!=NULL) return pwd->pw_dir;
    return PATHSEPSTRING;
    }
  if((pwd=getpwnam(user.text()))!=NULL) return pwd->pw_dir;
  return PATHSEPSTRING;
#else
  if(user.empty()){
    register const FXchar *str1,*str2;
    if((str1=getenv("HOME"))!=NULL) return str1;
    if((str2=getenv("HOMEPATH"))!=NULL){      // This should be good for WinNT, Win2K according to MSDN       
      if((str1=getenv("HOMEDRIVE"))==NULL) str1="c:";
      return FXString(str1,str2);
      }
//    FXchar buffer[MAX_PATH]
//    if(SHGetFolderPath(NULL,CSIDL_PERSONAL|CSIDL_FLAG_CREATE,NULL,O,buffer)==S_OK){
//      return buffer; 
//      }
    HKEY hKey;
    if(RegOpenKeyEx(HKEY_CURRENT_USER,"Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders",0,KEY_READ,&hKey)==ERROR_SUCCESS){
      FXchar home[MAXPATHLEN];
      DWORD size=MAXPATHLEN;
      LONG result=RegQueryValueEx(hKey,"Personal",NULL,NULL,(LPBYTE)home,&size);  // Change "Personal" to "Desktop" if you want...
      RegCloseKey(hKey);
      if(result==ERROR_SUCCESS) return home;
      }
    return "c:" PATHSEPSTRING;
    }
  return "c:" PATHSEPSTRING;
#endif
  }


// Return the home directory for the current user.
FXString FXFile::getHomeDirectory(){
  return getUserDirectory(FXString::null);
  }


// Get executable path
FXString FXFile::getExecPath(){
  return FXString(getenv("PATH"));
  }


// Return directory part of pathname, assuming full pathname.
// Note that directory("/bla/bla/") is "/bla/bla" and NOT "/bla".
// However, directory("/bla/bla") is "/bla" as we expect!
FXString FXFile::directory(const FXString& file){
  register FXint n,i;
  if(!file.empty()){
    i=0;
#ifdef WIN32
    if(isalpha((FXuchar)file[0]) && file[1]==':') i=2;
#endif
    if(ISPATHSEP(file[i])) i++;
    n=i;
    while(file[i]){
      if(ISPATHSEP(file[i])) n=i;
      i++;
      }
    return FXString(file.text(),n);
    }
  return FXString::null;
  }


// Return name and extension part of pathname.
// Note that name("/bla/bla/") is "" and NOT "bla". 
// However, name("/bla/bla") is "bla" as we expect!
FXString FXFile::name(const FXString& file){
  register FXint f,n;
  if(!file.empty()){
    n=0;
#ifdef WIN32
    if(isalpha((FXuchar)file[0]) && file[1]==':') n=2;
#endif
    f=n;
    while(file[n]){
      if(ISPATHSEP(file[n])) f=n+1;
      n++;
      }
    return FXString(file.text()+f,n-f);
    }
  return FXString::null;
  }


// Return file title, i.e. document name only:
//
//  /path/aa        -> aa
//  /path/aa.bb     -> aa
//  /path/aa.bb.cc  -> aa.bb
//  /path/.aa       -> .aa
FXString FXFile::title(const FXString& file){
  register FXint f,e,b,i;
  if(!file.empty()){
    i=0;
#ifdef WIN32
    if(isalpha((FXuchar)file[0]) && file[1]==':') i=2;
#endif
    f=i;
    while(file[i]){
      if(ISPATHSEP(file[i])) f=i+1;
      i++;
      }
    b=f;
    if(file[b]=='.') b++;     // Leading '.'
    e=i;
    while(b<i){
      if(file[--i]=='.'){ e=i; break; }
      }
    return FXString(file.text()+f,e-f);
    }
  return FXString::null;
  }


// Return extension, if there is one:
//
//  /path/aa        -> ""
//  /path/aa.bb     -> bb
//  /path/aa.bb.cc  -> cc
//  /path/.aa       -> ""
FXString FXFile::extension(const FXString& file){
  register FXint f,e,i,n;
  if(!file.empty()){
    n=0;
#ifdef WIN32
    if(isalpha((FXuchar)file[0]) && file[1]==':') n=2;
#endif
    f=n;
    while(file[n]){
      if(ISPATHSEP(file[n])) f=n+1;
      n++;
      }
    if(file[f]=='.') f++;     // Leading '.'
    e=i=n;
    while(f<i){
      if(file[--i]=='.'){ e=i+1; break; }
      }
    return FXString(file.text()+e,n-e);
    }
  return FXString::null;
  }


// Return file name less the extension
//
//  /path/aa        -> /path/aa
//  /path/aa.bb     -> /path/aa
//  /path/aa.bb.cc  -> /path/aa.bb
//  /path/.aa       -> /path/.aa
FXString FXFile::stripExtension(const FXString& file){
  register FXint f,e,n;
  if(!file.empty()){
    n=0;
#ifdef WIN32
    if(isalpha((FXuchar)file[0]) && file[1]==':') n=2;
#endif
    f=n;
    while(file[n]){
      if(ISPATHSEP(file[n])) f=n+1;
      n++;
      }
    if(file[f]=='.') f++;     // Leading '.'
    e=n;
    while(f<n){
      if(file[--n]=='.'){ e=n; break; }
      }
    return FXString(file.text(),e);
    }
  return FXString::null;
  }


// Return drive letter prefix "c:"
FXString FXFile::drive(const FXString& file){
#ifdef WIN32
  FXchar buffer[3];
  if(isalpha((FXuchar)file[0]) && file[1]==':'){
    buffer[0]=tolower((FXuchar)file[0]);
    buffer[1]=':';
    buffer[2]='\0';
    return FXString(buffer,2);
    }
#endif
  return FXString::null;
  }


// Perform tilde or environment variable expansion
FXString FXFile::expand(const FXString& file){
#ifndef WIN32
  register FXint b,e,n;
  FXString result;

  // Expand leading tilde of the form ~/filename or ~user/filename
  n=0;
  if(file[n]=='~'){
    n++;
    b=n;
    while(file[n] && !ISPATHSEP(file[n])) n++;
    e=n;
    result.append(getUserDirectory(file.mid(b,e-b)));
    }

  // Expand environment variables of the form $HOME, ${HOME}, or $(HOME)
  while(file[n]){
    if(file[n]=='$'){
      n++;
      if(file[n]=='{' || file[n]=='(') n++;
      b=n;
      while(isalnum((FXuchar)file[n]) || file[n]=='_') n++;
      e=n;
      if(file[n]=='}' || file[n]==')') n++;
      result.append(getEnvironment(file.mid(b,e-b)));
      continue;
      }
    result.append(file[n]);
    n++;
    }
  return result;
#else
  FXchar buffer[2048];

  // Expand environment variables of the form %HOMEPATH%
  if(ExpandEnvironmentStrings(file.text(),buffer,sizeof(buffer))){
    return buffer;
    }
  return file;
#endif
  }


// Simplify a file path; the path will remain relative if it was relative
// For example, simplify("..//aaa/./bbb//../c/") becomes "../aaa/c/".
// Note that a single trailing "/" is not removed!
FXString FXFile::simplify(const FXString& file){
  if(!file.empty()){
    FXString result=file;
    FXint beg,end,head,cur;
    head=0;
#ifndef WIN32
    if(ISPATHSEP(result[0])){
      head++;
      }
    else{
      while(result[head]=='.' && result[head+1]=='.' && ISPATHSEP(result[head+2])) head+=3;
      if(result[head]=='.' && result[head+1]=='.' && result[head+2]==0) head+=2;
      }
#else
    if(ISPATHSEP(result[0])){
      head++;
      if(ISPATHSEP(result[1])) head++;      // UNC
      }
    else if(isalpha((FXuchar)result[0]) && result[1]==':'){
      head+=2;
      if(ISPATHSEP(result[2])) head++;
      }
    else{
      while(result[head]=='.' && result[head+1]=='.' && ISPATHSEP(result[head+2])) head+=3;
      if(result[head]=='.' && result[head+1]=='.' && result[head+2]==0) head+=2;
      }
#endif
    for(beg=end=cur=head; result[beg]; beg=end){
      while(result[end] && !ISPATHSEP(result[end])) end++;
      if(beg+1==end && result[beg]=='.'){
        }
      else if(beg+2==end && result[beg]=='.' && result[beg+1]=='.'){
        if(head<cur) cur--;
        while(head<cur && !ISPATHSEP(result[cur-1])) cur--;
        }
      else if(beg<end){
        while(beg<end){ result[cur++]=result[beg++]; }
        if(result[end]){ result[cur++]=PATHSEP; }
        }
      while(result[end] && ISPATHSEP(result[end])) end++;
      }
    result[cur]=0;
    return result;
    }
  return FXString::null;
  }


// Build absolute pathname
FXString FXFile::absolute(const FXString& file){
  FXString pathfile=FXFile::expand(file);
#ifndef WIN32
  if(ISPATHSEP(pathfile[0])) return FXFile::simplify(pathfile);
#else
  if(ISPATHSEP(pathfile[0])){
    if(ISPATHSEP(pathfile[1])) return FXFile::simplify(pathfile);   // UNC
    return FXFile::simplify(getCurrentDrive()+pathfile);
    }
  if(isalpha((FXuchar)pathfile[0]) && pathfile[1]==':'){
    if(ISPATHSEP(pathfile[2])) return FXFile::simplify(pathfile);
    return FXFile::simplify(pathfile.mid(0,2)+PATHSEPSTRING+pathfile.mid(2,MAXPATHLEN));
    }
#endif
  return FXFile::simplify(getCurrentDirectory()+PATHSEPSTRING+pathfile);
  }


// Build absolute pathname from parts
FXString FXFile::absolute(const FXString& base,const FXString& file){
  FXString pathfile=FXFile::expand(file);
#ifndef WIN32
  if(ISPATHSEP(pathfile[0])) return FXFile::simplify(pathfile);
#else
  if(ISPATHSEP(pathfile[0])){
    if(ISPATHSEP(pathfile[1])) return FXFile::simplify(pathfile);   // UNC
    return FXFile::simplify(getCurrentDrive()+pathfile);
    }
  if(isalpha((FXuchar)pathfile[0]) && pathfile[1]==':'){
    if(ISPATHSEP(pathfile[2])) return FXFile::simplify(pathfile);
    return FXFile::simplify(pathfile.mid(0,2)+PATHSEPSTRING+pathfile.mid(2,MAXPATHLEN));
    }
#endif
  return FXFile::simplify(FXFile::absolute(base)+PATHSEPSTRING+pathfile);
  }


// Return relative path of file to given base directory
//
// Examples:
//
//  Base       File         Result
//  /a/b/c     /a/b/c/d     d
//  /a/b/c/    /a/b/c/d     d
//  /a/b/c/d   /a/b/c       ../
//  ../a/b/c   ../a/b/c/d   d
//  /a/b/c/d   /a/b/q       ../../q
//  /a/b/c     /a/b/c       .
//  /a/b/c/    /a/b/c/      .
//  ./a        ./b          ../b
//  a          b            ../b
FXString FXFile::relative(const FXString& base,const FXString& file){
  register FXint p,q,b;
  FXString result;
  
  // Find branch point
#ifndef WIN32
  for(p=b=0; base[p] && base[p]==file[p]; p++){
    if(ISPATHSEP(file[p])) b=p;
    }
#else
  for(p=b=0; base[p] && tolower((FXuchar)base[p])==tolower((FXuchar)file[p]); p++){
    if(ISPATHSEP(file[p])) b=p;
    }
#endif
  
  // Paths are equal
  if((base[p]=='\0' || (ISPATHSEP(base[p]) && base[p+1]=='\0')) && (file[p]=='\0' || (ISPATHSEP(file[p]) && file[p+1]=='\0'))){
    return ".";
    }
    
  // Directory base is prefix of file
  if((base[p]=='\0' && ISPATHSEP(file[p])) || (file[p]=='\0' && ISPATHSEP(base[p]))){
    b=p;
    }
  
  // Up to branch point
  for(p=q=b; base[p]; p=q){
    while(base[q] && !ISPATHSEP(base[q])) q++;
    if(q>p) result.append(".." PATHSEPSTRING);
    while(base[q] && ISPATHSEP(base[q])) q++;
    }
  
  // Strip leading path character off, if any
  while(ISPATHSEP(file[b])) b++;
  
  // Append tail end
  result.append(&file[b]);
  
  return result;
  }


// Return relative path of file to the current directory
FXString FXFile::relative(const FXString& file){
  return FXFile::relative(getCurrentDirectory(),file);
  }


// Generate unique filename of the form pathnameXXX.ext, where
// pathname.ext is the original input file, and XXX is a number,
// possibly empty, that makes the file unique.
// (From: Mathew Robertson <mathew.robertson@mi-services.com>)
FXString FXFile::unique(const FXString& file){
  if(!exists(file)) return file;
  FXString ext=extension(file);
  FXString path=stripExtension(file);           // Use the new API (Jeroen)
  FXString filename;
  register FXint count=0;
  if(!ext.empty()) ext.prepend('.');            // Only add period when non-empty extension
  while(count<1000){
    filename.format("%s%i%s",path.text(),count,ext.text());
    if(!exists(filename)) return filename;      // Return result here (Jeroen)
    count++;
    }
  return FXString::null;
  }


// Search pathlist for file
FXString FXFile::search(const FXString& pathlist,const FXString& file){
  FXString pathfile=FXFile::simplify(FXFile::expand(file));
  FXString path;
  FXint beg,end;
#ifndef WIN32
  if(ISPATHSEP(pathfile[0])){
    if(exists(pathfile)) return pathfile;
    return FXString::null;
    }
#else
  if(ISPATHSEP(pathfile[0]) && ISPATHSEP(pathfile[1])){
    if(exists(pathfile)) return pathfile;   // UNC
    return FXString::null;
    }
  if(isalpha((FXuchar)pathfile[0]) && pathfile[1]==':'){
    if(exists(pathfile)) return pathfile;
    return FXString::null;
    }
//  if(isalpha((FXuchar)pathfile[0]) && pathfile[1]==':' && ISPATHSEP(pathfile[2])){
//    if(exists(pathfile)) return pathfile;
//    return FXString::null;
//    }
  if(ISPATHSEP(pathfile[0])){
    pathfile=getCurrentDrive()+pathfile;
    if(exists(pathfile)) return pathfile;
    return FXString::null;
    }
#endif
  for(beg=0; pathlist[beg]; beg=end){
    while(pathlist[beg]==PATHLISTSEP) beg++;
    for(end=beg; pathlist[end] && pathlist[end]!=PATHLISTSEP; end++);
    if(beg==end) break;
    path=absolute(pathlist.mid(beg,end-beg),pathfile);
    if(exists(path)) return path;
    }
  return FXString::null;
  }


// Up one level, given absolute path
FXString FXFile::upLevel(const FXString& file){
  if(!file.empty()){
    FXint beg=0;
    FXint end=file.length();
#ifndef WIN32
    if(ISPATHSEP(file[0])) beg++;
#else
    if(ISPATHSEP(file[0])){
      beg++;
      if(ISPATHSEP(file[1])) beg++;     // UNC
      }
    else if(isalpha((FXuchar)file[0]) && file[1]==':'){
      beg+=2;
      if(ISPATHSEP(file[2])) beg++;
      }
#endif
    if(beg<end && ISPATHSEP(file[end-1])) end--;
    while(beg<end){ --end; if(ISPATHSEP(file[end])) break; }
    return file.mid(0,end);
    }
  return file;
  }

 
// Check if file represents absolute pathname
FXbool FXFile::isAbsolute(const FXString& file){
  if(!file.empty()){
#ifndef WIN32
    return ISPATHSEP(file[0]);
#else
    return ISPATHSEP(file[0]) || (isalpha((FXuchar)file[0]) && file[1]==':');
#endif
    }
  return FALSE;
  }


// Does file represent topmost directory
FXbool FXFile::isTopDirectory(const FXString& file){
  if(!file.empty()){
#ifndef WIN32
    return (ISPATHSEP(file[0]) && file[1]=='\0');
#else
    return (ISPATHSEP(file[0]) && (file[1]=='\0' || (ISPATHSEP(file[1]) && file[2]=='\0'))) || (isalpha((FXuchar)file[0]) && file[1]==':' && (file[2]=='\0' || (ISPATHSEP(file[2]) && file[3]=='\0')));
#endif
    }
  return FALSE;
  }


// Check if file represents a file
FXbool FXFile::isFile(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && S_ISREG(status.st_mode);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Check if file represents a link
FXbool FXFile::isLink(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && S_ISLNK(status.st_mode);
#else
  return FALSE;
#endif
  }

// Check if file represents a directory
FXbool FXFile::isDirectory(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && S_ISDIR(status.st_mode);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && (atts&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Return true if file is readable (thanks to gehriger@linkcad.com)
FXbool FXFile::isReadable(const FXString& file){
  return !file.empty() && access(file.text(),R_OK)==0;
  }


// Return true if file is writable (thanks to gehriger@linkcad.com)
FXbool FXFile::isWritable(const FXString& file){
  return !file.empty() && access(file.text(),W_OK)==0;
  }


// Return true if file is executable (thanks to gehriger@linkcad.com)
FXbool FXFile::isExecutable(const FXString& file){
  return !file.empty() && access(file.text(),X_OK)==0;
  }


// Check if owner has full permissions
FXbool FXFile::isOwnerReadWriteExecute(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IRUSR) && (status.st_mode&S_IWUSR) && (status.st_mode&S_IXUSR);
#else
  return TRUE;
#endif
  }


// Check if owner can read
FXbool FXFile::isOwnerReadable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IRUSR);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF);
#endif
  }


// Check if owner can write
FXbool FXFile::isOwnerWritable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IWUSR);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_READONLY);
#endif
  }


// Check if owner can execute
FXbool FXFile::isOwnerExecutable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IXUSR);
#else
  return TRUE;
#endif
  }


// Check if group has full permissions
FXbool FXFile::isGroupReadWriteExecute(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IRGRP) && (status.st_mode&S_IWGRP) && (status.st_mode&S_IXGRP);
#else
  return TRUE;
#endif
  }


// Check if group can read
FXbool FXFile::isGroupReadable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IRGRP);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF);
#endif
  }


// Check if group can write
FXbool FXFile::isGroupWritable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IWGRP);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_READONLY);
#endif
  }


// Check if group can execute
FXbool FXFile::isGroupExecutable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IXGRP);
#else
  return TRUE;
#endif
  }


// Check if everybody has full permissions
FXbool FXFile::isOtherReadWriteExecute(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IROTH) && (status.st_mode&S_IWOTH) && (status.st_mode&S_IXOTH);
#else
  return TRUE;
#endif
  }


// Check if everybody can read
FXbool FXFile::isOtherReadable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IROTH);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF);
#endif
  }


// Check if everybody can write
FXbool FXFile::isOtherWritable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IWOTH);
#else
  DWORD atts;
  return !file.empty() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_READONLY);
#endif
  }


// Check if everybody can execute
FXbool FXFile::isOtherExecutable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_IXOTH);
#else
  return TRUE;
#endif
  }


// These 5 functions below contributed by calvin@users.sourceforge.net


// Test if suid bit set
FXbool FXFile::isSetUid(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_ISUID);
#else
  return FALSE;
#endif
  }


// Test if sgid bit set
FXbool FXFile::isSetGid(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_ISGID);
#else
  return FALSE;
#endif
  }


// Test if sticky bit set
FXbool FXFile::isSetSticky(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) && (status.st_mode&S_ISVTX);
#else
  return FALSE;
#endif
  }


// Return owner name of file
FXString FXFile::owner(const FXString& file){
  struct stat status;
  FXchar owner[64];
  if(file.empty() || (::stat(file.text(),&status)!=0)) return FXString::null;
#ifndef WIN32
  fxgetusername(owner,status.st_uid);
#else
  fxgetusername(owner,0);
#endif
  return owner;
  }


// Return group name of file
FXString FXFile::group(const FXString& file){
  struct stat status;
  FXchar group[64];
  if(file.empty() || (::stat(file.text(),&status)!=0)) return FXString::null;
#ifndef WIN32
  fxgetgroupname(group,status.st_gid);
#else
  fxgetgroupname(group,0);
#endif
  return group;
  }


// Return time file was last modified
FXTime FXFile::modified(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)status.st_mtime : 0L;
#else
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)status.st_mtime : 0L;
#endif
  }


// Return time file was last accessed
FXTime FXFile::accessed(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)status.st_atime : 0L;
#else
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)status.st_atime : 0L;
#endif
  }


// Return time when created
FXTime FXFile::created(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)status.st_ctime : 0L;
#else
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)status.st_ctime : 0L;
#endif
  }


// Return time when "touched"
FXTime FXFile::touched(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)FXMAX(status.st_ctime,status.st_mtime) : 0L;
#else
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (FXTime)FXMAX(status.st_ctime,status.st_mtime) : 0L;
#endif
  }


#ifndef WIN32                 // UNIX


// List all the files in directory
FXint FXFile::listFiles(FXString*& list,const FXString& path,const FXString& pattern,FXuint flags){
  register const char *name;
  register struct dirent *dp;
  register FXString *newlist;
  register DIR *dirp;
  register FXint count=0;
  register FXint size=0;
  
  // Initialize to empty
  list=NULL;
    
  // One single root on UNIX
  if(path.empty()){
    list=new FXString[2];
    list[0]=PATHSEPSTRING;
    return 1;
    }
  
  // Get directory stream pointer
  dirp=opendir(path.text());
  if(!dirp) return 0;

  // Loop over directory entries
  while((dp=readdir(dirp))!=NULL){
    name=dp->d_name;

    // Skip '.' and '..'
    //if(name[0]=='.' && (name[1]==0 || (name[1]=='.' && name[2]==0))) continue;

    // Is it a directory or does it match the pattern?
    if(!fxfilematch(pattern.text(),name,flags)) continue;

    // Grow list
    if(count+1>=size){
      size=size?(size<<1):256;
      newlist=new FXString [size];
      for(int i=0; i<count; i++) newlist[i]=list[i];
      delete [] list;
      list=newlist;
      }

    // Add to list
    list[count++]=name;
    }
  closedir(dirp);
  return count;
  }


#else                         // WINDOWS


// List all the files in directory
FXint FXFile::listFiles(FXString*& list,const FXString& path,const FXString& pattern,FXuint flags){
  register const char *name;
  register FXString *newlist;
  register FXint count=0;
  register FXint size=0;
  WIN32_FIND_DATA ffData;
  DWORD drivemask,mask,nCount,nSize,i,j;
  HANDLE hFindFile,hEnum;
  NETRESOURCE host;
  FXchar letter[4],server[200];
  
  // Initialize to empty
  list=NULL;
    
  // Each drive is a root on windows
  if(path.empty()){
    
    // Get drive letter mask
    drivemask=GetLogicalDrives();
    
    // Count drives
    for(mask=drivemask; mask; mask>>=1){ if(mask&1) size++; }
  
    // Allocate root list
    list=new FXString[size+2];
    
    // Initialize string to "a:\"
    letter[0]='a';
    letter[1]=':';
    letter[2]=PATHSEP;
    letter[3]='\0';
    
    // Add drive letters
    for(mask=drivemask; mask; mask>>=1,letter[0]++){
      if(mask&1) list[count++]=letter;
      }
    
    // Add top level for file shares "\\"
    list[count++]=PATHSEPSTRING PATHSEPSTRING;    // UNC for file shares
    
    // Return number of roots
    return count;
    }

  // A UNC name was given of the form "\\" or "\\server"
  if(ISPATHSEP(path[0]) && ISPATHSEP(path[1]) && path.findf(PATHSEP,2)<0){
    
    // Fill in
    host.dwScope=RESOURCE_GLOBALNET;
    host.dwType=RESOURCETYPE_DISK;
    host.dwDisplayType=RESOURCEDISPLAYTYPE_GENERIC;
    host.dwUsage=RESOURCEUSAGE_CONTAINER;
    host.lpLocalName=NULL;
    host.lpRemoteName=(char*)path.text();
    host.lpComment=NULL;
    host.lpProvider=NULL;
    
    // Open network enumeration
    if(WNetOpenEnum((path[2]?RESOURCE_GLOBALNET:RESOURCE_CONTEXT),RESOURCETYPE_DISK,0,(path[2]?&host:NULL),&hEnum)==NO_ERROR){
      NETRESOURCE resource[16384/sizeof(NETRESOURCE)];
      FXTRACE((100,"Enumerating=%s\n",path.text()));
      while(1){
        nCount=-1;    // Read as many as will fit
        nSize=sizeof(resource);
        if(WNetEnumResource(hEnum,&nCount,resource,&nSize)!=NO_ERROR) break;
        for(i=0; i<nCount; i++){
 
          // Dump what we found
          FXTRACE((100,"dwScope=%s\n",resource[i].dwScope==RESOURCE_CONNECTED?"RESOURCE_CONNECTED":resource[i].dwScope==RESOURCE_GLOBALNET?"RESOURCE_GLOBALNET":resource[i].dwScope==RESOURCE_REMEMBERED?"RESOURCE_REMEMBERED":"?"));
          FXTRACE((100,"dwType=%s\n",resource[i].dwType==RESOURCETYPE_ANY?"RESOURCETYPE_ANY":resource[i].dwType==RESOURCETYPE_DISK?"RESOURCETYPE_DISK":resource[i].dwType==RESOURCETYPE_PRINT?"RESOURCETYPE_PRINT":"?"));
          FXTRACE((100,"dwDisplayType=%s\n",resource[i].dwDisplayType==RESOURCEDISPLAYTYPE_DOMAIN?"RESOURCEDISPLAYTYPE_DOMAIN":resource[i].dwDisplayType==RESOURCEDISPLAYTYPE_SERVER?"RESOURCEDISPLAYTYPE_SERVER":resource[i].dwDisplayType==RESOURCEDISPLAYTYPE_SHARE?"RESOURCEDISPLAYTYPE_SHARE":resource[i].dwDisplayType==RESOURCEDISPLAYTYPE_GENERIC?"RESOURCEDISPLAYTYPE_GENERIC":resource[i].dwDisplayType==6?"RESOURCEDISPLAYTYPE_NETWORK":resource[i].dwDisplayType==7?"RESOURCEDISPLAYTYPE_ROOT":resource[i].dwDisplayType==8?"RESOURCEDISPLAYTYPE_SHAREADMIN":resource[i].dwDisplayType==9?"RESOURCEDISPLAYTYPE_DIRECTORY":resource[i].dwDisplayType==10?"RESOURCEDISPLAYTYPE_TREE":resource[i].dwDisplayType==11?"RESOURCEDISPLAYTYPE_NDSCONTAINER":"?"));
          FXTRACE((100,"dwUsage=%s\n",resource[i].dwUsage==RESOURCEUSAGE_CONNECTABLE?"RESOURCEUSAGE_CONNECTABLE":resource[i].dwUsage==RESOURCEUSAGE_CONTAINER?"RESOURCEUSAGE_CONTAINER":"?"));
          FXTRACE((100,"lpLocalName=%s\n",resource[i].lpLocalName));
          FXTRACE((100,"lpRemoteName=%s\n",resource[i].lpRemoteName));
          FXTRACE((100,"lpComment=%s\n",resource[i].lpComment));
          FXTRACE((100,"lpProvider=%s\n\n",resource[i].lpProvider));
 
          // Grow list
          if(count+1>=size){
            size=size?(size<<1):256;
            newlist=new FXString[size];
            for(j=0; j<count; j++) newlist[j]=list[j];
            delete [] list;
            list=newlist;
            }

          // Add remote name to list
          list[count++]=resource[i].lpRemoteName;
          }
        }
      WNetCloseEnum(hEnum);
      }
    return count;
    }

  // Open directory 
  hFindFile=FindFirstFile((path+PATHSEPSTRING "*").text(),&ffData);
  if(hFindFile==INVALID_HANDLE_VALUE) return 0;

  // Loop over directory entries
  do{
    name=ffData.cFileName;

    // Skip '.' and '..'
    //if(name[0]=='.' && (name[1]==0 || (name[1]=='.' && name[2]==0))) continue;
    // Is it a directory or does it match the pattern?
    if(!fxfilematch(pattern.text(),name,flags)) continue;

    // Grow list
    if(count+1>=size){
      size=size?(size<<1):256;
      newlist=new FXString[size];
      for(i=0; i<count; i++) newlist[i]=list[i];
      delete [] list;
      list=newlist;
      }

    // Add to list
    list[count++]=name;
    }
  while(FindNextFile(hFindFile,&ffData));
  FindClose(hFindFile);
  return count;
  }

#endif


// Convert file time to string as per strftime format
FXString FXFile::time(const FXchar *format,FXTime filetime){
  time_t tmp=(time_t)filetime;
  FXchar buffer[512];
  FXint len;
  len=strftime(buffer,sizeof(buffer),format,localtime(&tmp));
  return FXString(buffer,len);
  }


// Convert file time to string
FXString FXFile::time(FXTime filetime){
  return FXFile::time(TIMEFORMAT,filetime);
  }


// Return current time
FXTime FXFile::now(){
  return (FXTime)::time(NULL);
  }


// Get file info
FXbool FXFile::info(const FXString& file,struct stat& info){
#ifndef WIN32
  return !file.empty() && (::stat(file.text(),&info)==0);
#else
  return !file.empty() && (::stat(file.text(),&info)==0);
#endif
  }


// Get file size
unsigned long FXFile::size(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (unsigned long)status.st_size : 0L;
#else
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? (unsigned long)status.st_size : 0L;
#endif
  }


FXbool FXFile::exists(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0);
#else
  return !file.empty() && (GetFileAttributes(file.text())!=0xFFFFFFFF);
#endif
  }



#ifndef WIN32                 // UNIX

// Enquote filename to make safe for shell
FXString FXFile::enquote(const FXString& file,FXbool forcequotes){
  FXString result;
  register FXint i,c;
  for(i=0; (c=file[i])!='\0'; i++){
    switch(c){
      case '\'':              // Quote needs to be escaped
        result+="\\\'";
        break;
      case '\\':              // Backspace needs to be escaped, of course
        result+="\\\\";
        break;
      case '#':
      case '~':
        if(i) goto noquote;   // Only quote if at begin of filename
      case '!':               // Special in csh
      case '"':
      case '$':               // Variable substitution
      case '&':
      case '(':
      case ')':
      case ';':
      case '<':               // Redirections, pipe
      case '>':
      case '|':
      case '`':               // Command substitution
      case '^':               // Special in sh
      case '*':               // Wildcard characters
      case '?':
      case '[':
      case ']':
      case '\t':              // White space
      case '\n':
      case ' ':
        forcequotes=TRUE;
      default:                // Normal characters just added
noquote:result+=c;
        break;
      }
    }
  if(forcequotes) return "'"+result+"'";
  return result;
  }


// Decode filename to get original again
FXString FXFile::dequote(const FXString& file){
  FXString result;
  register FXint i,c;
  i=0;
  while((c=file[i])!='\0' && isspace((FXuchar)c)) i++;
  if(file[i]=='\''){
    i++;
    while((c=file[i])!='\0' && c!='\''){
      if(c=='\\' && file[i+1]!='\0') c=file[++i];
      result+=c;
      i++;
      }
    }
  else{
    while((c=file[i])!='\0' && !isspace((FXuchar)c)){
      if(c=='\\' && file[i+1]!='\0') c=file[++i];
      result+=c;
      i++;
      }
    }
  return result;
  }



#else                         // WINDOWS

// Enquote filename to make safe for shell
FXString FXFile::enquote(const FXString& file,FXbool forcequotes){
  FXString result;
  register FXint i,c;
  for(i=0; (c=file[i])!='\0'; i++){
    switch(c){
      case '"':               // Quote needs to be escaped
        result+="\\\"";
        break;
      case '\\':              // Backspace needs to be escaped, of course
        result+="\\\\";
        break;
      case '<':               // Redirections
      case '>':
      case '|':
      case '$':
      case ':':
      case '*':               // Wildcards
      case '?':
      case ' ':               // White space
        forcequotes=TRUE;
      default:                // Normal characters just added
        result+=c;
        break;
      }
    }
  if(forcequotes) return "\""+result+"\"";
  return result;
  }


// Decode filename to get original again
FXString FXFile::dequote(const FXString& file){
  FXString result;
  register FXint i,c;
  i=0;
  while((c=file[i])!='\0' && isspace((FXuchar)c)) i++;
  if(file[i]=='"'){
    i++;
    while((c=file[i])!='\0' && c!='"'){
      if(c=='\\' && file[i+1]!='\0') c=file[++i];
      result+=c;
      i++;
      }
    }
  else{
    while((c=file[i])!='\0' && !isspace((FXuchar)c)){
      if(c=='\\' && file[i+1]!='\0') c=file[++i];
      result+=c;
      i++;
      }
    }
  return result;
  }

#endif


// Match filenames using *, ?, [^a-z], and so on
FXbool FXFile::match(const FXString& pattern,const FXString& file,FXuint flags){
  return fxfilematch(pattern.text(),file.text(),flags);
  }


// Return true if files are identical
FXbool FXFile::identical(const FXString& file1,const FXString& file2){
  if(file1!=file2){
#ifndef WIN32
    struct stat stat1,stat2;
    return !::lstat(file1.text(),&stat1) && !::lstat(file2.text(),&stat2) && stat1.st_ino==stat2.st_ino;
#else
    FXbool same=FALSE;
    HANDLE hFile1;
    HANDLE hFile2;
    hFile1=CreateFile(file1.text(),GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
    if(hFile1!=INVALID_HANDLE_VALUE){
      hFile2=CreateFile(file2.text(),GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
      if(hFile2!=INVALID_HANDLE_VALUE){
        BY_HANDLE_FILE_INFORMATION info1;
        BY_HANDLE_FILE_INFORMATION info2;
        if(GetFileInformationByHandle(hFile1,&info1) && GetFileInformationByHandle(hFile2,&info2)){
          same=(info1.nFileIndexLow==info2.nFileIndexLow && info1.nFileIndexHigh==info2.nFileIndexHigh && info1.dwVolumeSerialNumber==info2.dwVolumeSerialNumber);
          }
        CloseHandle(hFile2);
        }
      CloseHandle(hFile1);
      }
    return same;
#endif
    }
  return TRUE;
  }


// Return file mode flags
FXuint FXFile::mode(const FXString& file){
#ifndef WIN32
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? status.st_mode : 0;
#else
  struct stat status;
  return !file.empty() && (::stat(file.text(),&status)==0) ? status.st_mode : 0;
#endif
  }


// Change the mode flags for this file
FXbool FXFile::mode(const FXString& file,FXuint mode){
#ifndef WIN32
  return !file.empty() && chmod(file.text(),mode)==0;
#else
  return FALSE; // Unimplemented yet
#endif
  }


// Create new directory
FXbool FXFile::createDirectory(const FXString& path,FXuint mode){
#ifndef WIN32
  return mkdir(path.text(),mode)==0;
#else
  return CreateDirectory(path.text(),NULL)!=0;
#endif
  }


// Create new (empty) file
FXbool FXFile::createFile(const FXString& file,FXuint mode){
#ifndef WIN32
  FXint fd=open(file.text(),O_CREAT|O_WRONLY|O_TRUNC,mode);
  if(fd>=0){ close(fd); return TRUE; }
  return FALSE;
#else
  HANDLE hFile=CreateFile(file.text(),GENERIC_WRITE,FILE_SHARE_READ,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
  if(hFile!=INVALID_HANDLE_VALUE){ CloseHandle(hFile); return TRUE; }
  return FALSE;
#endif
  }


// Hack code below for testing if volume is mounted

// #if defined (HKS_NT)
//
// static int check_nfs (const char* name)
// {
// char drive[8];
//
// char* cp = strchr (name, ':');
// if (cp)
// {
// strncpy (drive, name, cp - name);
// drive[cp - name] = '\0';
// }
// else
// {
// drive[0] = 'A' + _getdrive() - 1;
// drive[1] = '\0';
// }
//
// strcat (drive, ":\\");
//
// return GetDriveType(drive) == DRIVE_REMOTE;
// }
//
// #elif defined(LINUX)
//
// static int check_nfs (int fd)
// {
// struct statfs statbuf;
// if (fstatfs(fd,&statbuf) < 0)
// {
// RFM_RAISE_SYSTEM_ERROR("statfs");
// return 0;
// }
// if (statbuf.f_type == NFS_SUPER_MAGIC)
// return 1;
// else
// return 0;
// }
//
// #else
//
// static int check_nfs (int fd)
// {
//
// struct statvfs statbuf;
//
// if (fstatvfs (fd, &statbuf) < 0)
// {
// RFM_RAISE_SYSTEM_ERROR ("fstatvfs");
// }
// return strncmp (statbuf.f_basetype, "nfs", 3) == 0 || strncmp
// (statbuf.f_basetype, "NFS", 3) == 0;
// }
// #endif






#ifndef WIN32


// Read bytes
static long fullread(int fd,unsigned char *ptr,long len){
  long nread;
#ifdef EINTR
  do{nread=read(fd,ptr,len);}while(nread<0 && errno==EINTR);
#else
  nread=read(fd,ptr,len);
#endif
  return nread;
  }


// Write bytes
static long fullwrite(int fd,const unsigned char *ptr,long len){
  long nwritten,ntotalwritten=0;
  while(len>0){
    nwritten=write(fd,ptr,len);
    if(nwritten<0){
#ifdef EINTR
      if(errno==EINTR) continue;
#endif
      return -1;
      }
    ntotalwritten+=nwritten;
    ptr+=nwritten;
    len-=nwritten;
    }
  return ntotalwritten;
  }


// Concatenate srcfile1 and srcfile2 to a dstfile
FXbool FXFile::concatenate(const FXString& srcfile1,const FXString& srcfile2,const FXString& dstfile,FXbool overwrite){
  unsigned char buffer[4096];
  struct stat status;
  int src1,src2,dst;
  long nread,nwritten;
  FXbool ok=FALSE;
  if(srcfile1==dstfile || srcfile2==dstfile) return FALSE;
  if(::lstat(dstfile.text(),&status)==0){
    if(!overwrite) return FALSE;
    }
  dst=open(dstfile.text(),O_CREAT|O_WRONLY|O_TRUNC,0777);
  if(0<=dst){
    src1=open(srcfile1.text(),O_RDONLY);
    if(0<=src1){
      src2=open(srcfile2.text(),O_RDONLY);
      if(0<=src2){
        while(1){
          nread=fullread(src1,buffer,sizeof(buffer));
          if(nread<0) goto err;
          if(nread==0) break;
          nwritten=fullwrite(dst,buffer,nread);
          if(nwritten<0) goto err;
          }
        while(1){
          nread=fullread(src2,buffer,sizeof(buffer));
          if(nread<0) goto err;
          if(nread==0) break;
          nwritten=fullwrite(dst,buffer,nread);
          if(nwritten<0) goto err;
          }
        ok=TRUE;
err:    close(src2);
        }
      close(src1);
      }
    close(dst);
    }
  return ok;
  }


// Copy ordinary file
static FXbool copyfile(const FXString& oldfile,const FXString& newfile){
  unsigned char buffer[4096];
  struct stat status;
  long nread,nwritten;
  int src,dst;
  FXbool ok=FALSE;
  if((src=open(oldfile.text(),O_RDONLY))>=0){
    if(::stat(oldfile.text(),&status)==0){
      if((dst=open(newfile.text(),O_WRONLY|O_CREAT|O_TRUNC,status.st_mode))>=0){
        while(1){
          nread=fullread(src,buffer,sizeof(buffer));
          if(nread<0) goto err;
          if(nread==0) break;
          nwritten=fullwrite(dst,buffer,nread);
          if(nwritten<0) goto err;
          }
        ok=TRUE;
err:    close(dst);
        }
      }
    close(src);
    }
  return ok;
  }


// To search visited inodes
struct inodelist {
  ino_t st_ino;
  inodelist *next;
  };


// Forward declararion
static FXbool copyrec(const FXString& oldfile,const FXString& newfile,FXbool overwrite,inodelist* inodes);


// Copy directory
static FXbool copydir(const FXString& oldfile,const FXString& newfile,FXbool overwrite,struct stat& parentstatus,inodelist* inodes){
  DIR *dirp;
  struct dirent *dp;
  struct stat status;
  inodelist *in,inode;
  FXString oldchild,newchild;

  // See if visited this inode already
  for(in=inodes; in; in=in->next){
    if(in->st_ino==parentstatus.st_ino) return TRUE;
    }

  // Try make directory, if none exists yet
  if(mkdir(newfile.text(),parentstatus.st_mode|S_IWUSR)!=0 && errno!=EEXIST) return FALSE;

  // Can we stat it
  if(::lstat(newfile.text(),&status)!=0 || !S_ISDIR(status.st_mode)) return FALSE;

  // Try open directory to copy
  dirp=opendir(oldfile.text());
  if(!dirp) return FALSE;

  // Add this to the list
  inode.st_ino=status.st_ino;
  inode.next=inodes;

  // Copy stuff
  while((dp=readdir(dirp))!=NULL){
    if(dp->d_name[0]!='.' || (dp->d_name[1]!='\0' && (dp->d_name[1]!='.' || dp->d_name[2]!='\0'))){
      oldchild=oldfile;
      if(!ISPATHSEP(oldchild[oldchild.length()-1])) oldchild.append(PATHSEP);
      oldchild.append(dp->d_name);
      newchild=newfile;
      if(!ISPATHSEP(newchild[newchild.length()-1])) newchild.append(PATHSEP);
      newchild.append(dp->d_name);
      if(!copyrec(oldchild,newchild,overwrite,&inode)){
        closedir(dirp);
        return FALSE;
        }
      }
    }

  // Close directory
  closedir(dirp);

  // Success
  return TRUE;
  }




// Recursive copy
static FXbool copyrec(const FXString& oldfile,const FXString& newfile,FXbool overwrite,inodelist* inodes){
  struct stat status1,status2;

  // Old file or directory does not exist
  if(::lstat(oldfile.text(),&status1)!=0) return FALSE;

  // If target is not a directory, remove it if allowed
  if(::lstat(newfile.text(),&status2)==0){
    if(!S_ISDIR(status2.st_mode)){
      if(!overwrite) return FALSE;
      FXTRACE((100,"unlink(%s)\n",newfile.text()));
      if(::unlink(newfile.text())!=0) return FALSE;
      }
    }

  // Source is direcotory: copy recursively
  if(S_ISDIR(status1.st_mode)){
    return copydir(oldfile,newfile,overwrite,status1,inodes);
    }

  // Source is regular file: copy block by block
  if(S_ISREG(status1.st_mode)){
    FXTRACE((100,"copyfile(%s,%s)\n",oldfile.text(),newfile.text()));
    return copyfile(oldfile,newfile);
    }

  // Source is fifo: make a new one
  if(S_ISFIFO(status1.st_mode)){
    FXTRACE((100,"mkfifo(%s)\n",newfile.text()));
    return ::mkfifo(newfile.text(),status1.st_mode);
    }

  // Source is device: make a new one
  if(S_ISBLK(status1.st_mode) || S_ISCHR(status1.st_mode) || S_ISSOCK(status1.st_mode)){
    FXTRACE((100,"mknod(%s)\n",newfile.text()));
    return ::mknod(newfile.text(),status1.st_mode,status1.st_rdev)==0;
    }

  // Source is symbolic link: make a new one
  if(S_ISLNK(status1.st_mode)){
    FXString lnkfile=FXFile::symlink(oldfile);
    FXTRACE((100,"symlink(%s,%s)\n",lnkfile.text(),newfile.text()));
    return ::symlink(lnkfile.text(),newfile.text())==0;
    }

  // This shouldn't happen
  return FALSE;
  }


#else


// Concatenate srcfile1 and srcfile2 to a dstfile
FXbool FXFile::concatenate(const FXString& srcfile1,const FXString& srcfile2,const FXString& dstfile,FXbool overwrite){
  unsigned char buffer[4096];
  HANDLE src1,src2,dst;
  DWORD nread,nwritten;
  FXbool ok=FALSE;
  if(srcfile1==dstfile || srcfile2==dstfile) return FALSE;
  if(GetFileAttributes(dstfile.text())!=0xFFFFFFFF){
    if(!overwrite) return FALSE;
    }
  dst=CreateFile(dstfile.text(),GENERIC_WRITE,FILE_SHARE_READ,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
  if(dst!=INVALID_HANDLE_VALUE){
    src1=CreateFile(srcfile1.text(),GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
    if(src1!=INVALID_HANDLE_VALUE){
      src2=CreateFile(srcfile2.text(),GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
      if(src2!=INVALID_HANDLE_VALUE){
        while(1){
          if(!ReadFile(src1,buffer,sizeof(buffer),&nread,NULL)) goto err;
          if(nread==0) break;
          if(!WriteFile(dst,buffer,nread,&nwritten,NULL)) goto err;
          }
        while(1){
          if(!ReadFile(src1,buffer,sizeof(buffer),&nread,NULL)) goto err;
          if(nread==0) break;
          if(!WriteFile(dst,buffer,nread,&nwritten,NULL)) goto err;
          }
        ok=TRUE;
err:    CloseHandle(src2);
        }
      CloseHandle(src1);
      }
    CloseHandle(dst);
    }
  return ok;
  }


// Forward declararion
static FXbool copyrec(const FXString& oldfile,const FXString& newfile,FXbool overwrite);


// Copy directory
static FXbool copydir(const FXString& oldfile,const FXString& newfile,FXbool overwrite){
  FXString oldchild,newchild;
  DWORD atts;
  WIN32_FIND_DATA ffData;
  HANDLE hFindFile;

  // Try make directory, if none exists yet
  if(CreateDirectory(newfile.text(),NULL)==0 && GetLastError()!=ERROR_FILE_EXISTS) return FALSE;

  // Can we stat it
  if((atts=GetFileAttributes(newfile.text()))==0xffffffff || !(atts&FILE_ATTRIBUTE_DIRECTORY)) return FALSE;

  // Try open directory to copy
  hFindFile=FindFirstFile((oldfile+PATHSEPSTRING+"*").text(),&ffData);
  if(hFindFile==INVALID_HANDLE_VALUE) return FALSE;

  // Copy stuff
  do{
    if(ffData.cFileName[0]!='.' && (ffData.cFileName[1]!='\0' && (ffData.cFileName[1]!='.' || ffData.cFileName[2]!='\0'))){
      oldchild=oldfile;
      if(!ISPATHSEP(oldchild[oldchild.length()-1])) oldchild.append(PATHSEP);
      oldchild.append(ffData.cFileName);
      newchild=newfile;
      if(!ISPATHSEP(newchild[newchild.length()-1])) newchild.append(PATHSEP);
      newchild.append(ffData.cFileName);
      if(!copyrec(oldchild,newchild,overwrite)){
        FindClose(hFindFile);
        return FALSE;
        }
      }
    }
  while(FindNextFile(hFindFile,&ffData));

  // Close directory
  FindClose(hFindFile);

  // Success
  return TRUE;
  }


// Recursive copy
static FXbool copyrec(const FXString& oldfile,const FXString& newfile,FXbool overwrite){
  DWORD atts1,atts2;

  // Old file or directory does not exist
  if((atts1=GetFileAttributes(oldfile.text()))==0xffffffff) return FALSE;

  // If target is not a directory, remove it if allowed
  if((atts2=GetFileAttributes(newfile.text()))!=0xffffffff){
    if(!(atts2&FILE_ATTRIBUTE_DIRECTORY)){
      if(!overwrite) return FALSE;
      FXTRACE((100,"DeleteFile(%s)\n",newfile.text()));
      if(DeleteFile(newfile.text())!=0) return FALSE;
      }
    }

  // Source is direcotory: copy recursively
  if(atts1&FILE_ATTRIBUTE_DIRECTORY){
    return copydir(oldfile,newfile,overwrite);
    }

  // Source is regular file: copy block by block
  if(!(atts1&FILE_ATTRIBUTE_DIRECTORY)){
    FXTRACE((100,"CopyFile(%s,%s)\n",oldfile.text(),newfile.text()));
    return CopyFile(oldfile.text(),newfile.text(),!overwrite);
    }

  // This shouldn't happen
  return FALSE;
  }


#endif


// Copy file
FXbool FXFile::copy(const FXString& oldfile,const FXString& newfile,FXbool overwrite){
  if(newfile!=oldfile){
#ifndef WIN32
    return copyrec(oldfile,newfile,overwrite,NULL);
#else
    return copyrec(oldfile,newfile,overwrite);      // No symlinks, so no need to check if directories are visited already
#endif
    }
  return FALSE;
  }


// Remove file or directory
FXbool FXFile::remove(const FXString& file){
#ifndef WIN32
  struct stat status;
  if(::lstat(file.text(),&status)==0){
    if(S_ISDIR(status.st_mode)){
      DIR *dirp=::opendir(file.text());
      if(dirp){
        struct dirent *dp;
        FXString child;
        while((dp=::readdir(dirp))!=NULL){
          if(dp->d_name[0]!='.' || (dp->d_name[1]!='\0' && (dp->d_name[1]!='.' || dp->d_name[2]!='\0'))){
            child=file;
            if(!ISPATHSEP(child[child.length()-1])) child.append(PATHSEP);
            child.append(dp->d_name);
            if(!FXFile::remove(child)){
              ::closedir(dirp);
              return FALSE;
              }
            }
          }
        ::closedir(dirp);
        }
      FXTRACE((100,"rmdir(%s)\n",file.text()));
      return ::rmdir(file.text())==0;
      }
    else{
      FXTRACE((100,"unlink(%s)\n",file.text()));
      return ::unlink(file.text())==0;
      }
    }
  return FALSE;
#else
  DWORD atts;
  if((atts=GetFileAttributes(file.text()))!=0xffffffff){
    if(atts&FILE_ATTRIBUTE_DIRECTORY){
      WIN32_FIND_DATA ffData;
      HANDLE hFindFile;
      hFindFile=FindFirstFile((file+PATHSEPSTRING+"*").text(),&ffData); // FIXME we may want to formalize the "walk over directory" in a few API's here also...
      if(hFindFile!=INVALID_HANDLE_VALUE){
        FXString child;
        do{
          if(ffData.cFileName[0]!='.' && (ffData.cFileName[1]!='\0' && (ffData.cFileName[1]!='.' || ffData.cFileName[2]!='\0'))){
            child=file;
            if(!ISPATHSEP(child[child.length()-1])) child.append(PATHSEP);
            child.append(ffData.cFileName);
            if(!FXFile::remove(child)){
              FindClose(hFindFile);
              return FALSE;
              }
            }
          }
        while(FindNextFile(hFindFile,&ffData));
        FindClose(hFindFile);
        }
      FXTRACE((100,"RemoveDirectory(%s)\n",file.text()));
      return RemoveDirectory(file.text())!=0;
      }
    else{
      FXTRACE((100,"DeleteFile(%s)\n",file.text()));
      return DeleteFile(file.text())!=0;
      }
    }
  return FALSE;
#endif
  }


// Rename or move file, or copy and delete old if different file systems
FXbool FXFile::move(const FXString& oldfile,const FXString& newfile,FXbool overwrite){
  if(newfile!=oldfile){
#ifndef WIN32
    if(!FXFile::exists(oldfile)) return FALSE;
    if(FXFile::exists(newfile)){
      if(!overwrite) return FALSE;
      if(!FXFile::remove(newfile)) return FALSE;
      }
    FXTRACE((100,"rename(%s,%s)\n",oldfile.text(),newfile.text()));
    if(::rename(oldfile.text(),newfile.text())==0) return TRUE;
    if(errno!=EXDEV) return FALSE;
    if(FXFile::copy(oldfile,newfile)){
      return FXFile::remove(oldfile);
      }
#else
    if(!FXFile::exists(oldfile)) return FALSE;
    if(FXFile::exists(newfile)){
      if(!overwrite) return FALSE;
      if(!FXFile::remove(newfile)) return FALSE;
      }
    FXTRACE((100,"MoveFile(%s,%s)\n",oldfile.text(),newfile.text()));
    if(::MoveFile(oldfile.text(),newfile.text())!=0) return TRUE;
    if(GetLastError()!=ERROR_NOT_SAME_DEVICE) return FALSE;
    if(FXFile::copy(oldfile,newfile)){
      return FXFile::remove(oldfile);
      }
#endif
    }
  return FALSE;
  }


// Link file
FXbool FXFile::link(const FXString& oldfile,const FXString& newfile,FXbool overwrite){
  if(newfile!=oldfile){
#ifndef WIN32
    if(!FXFile::exists(oldfile)) return FALSE;
    if(FXFile::exists(newfile)){
      if(!overwrite) return FALSE;
      if(!FXFile::remove(newfile)) return FALSE;
      }
    FXTRACE((100,"link(%s,%s)\n",oldfile.text(),newfile.text()));
    return ::link(oldfile.text(),newfile.text())==0;
#else
#if (_WIN32_WINNT >=0x0500)   // WinNT, Win2K, and later
    if(!FXFile::exists(oldfile)) return FALSE;
    if(FXFile::exists(newfile)){
      if(!overwrite) return FALSE;
      if(!FXFile::remove(newfile)) return FALSE;
      }
    FXTRACE((100,"CreateHardLink(%s,%s)\n",oldfile.text(),newfile.text()));
    return ::CreateHardLink(newfile.text(),oldfile.text(),NULL)!=0;
#else
    return FALSE;
#endif
#endif
    }
  return FALSE;
  }


// Symbolic Link file
FXbool FXFile::symlink(const FXString& oldfile,const FXString& newfile,FXbool overwrite){
#ifndef WIN32
  if(newfile!=oldfile){
    if(!FXFile::exists(oldfile)) return FALSE;
    if(FXFile::exists(newfile)){
      if(!overwrite) return FALSE;
      if(!FXFile::remove(newfile)) return FALSE;
      }
    FXTRACE((100,"symlink(%s,%s)\n",oldfile.text(),newfile.text()));
    return ::symlink(oldfile.text(),newfile.text())==0;
    }
#endif
  return FALSE;
  }


// Read symbolic link
FXString FXFile::symlink(const FXString& file){
#ifndef WIN32
  FXchar lnk[MAXPATHLEN+1];
  FXint len=::readlink(file.text(),lnk,MAXPATHLEN);
  if(0<=len) return FXString(lnk,len);
#endif
  return FXString::null;
  }

