/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <vtimezonecomponent.h>

extern Toolbox tb;

int VTimeZoneComponent::TZC_UID=0;

#define IDSTRING "VTimeZoneComponent: "

//#define DEBUG_VTC
#define DEBUG_VTC_SET


VTimeZoneComponent::VTimeZoneComponent(MetaObject *object)
{
	// init all the pointers NULL!
	rdate=NULL;
	rrule=NULL;
	mObj=NULL;
	parse(object);
#ifdef DEBUG_VTC
	if(errorCode)
	{
		printf(IDSTRING"This object is faulty and should be deleted\n");
	}
#endif
}
VTimeZoneComponent::~VTimeZoneComponent()
{
	// deleting the rdate thing.
	if(rdate) delete rdate;
	rdate=NULL;
	// deleting the rrule thing.
	if(rrule) delete rrule;
	rrule=NULL;
}

QString VTimeZoneComponent::getErrorString()
{
	QString result="";
	switch(errorCode)
	{
		case VTC_OK:
			result="OK. No error.";
			break;
		case VTC_ERROR_NULL:
			result="Error: Null object passed as parameter.";
			break;
		case VTC_ERROR_DTSTART:
			result="Error: Invalid number of \'DTSTART\' occurences.";
			break;
		case VTC_ERROR_TZOFFSETFROM:
			result="Error: Invalid number of \'TZOFFSETFROM\' occurences.";
			break;
		case VTC_ERROR_TZOFFSETTO:
			result="Error: Invalid number of \'TZOFFSETTO\' occurences.";
			break;
		case VTC_ERROR_NOLOCAL:
			result="Error: \'DTSTART\' date is not local.";
			break;
	}
	/*
	 * The errors are more important than the warnings
	 */
	if(!result.isEmpty()) return result;
	switch(warningCode)
	{
		case VTC_WARNING_TZNAME:
			result="Warning: No name for the component.";
			break;
		case VTC_WARNING_MULTIPLE_RDATE:
			result="Warning: Multiple RDATE occurences. Getting the first one.";
			break;
	}
	return result;
}
int VTimeZoneComponent::getErrorCode()
{
	return errorCode;
}
void VTimeZoneComponent::parse(MetaObject *object)
{
	mObj=object;
	errorCode=VTC_OK;
	
	if(mObj==NULL)
	{
		errorCode=VTC_ERROR_NULL;
		return;
	}
	// setting up the object type (daylight or standard)
	QString objectType=object->getObjectName();
	if(objectType.simplifyWhiteSpace().upper()=="DAYLIGHT")
		type=VTimeZoneComponent::Daylight;
	else
		type=VTimeZoneComponent::Standard;
	/*
	 * getting the DTSTART
	 */
	Toolbox::TimeZone tz;
	QValueList<int> matches;
	matches=object->query("DTSTART");
	if(matches.count()!=1)
	{
		errorCode=VTC_ERROR_DTSTART;
		return;
	}
	dtStart=tb.getDateTime(object->getLine(matches[0]).getContent(), tz);
	if(tz!=Toolbox::UnknownTimeZone)
	{
		errorCode=VTC_ERROR_NOLOCAL;
		return;
	}
#ifdef DEBUG_VTC
	printf(IDSTRING"DTSTART = %s\n", (const char *)dtStart.toString());
#endif
	/*
	 * getting the TZOFFSETFROM
	 */
	matches=object->query("TZOFFSETFROM");
	if(matches.count()!=1)
	{
		errorCode=VTC_ERROR_TZOFFSETFROM;
		return;
	}
	offsetFrom=tb.getTimeOffset(object->getLine(matches[0]).getContent());
#ifdef DEBUG_VTC
	printf(IDSTRING"\'TZOFFSETFROM\': Length in seconds: %d\n",offsetFrom);
#endif	 
	/*
	 * getting the TZOFFSETTO
	 */
	matches=object->query("TZOFFSETTO");
	if(matches.count()!=1)
	{
		errorCode=VTC_ERROR_TZOFFSETTO;
		return;
	}
	offsetTo=tb.getTimeOffset(object->getLine(matches[0]).getContent());
#ifdef DEBUG_VTC
	printf(IDSTRING"\'TZOFFSETTO\': Length in seconds: %d\n",offsetTo);
#endif	 
	/*
	 * getting the TZNAME(s)
	 */
	matches=object->query("TZNAME");
	if(matches.count()==0)
	{
		warningCode=VTC_WARNING_TZNAME;
	}
	for(int i=0;i<matches.count();i++)
		names.append(object->getLine(matches[i]).getContent());
	/*
	 *  getting the Comment
	 */
	matches=object->query("TZNAME");
	if(matches.count())
		comment=object->getLine(matches[0]).getContent();
	/*
	 *	getting the RDATE
	 */
	if(matches.count()!=1)
	{
		warningCode=VTC_WARNING_MULTIPLE_RDATE;
	}
	matches=object->query("RDATE");
	if(matches.count())
		rdate=new RDate(object->getLine(matches[0]).getContent());
#ifdef DEBUG_VTC
	else
		printf(IDSTRING"No RDATE field encountered\n");
#endif	 
	matches=object->query("RRULE");
	if(matches.count())
		rrule=new RRule(object->getLine(matches[0]).getContent());
#ifdef DEBUG_VTC
	else
		printf(IDSTRING"No RRULE field encountered\n");
#endif	 
}

void VTimeZoneComponent::setCurrentStart(int currentYear)
{
	setIsValid(false);
	lastRequired=false;
	QDateTime current(QDate(currentYear,1,1), QTime(0,0,0));
#ifdef DEBUG_VTC
	printf(IDSTRING"Current date used for parsing: %s\n",
		(const char *)current.toString());
#endif
	if(abs(current.date().year()-dtStart.date().year())==0)
	{
#ifdef DEBUG_VTC
		printf(IDSTRING"Getting the date from dtstart\n");
#endif
		currentStart=dtStart; // should I do .addSecs(offsetFrom) ?;
		setIsValid(true);
		return;
	}
	if((current.date().year()!=dtStart.date().year()) &&
			!rdate && !rrule)
	{
#ifdef DEBUG_VTC
		printf(IDSTRING"Still getting the date from dtstart (+1)\n");
#endif
		lastRequired=true;
		currentStart=dtStart; // should I do .addSecs(offsetFrom) ?;
		setIsValid(true);
		return;
	}
	/*
	 *	Checking the RDATE
	 */
	if(rdate)
	{
		Period p;
		p=rdate->getNextOccurence(current);
#ifdef DEBUG_VTC
		printf(IDSTRING"Start date: %s\n", (const char *)p.start.toString());
		printf(IDSTRING"Length    : %d\n", p.length);
		printf(IDSTRING"finished  : %s\n", (p.isFinished)?"true":"false");
#endif
		setIsValid(!p.isFinished);
		currentStart=p.start;	// should I do .addSecs(offsetFrom) ?;
		return;
	}
	/*
	 *	Checking the RRULE
	 */
	if(rrule)
	{
		QValueList<QDateTime> all;
		QValueList<QDateTime> matches;
		all=rrule->getGlobalOccurences(dtStart, current, true);
		// get all the dates in the current year
#ifdef DEBUG_VTC_SET
		printf(IDSTRING"Matches: [.");
#endif
		for(int i=0;i<all.count();i++)
			if(all[i].date().year()==currentYear)
			{
				matches.append(all[i]);
#ifdef DEBUG_VTC_SET
				if(!(matches.count()%10)) printf(".");
#endif
			}
#ifdef DEBUG_VTC_SET
		printf("]\n");
#endif
		if(matches.count()<=0)
		{
			printf(IDSTRING"This component is expired.\n");
			setIsValid(false);
			return;
		}
		currentStart=matches[0];
		for(int i=01;i<matches.count();i++)
			if(currentStart.date()<matches[i].date()) currentStart=matches[i];
		setIsValid(true);
	}
}

void VTimeZoneComponent::setIsValid(bool validity)
{
	validDate=validity;
}

TZData VTimeZoneComponent::getCurrentStart()
{
	TZData data;
	data.start=currentStart;
	data.offsetGMT=offsetFrom;
	return data;
}

QString VTimeZoneComponent::generateUID()
{
	TZC_UID++;
	QString result;
	result.sprintf("%08d",TZC_UID);
	return result;
}





