/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * $Id: sortingwidget.cpp,v 1.6 2001/05/19 10:53:18 alex Exp $
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <tilewidget.h>
#include <sortingwidget.h>
#include <clipboard.h>
#include <pixmaps.h>
#include <mailer.h>
#include <datahandlerimpl.h>

#include <qapplication.h>
#include <qclipboard.h>
#include <qcursor.h>
#include <qtimer.h>

#define BARCOLOR QColor(0xd6,0xd2,0xc7)

SortingWidget::SortingWidget(QWidget *parent, const char *name, WFlags f)
		:QScrollView(parent, name, f/* | WRepaintNoErase*/), DataHandlerImpl(),
m_isDragging(false)
{
  childViews.setAutoDelete(false);

  mainPopup=0;
  sortPopup=0;
  childPopup=0;
	currentKey=0;
	
  viewport()->setMouseTracking(true);
    	
  viewport()->setBackgroundColor(QColor("white"));
  xSpacing=ySpacing=childW=childH=0;
    	
  setVScrollBarMode(QScrollView::Auto);
  setHScrollBarMode(QScrollView::Auto);
  setResizePolicy(QScrollView::Manual);

  isDoResize = false;
  canDoResizeHere = false;
  fixedHeight=true;
  wResize=false;
  allwaysPopup=true;
  enableDrags=false;
  showSort=true;

  setAcceptDrops(false);
    	
  unclippedPainter=0;
	setCanDrawLines(false);
}

SortingWidget::~SortingWidget()
{
}

void SortingWidget::setXSpacing(int xsp)
{
	oldColumnWidth=currentColumnWidth = xSpacing + childW;
	xSpacing=xsp;
	setMinimumWidth(childW+2*(xSpacing+lineWidth()));
}

void SortingWidget::setYSpacing(int ysp)
{
	ySpacing=ysp;
	setMinimumHeight(childH+2*(ySpacing+lineWidth()));
}

void SortingWidget::setChildHeight(int h)
{
	childH=h;
	setMinimumHeight(childH+2*(ySpacing+lineWidth()));
}

void SortingWidget::setChildWidth(int w)
{
	childW=w;
	oldColumnWidth=currentColumnWidth=xSpacing + childW;
	setMinimumWidth(childW+2*(xSpacing+lineWidth()));
}

void SortingWidget::setChildFixedHeight(bool fixed_height)
{
  fixedHeight=fixed_height;
}

bool SortingWidget::childFixedHeight()
{
  return fixedHeight;
}

void SortingWidget::getChildDimensions(int &w, int &h)
{
	w=childW;
	h=childH;
}

void SortingWidget::registerWidget(DataWidget *)
{
}

void SortingWidget::addChild(QWidget *child)
{
	// debug
	// printf("sortingwidget: [addChild]\n");
	
	// debug
	// printf("sortingwidget: [addChild] qwidget: %p\n", child);
	// printf("sortingwidget: [addChild] dataWidget: %p\n", dynamic_cast<DataWidget *>(child));
	// printf("sortingwidget: [addChild] tileWidget: %p\n", dynamic_cast<TileWidget *>(child));
	
	if(childViews.findRef(dynamic_cast<DataWidget *>(child))!=-1) return;

	DataHandler::registerWidget(dynamic_cast<DataWidget *>(child));
	
	// debug
	// printf("sortingwidget: [addChild] registered\n");
	
	QScrollView::addChild(child);
	
	// debug
	// printf("sortingwidget: [addChild] added to the scroll view\n");
	
	child->show();
	
	if(childViews.count()==0 && currentKey==-1) 
		querySorting();
	
	// debug
	// printf("sortingwidget: [addChild] doing a screen update...\n");
	
	screenUpdate();
}

void SortingWidget::unregisterWidget(DataWidget *)
{
}

void SortingWidget::removeChild(QWidget *child)
{
	// debug
	// printf("sortingwidget: [removeChild] \n");

	int index=childViews.findRef(dynamic_cast<DataWidget *>(child));
	
	for (unsigned int i = 0; i < childViews.count() && index==-1; i++)
		if (dynamic_cast<TileWidget *>(childViews.at(i))==0)
			index=i;
	
	// debug
	// printf("sortingwidget: [removeChild] child index: %d\n", index);
	
	// debug
	// printf("sortingwidget: [removeChild] child count: %d\n", childViews.count());
	
  if(child && index>-1)
  {
		childViews.remove(index);
		
		// debug
		// printf("sortingwidget: [removeChild] child count after unregister: %d\n", childViews.count());
	
    QScrollView::removeChild(child);
		
    screenUpdate();
  }
}

void SortingWidget::querySorting()
{
	if(childViews.isEmpty())
	  return;
	
	int knum=dynamic_cast<TileWidget *>(childViews.at(0))->keyCount();
	
	if(knum==0)
	{
		currentKey=-1;
		return;
	}
	
  currentKey=0;
	
	if(knum==1)
	{
		return;
	}
	
	sortPopup=new QPopupMenu();
	for(int i=0;i<knum;i++)
		sortPopup->insertItem(dynamic_cast<TileWidget *>(childViews.at(0))->keyName(i), i+1);
  
  if(mainPopup && showSort)
  {
    mainPopup->insertSeparator();
    mainPopup->insertItem(QIconSet(QPixmap(sort_xpm)), "Sort by ...", sortPopup);
    connect(sortPopup, SIGNAL(activated(int)), this, SLOT(sort(int)));
  }
}

TileWidget* SortingWidget::childAt(QPoint p)
{
  for(unsigned int i=0;i<childViews.count();i++)
  {
		TileWidget *tileWidget=dynamic_cast<TileWidget *>(childViews.at(i));
    if(tileWidget && tileWidget->geometry().contains(p))
      return tileWidget;
  }
	
  return 0;
}

void SortingWidget::sort(int number)
{
	// debug
	// printf("sortingwidget: [sort]\n");
	
  if(number<1)
    return;
	
  currentKey=number-1;
  QList<DataWidget> sList;
  int index;
  DataWidget *cChild;

	// debug
	// printf("sortingwidget: [sort] displaying keys...\n");
	
	// debug
#if 0
	for(unsigned int i = 0; i < childViews.count(); i++)
	{
		// debug
		printf("sortingwidget: [sort] getting child at pos %d\n", i);

		DataWidget *dataWidget=childViews.at(i);

		// debug
		printf("sortingwidget: [sort] dataWidget: %p\n", dataWidget);		

		TileWidget *tileWidget=dynamic_cast<TileWidget *>(dataWidget);
		printf("\ttileWidget: %p\n", tileWidget);
		printf("\tkey[%d]: %s\n", currentKey, (const char *)tileWidget->key(currentKey));
	}
#endif
		
  while(childViews.isEmpty()==false)
  {
    cChild=childViews.at(0);
    index=0;
    for(unsigned int i=1;i<childViews.count();i++)
    {
			if(((dynamic_cast<TileWidget *>(childViews.at(i))))->key(currentKey).lower()
					<(const char *)(dynamic_cast<TileWidget *>(cChild))->key(currentKey).lower())
      {
					cChild=childViews.at(i);
          index=i;
      }
    }
		
		// printf("sortingwidget: sort key: %s\n", (const char *)((TileWidget *)cChild)->key(currentKey));
		
    sList.append(cChild);
    childViews.removeRef(cChild);
	}
	
	childViews=sList;
  makePosChain();
}

void SortingWidget::screenUpdate()
{
	// debug
	// printf("sortingwidget: [screenUpdate] sorting...\n");
	
  sort(currentKey+1);
	
	// debug
	// printf("sortingwidget: [screenUpdate] repainting...\n");
	
  repaint();
}

void SortingWidget::triggerUpdate()
{
	QTimer::singleShot(0, this, SLOT(screenUpdate()));
}

void SortingWidget::getIndexPosition(int index, int &x, int &y)
{
  x=dynamic_cast<TileWidget *>(childViews.at(index))->geometry().x();
  y=dynamic_cast<TileWidget *>(childViews.at(index))->geometry().y();
}

bool SortingWidget::eventFilter(QObject *o, QEvent *e)
{
	if(e->type()==QEvent::ChildRemoved) 
	{
		removeChild((QWidget *)o);
		return true;
	}
	
  QPoint p=viewportToContents(viewport()->mapFromGlobal(QCursor::pos()));
  TileWidget* tw=childAt(p);

	if(e->type()==QEvent::Paint)
	{
		if(canDrawLines())
			blackLines();
	}

  if(e->type()==QEvent::MouseMove && (wResize && !fixedHeight))
  {
    if(isDoResize)
    {
      // continue resize
      int pos = mapFromGlobal(QCursor::pos()).x();
      int newWidth = QMAX( (pos - xSpacing/2)/currentColumn, 50);
      continueRectDraw(newWidth);
    }
    else
    {
      int x = p.x();
      int k = 0;
      while ( x > childW + xSpacing )
      {
        x -= childW + xSpacing;
        k++;
      }
      if (k == 0 || x > xSpacing || !viewport()->geometry().contains(mapFromGlobal(QCursor::pos())) )
      {
        if (canDoResizeHere) {
          setCursor(ArrowCursor);
          viewport()->releaseMouse();
          canDoResizeHere = false;
        }
      }
      else
      {
        if (!canDoResizeHere) {
          canDoResizeHere = true;
          currentColumn = k;
          viewport()->grabMouse();
          setCursor(SizeHorCursor);
        }
      }
    }
  }

  // child resize
  if(e->type()==QEvent::Resize && isUpdatesEnabled() && o->inherits("TileWidget"))
  {
    makePosChain();	
    return QScrollView::eventFilter(o, e);
  }

  if(e->type()==QEvent::MouseButtonRelease)
  {
    if(isDoResize)
    {
      // finish width resize;
      viewport()->releaseMouse();
      isDoResize=false;
      setCursor(ArrowCursor);
      endRectDraw();

      // resize all child
  		setChildWidth(currentColumnWidth-xSpacing);
  		for(unsigned int i=0;i<childViews.count();i++)
  		{
  		  setUpdatesEnabled(false);
  		  (dynamic_cast<TileWidget *>(childViews.at(i)))->setFixedWidth(childW);
  		  (dynamic_cast<TileWidget *>(childViews.at(i)))->update();
  		  setUpdatesEnabled(true);
  		}
      int w=makePosChain();	
      resizeContents(w, viewport()->height());
			if(canDrawLines())
			{
				eraseLines();
				drawLines();
			}
    }
  }

  if(tw) 
	{
    if(e->type()==QEvent::MouseButtonPress)
      handleMousePressEvent(tw, (QMouseEvent*)e);
    if(e->type()==QEvent::MouseButtonDblClick)
      handleMouseDoubleClickEvent(tw, (QMouseEvent*)e);
    if(e->type()==QEvent::MouseButtonRelease)
      handleMouseReleaseEvent(tw, (QMouseEvent *)e);
    if(e->type()==QEvent::MouseMove)
      handleMouseMoveEvent(tw, (QMouseEvent *)e);
    return QScrollView::eventFilter(o,e);
  }

  if(e->type()==QEvent::MouseButtonPress)
  {
    if(canDoResizeHere) 
		{
      // start resize;
      canDoResizeHere=false;
      isDoResize=true;
      currentColumnWidth=childW+xSpacing;
      startRectDraw();
    }
    else
    {
      // unselect all the selected widgets
			unselectAll();

      QMouseEvent *me=(QMouseEvent*)e;
      if(me->button()==RightButton)
        popupGlobal();
    }
  }

  return QScrollView::eventFilter(o,e);
}

void SortingWidget::handleMouseDoubleClickEvent(TileWidget *clicked, QMouseEvent *)
{
	select(clicked);
	clicked->open();
}
	
void SortingWidget::handleMousePressEvent(TileWidget *clicked, QMouseEvent *e)
{
	if(e->button() == RightButton) {
		popupChild();
	} else if(e->button() == LeftButton && e->state() & ControlButton) {
		if(clicked->state() == DataWidget::Normal) {
			addToSelection(clicked);
		} else {
			removeFromSelection(clicked);
		}
	} else if(e->button() == LeftButton) {
		if(clicked->state() != DataWidget::Selected)
			select(clicked);

		m_isDragging = true;
	}
}

void SortingWidget::handleMouseReleaseEvent(TileWidget */*clicked*/, QMouseEvent *e)
{
	if(e->button() == LeftButton)
		m_isDragging = false;
}

void SortingWidget::handleMouseMoveEvent(TileWidget */*clicked*/, QMouseEvent *)
{
	if(m_isDragging) {
		dragStart(viewport());
		m_isDragging = false;
	}
}

void SortingWidget::popupGlobal()
{
  if(mainPopup)
    mainPopup->popup(QCursor::pos());
}

void SortingWidget::popupChild()
{
  if(childPopup && !isSelectionEmpty())
    childPopup->popup(QCursor::pos());
  else
    if(allwaysPopup)
      popupGlobal();
}

void SortingWidget::setAllwaysShowPopup(bool t)
{
  allwaysPopup=t;
}
	
bool SortingWidget::allwaysShowPopup()
{
  return allwaysPopup;
}

void SortingWidget::resizeEvent(QResizeEvent *e)
{
  QScrollView::resizeEvent(e);

  int w=makePosChain();	
  if(fixedHeight)
    resizeContents(viewport()->width(), w);
  else
    resizeContents(w, viewport()->height());
}

void SortingWidget::setChildType(int type)
{
	childType=type;
}

int SortingWidget::getChildType()
{
	return childType;
}

void SortingWidget::clear()
{
	for (unsigned int i = 0; i < childViews.count(); i++)
		delete childViews.at(i);
}

void SortingWidget::enableWidthResize(bool wres)
{
	wResize=wres;
	if (canDrawLines())
		drawLines();
}

bool SortingWidget::widthResizeEnabled()
{
	return wResize;
}

void SortingWidget::setMultipleSelect(bool t)
{
	allowMultiple=t;
}

bool SortingWidget::multipleSelect()
{
	return allowMultiple;
}

void SortingWidget::setChildMimeType(const char *m)
{
	childMimeType=m;
}

const char *SortingWidget::getChildMimeType()
{
	return (const char *)childMimeType;
}

void SortingWidget::makeGlobalPopup()
{
}

void SortingWidget::makeChildPopup()
{
}

void SortingWidget::setShowSorting(bool t)
{
	showSort=t;
}

bool SortingWidget::showSorting()
{
  return showSort;
}

int SortingWidget::makePosChain()
{
  int currentCol=0, viewHeight=viewport()->height(), viewWidth=viewport()->width(), x=xSpacing, y=ySpacing, currentLine=0;
  if (fixedHeight)
  {
    // arrange using horizontal chains
    for(unsigned int i=0;i<childViews.count();i++)
    {
      if((x+childW+xSpacing)>viewWidth && (i>0))
      {
        // make another line
        currentLine++;
        x=xSpacing;
        y=(ySpacing+childH)*currentLine+ySpacing;
      }
      moveChild((dynamic_cast<TileWidget *>(childViews.at(i))),x,y);
      x+=childW+xSpacing;
    }
    return y+childH+ySpacing;
  }
  else
  {
    // arrange using vertical chains
    for(unsigned int i=0;i<childViews.count();i++)
    {
      if(((y+(dynamic_cast<TileWidget *>(childViews.at(i)))->height()+ySpacing)>viewHeight) && (i>0))
      {
        // make another column
        currentCol++;
        x=(xSpacing+childW)*currentCol+xSpacing;
        y=ySpacing;
      }
			
      moveChild((dynamic_cast<TileWidget *>(childViews.at(i))), x, y);
      y+=(dynamic_cast<TileWidget *>(childViews.at(i)))->height()+ySpacing;
    }
    return x+childW+xSpacing;
  }
}

void SortingWidget::beginUnclippedPainter()
{
  endUnclippedPainter();
  bool unclipped=testWFlags(WPaintUnclipped);
  setWFlags(WPaintUnclipped);
  unclippedPainter=new QPainter;
  unclippedPainter->begin(this);

  if(!unclipped)
    clearWFlags(WPaintUnclipped);
	unclippedPainter->setRasterOp(NotROP);
  unclippedPainter->setPen(QPen(gray, 1));
}

void SortingWidget::endUnclippedPainter()
{
  if(unclippedPainter) 
	{
    unclippedPainter->end();
    delete unclippedPainter;
    unclippedPainter=0;
  }
}

void SortingWidget::endRectDraw()
{
  if(!unclippedPainter)
    return;
	if(canDrawLines())
		blackLines();
  int x=currentColumnWidth+xSpacing/2;
  while (x<width())
  {
    unclippedPainter->drawLine(x, 0, x, height());
    x+=currentColumnWidth;
  }
  endUnclippedPainter();
}

void SortingWidget::startRectDraw()
{
  beginUnclippedPainter();
	oldColumnWidth=currentColumnWidth;
  int x=currentColumnWidth+xSpacing/2;
  while (x < width())
  {
    unclippedPainter->drawLine(x, 0, x, height());
    x+=currentColumnWidth;
  }
}

void SortingWidget::continueRectDraw(int newWidth)
{
  if (currentColumnWidth==newWidth)
    return;

  int x=currentColumnWidth+xSpacing/2;
  int x1=newWidth+xSpacing/2;
  while (x<width() || x1<width())
  {
    unclippedPainter->drawLine(x, 0, x, height());
    unclippedPainter->drawLine(x1, 0, x1, height());
    x+=currentColumnWidth;
    x1+=newWidth;
  }
  currentColumnWidth = newWidth;
}

void SortingWidget::drawLines()
{
	if(height()>10 && currentColumnWidth!=0 && wResize)
	{
		QPainter *p=new QPainter;
		p->begin(viewport());
		p->setPen(QPen(BARCOLOR,1));
		for(int i=currentColumnWidth-2; i<width(); i+=currentColumnWidth-2)
			p->drawLine(i + xSpacing/2, 5, i + xSpacing/2, height() - 5);
		p->end();
		delete p;
	}
}

void SortingWidget::eraseLines()
{
	if(height()>10 && currentColumnWidth!=0 && wResize)
	{
		QPainter *p=new QPainter;
		p->begin(viewport());
		p->setPen(QPen(white,1));
		for(int i=oldColumnWidth-2; i<width(); i+=oldColumnWidth-2)
			p->drawLine(i + xSpacing/2, 5, i + xSpacing/2,height() - 5);
		p->end();
		delete p;
	}
}

void SortingWidget::blackLines()
{
	if(height()>10 && currentColumnWidth!=0 && wResize)
	{
		QPainter *p=new QPainter;
		p->begin(viewport());
		p->setPen(QPen(white,1));
		for(int i=oldColumnWidth-2; i<width(); i+=oldColumnWidth-2)
			p->drawLine(i+xSpacing/2, 5, i + xSpacing/2, height() - 5);
		p->setPen(QPen(BARCOLOR,1));
		for(int i=currentColumnWidth-2; i<width(); i+=currentColumnWidth-2)
			p->drawLine(i + xSpacing/2, 5, i + xSpacing/2, height() - 5);
		p->end();
		delete p;
	}
}

bool SortingWidget::canDrawLines()
{
	return _canDrawLines;
}

void SortingWidget::setCanDrawLines(bool item)
{
	_canDrawLines=item;
	if(item)
		drawLines();
}

void SortingWidget::setStartView( const QString& z )
{
  TileWidget* f = 0L;
	
  for(unsigned int i=0;i<childViews.count();i++)
  {
    TileWidget* c = dynamic_cast<TileWidget *>(childViews.at(i));
    if( c->key(currentKey).find(z,0,false) == 0 )
    {
      setStartView(c);
      return;
    } else {
      if ( c->key(currentKey) > z )
        if ( !f || c->key(currentKey).lower() < f->key(currentKey).lower() )
          f = c;
    }
  }
  if (f)
    setStartView(f);
}

void SortingWidget::setStartView( TileWidget* z )
{
  scrollBy(z->x()-xSpacing,0);
}
