/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <qfile.h>
#include <qtextstream.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <mailer.h>
#include <indexmailitem.h>
#include <maillistview.h>
#include <pixmaps.h>
#include <qpainter.h>
#include <qcolor.h>
#include <qstring.h>
#include <mailclasses.h>
#include <qapplication.h>
#include <templateview.h>
#include <mailobj.h>
#include <kconfig.h>
#include <videotoolbox.h>
#include <pixlibrary.h>
#include <saveformatdialog.h>
#include <mainwindow.h>
#include <texthtml.h>

extern KConfig *GlobalConfig;
extern TextHtmlConvertor *textHtmlConverter;

IndexMailItem::IndexMailItem(MailListView *parent, MailObject *item):EListViewItem(parent), TemplateView(item), DataWidget(parent)
{
	update();
	
	if(parent) parent->registerWidget(this);
}

IndexMailItem::~IndexMailItem()
{
}


// TODO: insert a parameter for update (IMAP only)
// Error: we don't need the size of the message from the begining

void IndexMailItem::update()
{
	int type=((MailListView *)listView())->getType();
	MailObject *item=(MailObject *)getVItem();
	QString t;

	if(item)
	{	
		// get the display date option
		GlobalConfig->setGroup("Read Options");
		bool rdate=GlobalConfig->readEntry("Use received date")=="Yes"?true:false;

		// common columns
		int priority=PixLibrary::MailNormalIcon;
		QString messagePriority=item->priority();
		if(messagePriority=="Highest") priority=PixLibrary::MailHighestIcon;
		else if(messagePriority=="High") priority=PixLibrary::MailHigherIcon;
		else if(messagePriority=="Low") priority=PixLibrary::MailLowerIcon;
		else if(messagePriority=="Lowest") priority=PixLibrary::MailLowestIcon;

		setPixmap(0, PixLibrary::thisInstance()->pixmap(priority));
		_attachments=item->attachments();
		setPixmap(1, PixLibrary::thisInstance()->pixmap(_attachments?PixLibrary::MailAttachmentIcon:PixLibrary::MailNormalIcon));

		int messageIcon;
		if(item->status()=="New")
		{
			if(item->isReplied() & item->isForwarded()) messageIcon=PixLibrary::MailNewForwardReplyIcon;
			else if(item->isReplied()) messageIcon=PixLibrary::MailNewReplyIcon;
			else if(item->isForwarded()) messageIcon=PixLibrary::MailNewForwardIcon;
			else messageIcon=PixLibrary::MailNewNormalIcon;
		}
		else
		{
			if(item->isReplied() && item->isForwarded()) messageIcon=PixLibrary::MailOldForwardReplyIcon;
			else if(item->isReplied()) messageIcon=PixLibrary::MailOldReplyIcon;
			else if(item->isForwarded()) messageIcon=PixLibrary::MailOldForwardIcon;
			else messageIcon=PixLibrary::MailOldNormalIcon;
		}

		setPixmap(2, PixLibrary::thisInstance()->pixmap(messageIcon));

		switch(type)
		{
			case mail:
				setText(3, item->from().translated_name);
				setText(4, item->subject());
				
				_date=rdate?item->receivedDate():item->sentDate();
				setText(5, (QCString)_date);
				
				setText(6, item->status());
				
				_flag=item->flag();
				if(_flag)
					setPixmap(7, QPixmap(flag));
				else
					setPixmap(7, QPixmap(noflag));
				
				_size=MAX(1, (item->size())/1024);
				t.setNum(_size);
				t+=" kb";
				setText(8, t);
				break;
			case sent:
				setText(3, (const char *)item->to().toQCString());
				setText(4, item->subject());
				
				_date=item->sentDate();				
				setText(5, (QCString)_date);
				
				setText(6, item->status());
				_size=MAX(1, (item->size())/1024);
				t.setNum(_size);
				t+=" kb";
				setText(7, t);
				break;
			case drafts:
				setText(3, (const char *)item->to().toQCString());
				setText(4, item->subject());
				_size=MAX(1, (item->size())/1024);
				t.setNum(_size);
				t+=" kb";
				setText(5, t);
				break;
			case outgoing:	
				setText(3, (const char *)item->to().toQCString());
				setText(4, item->subject());
				setText(5, item->scheduling());
				_size=MAX(1, (item->size())/1024);
				t.setNum(_size);
				t+=" kb";
				setText(6, t);
				break;
			}

		repaint();
	}
}

void IndexMailItem::paintCell(QPainter * p, const QColorGroup & cg, int column, int width, int align)
{
	MailObject *item=(MailObject *)getVItem();
	
  const char *t=text(column);
  if (!p) return;

  QListView *lv=listView();
  int r=lv?lv->itemMargin():1;
  const QPixmap *icon = pixmap(column);
  p->fillRect(0, 0, width, height(), cg.base());

  int marg=lv?lv->itemMargin():1;

  if(isSelected())
	{
     p->fillRect(r-marg, 0, width-r+marg, height(), cg.text());
     p->setPen(cg.base());
  }
	else
	{
		p->setPen(item->color());
  }

  if(icon)
	{
      p->drawPixmap(r, (height()-icon->height())/2, *icon);
      r+=icon->width()+lv->itemMargin();
  }

  if(t)
		{
			QString src=t, dest;
			if(item->status()=="New")
				p->setFont(QPxFont("helvetica", 12, QFont::Bold));
			else
				p->setFont(QPxFont("helvetica", 12, QFont::Normal));

      p->drawText(r, 0, width-marg-r, height(), align|AlignVCenter, (const char *)fitText(src, width, p->font()));
	  }
}


void IndexMailItem::paintFocus(QPainter *, const QColorGroup &, const QRect &)
{
	// we use this to disable the Qt focus
}

QString IndexMailItem::key(int column, bool /*ascending*/) const
{
	QString tmpString;
	int type=((MailListView *)listView())->getType();
	MailObject *item=(MailObject *)getVItem();
	
	if(!item)
	{
		tmpString=text(column);
	}
	else if((type==mail || type==sent) && column==5)
	{
		QDateTime epoch(QDate(1980, 1, 1));
		tmpString.sprintf("%08d", epoch.secsTo(_date.toQDateTime()));
	}
	else if((type==mail && column==8) || (type==sent && column==7) || (type==outgoing && column==6) || (type==drafts && column==5))
	{
		tmpString.sprintf("%08d", _size);
	}
	else if(type==mail && column==7)
	{
		tmpString=_flag?"a":"b";
	}
	else if(column==1)
	{
		tmpString=_attachments?"a":"b";
	}
	else
	{
		tmpString=text(column);
	}

	return tmpString;
}

void IndexMailItem::setState(ViewState s)
{
	if (viewState==s)
		return;
	
	if (s==DataWidget::Normal)
	{
		QListViewItem::setSelected(false);
		DataWidget::setState(s);
	}
	else
	{
		QListViewItem::setSelected(true);
		DataWidget::setState(Selected);
	}
}

void IndexMailItem::setSelected(bool s)
{
	if (s)
		setState(Selected);
	else
		setState(Normal);
}

void IndexMailItem::save()
{
	SaveFormatDialog *formatDialog=new SaveFormatDialog(MainWindow::ref(), 0, true);
	MailObject *obj=static_cast<MailObject *>(getVItem());

	if(obj)
	{
		obj->sync();
		
		if(!obj->hasHTML())
		{
			formatDialog->originalHTMLRadio->setEnabled(false);
			formatDialog->formattedHTMLRadio->setEnabled(false);
		}
		if(!obj->hasText())
		{
			formatDialog->originalTextRadio->setEnabled(false);
			formatDialog->formattedTextRadio->setEnabled(false);
		}
		
		QString subject=obj->subject();
		subject.replace(QRegExp(QString::fromLatin1("[:/#]")), QString::fromLatin1(" "));
		subject.stripWhiteSpace();

		QString filter;

		if(formatDialog->exec()==QDialog::Accepted)
		{
			if(formatDialog->originalHTMLRadio->isOn()
					|| formatDialog->formattedTextRadio->isOn()
					|| formatDialog->formattedHTMLRadio->isOn())
			{
				filter="*.html";
				subject+=".html";
			}
			else if(formatDialog->entireMessageRadio->isOn())
			{
				filter="*.msg";
				subject+=".msg";
			}
			else
			{
				filter="*.txt";
				subject+=".txt";
			}
			
			QString filename=QFileDialog::getSaveFileName(subject, filter, MainWindow::ref());
			
			if(!filename.isEmpty())
			{			
				QString text;
				if(formatDialog->originalHTMLRadio->isOn())
				{
					text=obj->html();
				}
				else if(formatDialog->originalTextRadio->isOn())
				{
					text=obj->text();
				}
				else if(formatDialog->formattedTextRadio->isOn())
				{
					text=obj->text();
					textHtmlConverter->text2html(text);
				}
				else if(formatDialog->formattedHTMLRadio->isOn())
				{
					text=obj->html();
					textHtmlConverter->html2html(text);
				}
				else
				{
					text=obj->message();
				}
				
				QFile saveFile(filename);
				if(saveFile.open(IO_WriteOnly))
				{
					QTextStream txt(&saveFile);
					txt<<text;
					saveFile.close();
				}
				else
				{
					QMessageBox::warning(MainWindow::ref(), "Aethera: Could not save message",
							"Could not open file for saving.\nPlease check that you have write permissions and try again",
							QMessageBox::Ok, QMessageBox::NoButton);
				}
			}
		}
	}
}


