/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <orb.h>
#include <magellan.h>
#include <mailclasses.h>
#include <indexclass.h>
#include <transactionserver.h>
#include <attributemarshall.h>
#include <conversions.h>
#include <qfileinfo.h>
#include <messagedevice.h>
#include <messagedescriptor.h>
#include <messagefactory.h>
#include <imap4handler.h>

bool ObjectRequestBroker::getObjectData(const QString &url, QByteArray &data)
{
  // debug
  // printf("orb: entering getdata(%s)\n", (const char *)url);
	
  if(!exists(url))
  {
    err="Requested object does not exist";
    return false;
  }
  if(isFolder(url))
  {
    err="Can't apply getObjectData() on a folder";
		
		// debug
		
    return false;
  }
	
  int t=subtype(url);
	if(t==Magellan::FolderCache)
	{
		return getFolderCacheData(url, data);
	}
  if(t==Magellan::FolderAttribute)
  {
		return getFolderAttributeData(url, data);
  }
  if(t==Magellan::MailAttribute)
  {
		return getMailAttributeData(url, data);
  }
  if(t==Magellan::PartAttribute)
  {
		return getPartAttributeData(url, data);
  }
  if(t==Magellan::BlockObjectAttribute)
  {
		return getBlockObjectAttributeData(url, data);
  }
  if(t==Magellan::Attribute)
  {
		return getAttributeData(url, data);
  }
  if(t==Magellan::Message)
  {
		return getMailData(url, data);
  }
  if(t==Magellan::Part)
  {
		return getPartData(url, data);
  }
	if(t==Magellan::BlockObject)
	{
		return getBlockObjectData(url,data);
	}
	if(t==Magellan::FileObject)
	{
		return getFileObjectData(url,data);
	}
	err="Unknown error";
	return false;
}

bool ObjectRequestBroker::getObjectData(const QStringList &urls, QByteArray &data)
{
  QDataStream tempStream(data, IO_WriteOnly);
  QByteArray  tempData;
  bool bRet=true;

  // First insert the number of objects
  tempStream << urls.count();
  // Insert the urls
  if( !urls.count() )
  {
    tempStream << 0;
    printf("\nNothing to sent!");
    fflush(stdout);
    return true;
  }
  tempStream << urls;
  for(QStringList::ConstIterator urlIt=urls.begin(); (urlIt != urls.end()) && bRet; ++urlIt)
  {
  // debug
//   printf("orb: entering getObjectsData(%s)\n", (const char *)(*urlIt));

#if 0
    if(!exists(*urlIt))
    {
      err="Requested object does not exist";
      bRet=false;
      break;
    }
    if(isFolder(*urlIt))
    {
      err="Can't apply getObjectsData() on a folder";
      bRet=false;
      break;
    }
#endif
  	
    int t=subtype((*urlIt));
    tempData.resize(0);
    switch(t)
    {
      case Magellan::FolderCache :
  		  bRet=getFolderCacheData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::FolderAttribute :
  		  bRet=getFolderAttributeData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::MailAttribute :
  		  bRet=getMailAttributeData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::PartAttribute :
  		  bRet=getPartAttributeData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::BlockObjectAttribute :
  		  bRet=getBlockObjectAttributeData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::Attribute :
  		  bRet=getAttributeData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::Message :
  		  bRet=getMailData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::Part :
  		  bRet=getPartData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::BlockObject :
  		  bRet=getBlockObjectData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      case Magellan::FileObject :
  		  bRet=getFileObjectData((*urlIt), tempData);
  		  tempStream<<tempData;
        break;
      default:
        err="Unknown error";
        bRet=false;
        break;
    }
  }

//  printf("\nGetObjectsData----- Done !");
//  fflush(stdout);
	return bRet;
}

bool ObjectRequestBroker::getChildrenData(const QString &url, QByteArray &data)
{
  QDataStream tempStream(data, IO_WriteOnly);

  // debug
   printf("orb: entering getChildrenData(%s)\n", (const char *)url);
   fflush(stdout);
	
  if(!url || url.isNull() || url.isEmpty() || !exists(url))
  {
    err="Requested object does not exist";
    tempStream << 0;
    return false;
  }
  if(!isFolder(url))
  {
    err="Can apply getObjectsData() only for a folder!\n";
    err+=url;
    err+=" isn't a folder.";
    return false;
  }
  // get the url from the id.
  QStringList urls=children(url);
  if( !urls.count() )
  {
    tempStream << 0;
    return(true);
  }
  else
  {
    for(QStringList::Iterator it=urls.begin(); it!=urls.end(); it++)
      (*it).insert(0,url+"/");
    return getObjectData(urls, data);
  }
}

bool ObjectRequestBroker::getFolderCacheData(const QString &url, QByteArray &data)
{
	MailFolder *mailFolder=mailfolderReference(folder(url));
	if(!mailFolder)
	{
		err="Cannot retrieve cache for non-mail folders";
		printf("orb: Warning, folder %s is not a mail folder, cannot retrieve cache\n", (const char *)folder(url));
		return false;
	}
	else
	{
		data=_stream(mailFolder->entryCacheBlock());
		return true;
	}
}

bool ObjectRequestBroker::getFolderAttributeData(const QString &url, QByteArray &data)
{
	// debug
	// printf("orb: %sstatting folder %s\n", url.contains("fattr")?"fast ":"", (const char *)folderpath(url));

  QString folder=folderpath(url);
  _FolderAttribute fattr;
  if(type(folder)==Magellan::Message)
  {
		// debug
		// printf("orb: this is a mail folder, proceeding\n");

    MailFolder *mfolder=mailfolderReference(folder);

		if(url.contains(".attr"))
		{
			QFileInfo finfo(mfolder->getStorageDevice());

      fattr.owner=finfo.owner();
      fattr.device=mfolder->getStorageDevice();
      fattr.rtype="indexed,extended:(data descriptors,multipart storage,uniblock storage,incremental data change,incremental index change,data relocation)";
      fattr.size=finfo.size();
      fattr.lmodified=finfo.lastModified();	
	    fattr.folders=0; // no recursive folders
		}
    fattr.items=mfolder->getMessageCount();
		fattr.recent=mfolder->getUnread();

    data=_stream(fattr);
    return true;
  }
  else
  {
		// debug
		// printf("orb: this is a generic folder, proceeding\n");

    DataCollection *dfolder=folderReference(folder);

		if(url.contains(".attr"))
		{
			QFileInfo finfo(dfolder->storageDevice);

      fattr.owner=finfo.owner();
      fattr.device=dfolder->storageDevice;
      fattr.device=dfolder->type==DataCollection::FileStorage?"file":"folder";
      fattr.rtype="sequential";
      fattr.size=finfo.size();
      fattr.lmodified=finfo.lastModified();
      fattr.folders=0; // no recursive folders
		}

    fattr.items=dfolder->getEntryList().count();
		fattr.recent=0;

    data=_stream(fattr);
    return true;
  }
}

bool ObjectRequestBroker::getMailAttributeData(const QString &url, QByteArray &data)
{
	QString objectUrl=url.left(url.find('.'));
  _MailAttribute attr;
  IndexClass *idx=indexReference(objectUrl);

	if(!idx)
	{
		err="Error getting the mail attribute data: object does not exist";
		
		printf("Warning: %s\n", (const char *)err);
		
		return false;
	}
	
	if(url.contains(".attr"))
	{
		QFileInfo finfo(idx->getParentFolder()->getStorageDevice()+"/messages");

    attr.owner=finfo.owner();
    attr.device=idx->getParentFolder()->getStorageDevice()+"/messages";
    attr.device="file";
    attr.rtype="indexed,extended:(data descriptors,multipart storage,uniblock storage,incremental data change,incremental index change,data relocation)";
    attr.size=finfo.size();
    attr.lmodified=finfo.lastModified();
	}

	attr.uniblockOffset=idx->getUniblockOffset();
	attr.uniblockLength=idx->getUniblockLength();
	attr.descriptorOffset=idx->getDescriptorOffset();
	attr.descriptorLength=idx->getDescriptorLength();
  attr.partCount=idx->getPartCount();
	attr.multipartOnly=idx->isMultipartOnly();

  data=_stream(attr);
  return true;
}

bool ObjectRequestBroker::getPartAttributeData(const QString &url, QByteArray &data)
{
	QString objectUrl=url.left(url.find('.'));
  _PartAttribute attr;
  IndexClass *idx=indexReference(objectUrl);

	if(!idx)
	{
		err="Error getting the part attribute data: object does not exist";
		
		printf("Warning: %s\n", (const char *)err);
		
		return false;
	}
	
	if(url.contains(".attr"))
	{
		QFileInfo finfo(idx->getParentFolder()->getStorageDevice()+"/messages");

    attr.owner=finfo.owner();
    attr.device=idx->getParentFolder()->getStorageDevice()+"/messages";
    attr.device="file";
    attr.rtype="indexed,extended:(data descriptors,multipart storage,uniblock storage,incremental data change,incremental index change,data relocation)";
    attr.size=finfo.size();
    attr.lmodified=finfo.lastModified();
	}


  int pBegin=url.find('.')+1, pEnd=url.findRev('.');
  MimePart *part=idx->getPartAt(url.mid(pBegin, pEnd-pBegin).toInt());

  attr.mimetype=part->mimetype;
  attr.name=part->name;
  attr.cid=part->cid;
  attr.encoding=part->encoding;
  attr.charset=part->charset;
  attr.mtype=part->type;
  attr.partOffset=part->offset;
  attr.partLength=part->length;
	attr.embeddedPath=part->embeddedPath;

  data=_stream(attr);
  return true;
}

bool ObjectRequestBroker::getBlockObjectAttributeData(const QString &url, QByteArray &data)
{
	QString objectUrl=url.left(url.find('.'));
  _BlockObjectAttribute attr;
  ObjectReference *ref=objectReference(objectUrl);
  DataCollection *dfolder=folderReference(folderpath(url));

	if(!ref || !dfolder)
	{
		err="Error getting the block object attribute data: object does not exist";
		
		printf("Warning: %s\n", (const char *)err);
		
		return false;
	}
		
	if(url.contains(".attr"))
	{
		QFileInfo finfo(dfolder->storageDevice);

    attr.owner=finfo.owner();
    attr.device=dfolder->storageDevice;
    attr.device="file";
    attr.rtype="sequential";
    attr.size=ref->length;
    attr.lmodified=ref->lastModified;
	}

  data=_stream(attr);
  return true;
}

bool ObjectRequestBroker::getBlockObjectData(const QString &url, QByteArray &data)
{
	QString objectUrl=url.left(url.find('.'));
  ObjectReference *ref=objectReference(objectUrl);
	if(!ref)
		return false;
	data=ref->rawData;
	return true;
}

bool ObjectRequestBroker::getFileObjectData(const QString &url, QByteArray &data)
{
	QString objectUrl=url.left(url.find('.'));
  ObjectReference *ref=objectReference(objectUrl);
	printf("ref %p\n",ref);
	if(!ref)
		return false;
	data=ref->rawData;
	return true;
}

bool ObjectRequestBroker::getAttributeData(const QString &url, QByteArray &data)
{
	// debug
	// printf("orb: attribute query in progress\n");

	QString objectUrl=url.left(url.find('.'));
  _Attribute attr;

	// debug
	// printf("orb: %s is a file attribute, stat follows\n", (const char *)url);

  ObjectReference *ref=objectReference(objectUrl);

	// debug
	// printf("orb: object reference: %p\n", ref);
	// printf("orb: filename: %s\n", (const char *)(folderReference(folderpath(objectUrl))->storageDevice+"/"+ref->filename));

	if(!ref)
	{
		err="Error getting the object attribute data: object does not exist";
		
		printf("Warning: %s\n", (const char *)err);
		
		return false;
	}
	
	if(url.contains(".attr"))
	{
		QFileInfo finfo(folderReference(folderpath(objectUrl))->storageDevice+"/"+ref->filename);

    attr.owner=finfo.owner();
    attr.device=folderReference(folderpath(objectUrl))->storageDevice+"/"+ref->filename;
    attr.dtype="file, extended:(standalone data)";
    attr.rtype="random";
	  attr.size=finfo.size();
	  attr.lmodified=finfo.lastModified();
	}

	// printf("orb: got object attribute; owner:%s\n", (const char *)attr.owner);

  data=_stream(attr);
  return true;
}

bool ObjectRequestBroker::getMailData(const QString &url, QByteArray &data)
{
	// extract query
	QString property, objectUrl=object(url);
	
	if(url.contains('%'))
	{
		property=url.mid(url.find('%')+1).lower();
		objectUrl=url.left(url.findRev('%'));
		return getMailPropertyData(objectUrl, property, data);
	}

	IndexClass *idx=indexReference(url);
  if(!idx)
  {
    printf("Warning: indexReference() returned null when a vadid pointer was expected, bailing out\n");
    err="Internal error in URLManager::indexReference()";
    return false;
  }

	// debug
	// printf("orb: getting message descriptor for object %s, index %s(%p)...\n", (const char *)url, (const char *)idx->getID(), idx);
	
	QDataStream stream(data, IO_WriteOnly);
	
	// debug
	// printf("orb: index parent name is %s\n", (const char *)idx->getParentFolder()->name());
	// printf("orb: index message file is %s\n", (const char *)idx->getDataFilename());
	
	MessageDevice *dev=MessageFactory::thisInstance()->getMessage(idx);
	if(dev)
	{
		dev->loadDescriptor(false);
		stream<<dev->getDescriptor();
	}
	else
	{
		printf("Warning: MessageFactory::getMessage() returned null when a valid pointer was expected, bailing out\n");
		err="Internal error in MessageFactory::getMessage()";
		return false;
	}
	
	return true;
}

bool ObjectRequestBroker::getMailPropertyData(const QString &path, const QString &property, QByteArray &data)
{
	IndexClass *idx=indexReference(path);
  if(!idx)
  {
    printf("Warning: indexReference() returned null when a vadid pointer was expected, bailing out\n");
    err="Internal error in URLManager::indexReference()";
    return false;
  }

//  printf("\nProperty=%s\n", property.latin1());
//  fflush(stdout);

  // check if it is an IMAP folder
//  if( property!="size" && property!="hasattachments")
  if( property=="text" || property=="html" ||
      property=="alltext" || property=="allhtml" )
    if( MailFolder::IMAP4==idx->getParentFolder()->getFolderProtocol() && !idx->isFullMessage() )
   	  IMAP4Handler::ref()->updateMessage(*idx);
 	
	MessageDevice dev(idx);
 	
	if(property=="text")
	{
		data=dev.text();
	}
	else if(property=="alltext")
	{
		data=dev.text(true);
	}
	else if(property=="html")
	{
		data=dev.html();
	}
	else if(property=="allhtml")
	{
		data=dev.html(true);
	}
	else if(property=="rfc822header")
	{
		data=dev.rfc822Header();
	}
	else if(property=="rfc822message")
	{
		data=dev.rfc822Message();
	}
	else if(property=="textpart")
	{
		data=_stream(dev.textPart());
	}
	else if(property=="messagetext")
	{
		data=_stream(dev.messageText());
	}
	else if(property=="htmlpart")
	{
		data=_stream(dev.htmlPart());
	}
	else if(property=="partcount")
	{
		data=_stream(idx->getPartCount());
	}
	else if(property=="hashtml")
	{
		data=_stream(dev.hasHtml());
	}
	else if(property=="hastext")
	{
		data=_stream(dev.hasText());
	}
	else if(property=="hasattachments")
	{
		data=_stream(dev.hasAttachments());
	}
	else if(property=="size")
	{
		data=_stream(idx->getSize());
	}
	else
	{
		// debug
		printf("orb: warning: unknown message query (%s) on object %s\n", (const char *)property, (const char *)path);
		
		err="Unknown message query: "+property;
		return false;
	}
	
	return true;
}

bool ObjectRequestBroker::getPartData(const QString &url, QByteArray &data)
{
	QString objectUrl=url.left(url.find('.'));
  IndexClass *idx=indexReference(objectUrl);
	
  if(!idx)
  {
    printf("Warning: indexReference() returned null when a vadid pointer was expected, bailing out\n");
    err="Internal error in URLManager::indexReference()";
    return false;
  }

  int k=url.findRev('.');
  int p=url.mid(k+1).toInt();
  if(p<0 || p>idx->getPartCount())
  {
    err="Non-existent part requested";
    return false;
  }
	
	MessageDevice dev(idx);
	dev.getPartData(p, data);
	
	return true;
}

bool ObjectRequestBroker::getCollectionObjectData(const QString &url, QByteArray &data)
{
  ObjectReference *tobj=objectReference(url);
	
  if(tobj)
  {
    // debug
    // printf("orb: narrowed object reference (%p)\n", tobj);
		// printf("orb: object data size %d\n", tobj->rawData.size());

    data=tobj->rawData;
    return true;
  }

  printf("Warning: objectReference() returned null when a valid pointer was expected, bailing out\n");
  err="Internal error in URLManager::objectReference()";
  return false;
}





