/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */
 
#include <elineedit.h>
#include <qlineedit.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qlabel.h>
#include <qfont.h>
#include <stdio.h>
#include <qregexp.h>
#include <qapplication.h>
#include <qpainter.h>

#define IDSTRING "ELineEdit: "

//#define DEBUG_ELE
//#define DEBUG_MLE

ELineEdit::ELineEdit(QWidget *parent, const char *name):QWidget(parent, name)
{
	// build a new rtf label menu
	menu=new QLabel(0, 0, WStyle_Customize|WStyle_NoBorder|WStyle_Tool);
	menu->setFrameStyle(QFrame::Box|QFrame::Plain);
	menu->setLineWidth(1);
	menu->setFont(QFont("Helvetica",10));
	menu->setTextFormat(RichText);

	// build a new edit box
	edit=new MatchLineEdit(this);
	edit->resize(width(), height());
	
	// install the event filter
	menu->installEventFilter(this);
	edit->installEventFilter(this);
	
	// set the default active option to "none"
	activeOption=-1;
	
	// enable multiple matches
	multiString=true;
	
	// set the menu options
	xRel=5;
	yRel=5;
	menu->setBackgroundColor(white);
	thd=2;
	limit=10;

	connect(edit, SIGNAL(returnPressed()), this, SLOT(getReturnPressed()));
	connect(edit, SIGNAL(textChanged(const QString &)), this, SLOT(getTextChanged(const QString &)));
		
	edit->show();
}

ELineEdit::~ELineEdit()
{
	delete(menu);
}

QString ELineEdit::text()
{
	return edit->text();
}

void ELineEdit::setCompletions(QStringList cList)
{
	// save active option text
	QString opt;
	if(activeOption!=-1) opt=compList[activeOption];
	
	// set new completion list
	compList=cList;
	activeOption=-1;
	
	// try to find the old option
	for(int i=0;i<compList.count();i++)
		if(compList[i]==opt)
			activeOption=i;
	
	// update completion menu
	updateMenu();
}

bool ELineEdit::eventFilter(QObject *obj, QEvent *ev)
{
	if(ev->type()==QEvent::KeyPress && obj==edit)
	{
		updateMenu();
		int count=compList.count();
#ifdef DEBUG_ELE
		printf("%d options\n", count);
#endif		if(count>0)
		{
			switch(((QKeyEvent *)ev)->key())
			{
				case Key_Up:
					if(activeOption>0)
						activeOption--;
					break;
				case Key_Down:
					if(activeOption<(count-1))
						activeOption++;
					break;
				case Key_PageUp:
					activeOption=0;
					break;
				case Key_PageDown:
					activeOption=count;
			}
#ifdef DEBUG_ELE
			printf("active option: %d of %d\n", activeOption, count);
#endif
		}
		updateMenu();
	}
	
	if((ev->type()==QEvent::FocusIn || ev->type()==QEvent::Enter) && obj==edit)
		updateMenu();
	
	if((ev->type()==QEvent::FocusOut || ev->type()==QEvent::Leave) && obj==edit)
		menu->hide();
	
	return false;
}

void ELineEdit::getTextChanged(const QString &t)
{
//	edit->updateMatches(edit->cursorPosition(),t.length());
	QString token=getCurrentChunk();
#ifdef DEBUG_ELE
	printf("getTextChanged: TOKEN : <%s>\n", (const char *) token);
#endif
	int pos,len;
	getCurrentChunkPos(pos, len);
#ifdef DEBUG_ELE
	printf("getTextChanged: TOKEN : <%d %d>\n", pos, len);
#endif
	if(len>0)
	{
		if(compList.count()==1 && compList[0].lower().stripWhiteSpace()==token.lower().stripWhiteSpace())
			edit->setChunk(pos, len, MatchLineEdit::Matched);
		else
			edit->setChunk(pos, len, MatchLineEdit::Unmatched);
		updateMenu();
		emit(textChanged(getCurrentChunk()));
	}
#ifdef DEBUG_ELE
	printf("getTextChanged: DONE\n");
#endif
}

void ELineEdit::getReturnPressed()
{
	// debug
#ifdef DEBUG_ELE
	if(activeOption!=-1)
		printf("returnPressed: selected option: %s\n", (const char *)compList[activeOption]);
	else
		printf("returnPressed: no active option\n");
#endif
	
#ifdef DEBUG_ELE
	printf("returnPressed: current chunk: %s\n", (const char *)getCurrentChunk());
	printf("returnPressed: cursor position: %d\n", edit->cursorPosition());
#endif	
	// fill in the selected option, if any
	if(compList.count() && activeOption!=-1 && compList.count()>activeOption)
		setCurrentChunk(compList[activeOption]);
#ifdef DEBUG_ELE
	printf("returnPressed: --\n");
#endif	
	// keep QLineEdit compatibility
	emit(returnPressed());
}

void ELineEdit::updateMenu()
{
#ifdef DEBUG_ELE
	printf("updateMenu:\n");
#endif
	if(compList.count()==0)
	{
		int pos,len;
		getCurrentChunkPos(pos, len);
		edit->setChunk(pos, len, MatchLineEdit::Unmatched);
		if(menu->isVisible()) menu->hide();
#ifdef DEBUG_ELE
		printf("updateMenu: compList.count()=0\n");
#endif
		return;
	}
	QString token=getCurrentChunk();
	
	// check to see if it's already matching...
	// this is just an enhancement (not needed since you have the menu all the time)

	if(compList.count()==1 && compList[0].lower().stripWhiteSpace()==token.lower().stripWhiteSpace())
	{
		int pos,len;
		getCurrentChunkPos(pos, len);
		edit->setChunk(pos, len, MatchLineEdit::Matched);
#ifdef DEBUG_ELE
		printf("updateMenu: compList.count()=1 (single match)\n");
#endif
		menu->hide();
		return;
	}

#ifdef DEBUG_ELE
	printf("updateMenu: current chunk: %s\n", (const char *)token);
#endif
	
	// check the completion treshold
	if(token.stripWhiteSpace().length()<thd)
	{
		menu->hide();
		return;
	}
			
	// set the highlighted item to the previous value, if still present, or first value
	QString hItem;
	if(activeOption!=-1)
		hItem=compList[activeOption];
	
	for(int i=0;i<compList.count() && activeOption==-1 && !hItem.isEmpty();i++)
	{
		if(compList[i]==hItem)
		{
			activeOption=i;
		}
	}
	
#ifdef DEBUG_ELE
	printf("update: activeOption: %d of %d\n", activeOption, compList.count());
#endif	
	// update the menu options
	QString menuText;
	if(compList.count()>0)
	{
		if(compList.count()>limit)
		{
			menuText="..nbsp&;too many possibilities";
		}
		else
		{
			for(int i=0;i<compList.count();i++)
			{
				QString option=htmlFormat(compList[i]);
				if(i==activeOption) option.append("</b>").prepend("<b>");
				if(i) menuText.append("<br>");
				menuText.append(option);
			}
		}
	}
	
	// decide whether we are going to show the menu
	bool showMenu=false;
	
	if(menuText!=token.prepend("<b>").append("</b>") && compList.find(token)==compList.end())
	{
		showMenu=true;
		menu->setText(menuText);
		if(compList.count()<2)
			menu->setAlignment(AlignLeft | SingleLine);
		else
			menu->setAlignment(AlignLeft);
	}
		
	// move the menu according to the new cursor position; this has a strong limitation: if the text is scrolled the positioning gets screwed; unfortunately there is no way to find the real offset
	QPoint p=edit->mapToGlobal(edit->pos());
	int xOffset=edit->fontMetrics().width(edit->text().left(edit->cursorPosition()));
	p.setX(p.x()+xOffset+xRel);
	p.setY(p.y()+edit->height()+yRel);
	if(showMenu)
	{
		menu->move(p.x(), p.y());
		menu->adjustSize();
		menu->show();
		menu->repaint(true);
	}
	else
		menu->hide();
#ifdef DEBUG_ELE
	printf("updateMenu: finished\n");
#endif
}

void ELineEdit::resizeEvent(QResizeEvent *ev)
{
	QWidget::resizeEvent(ev);	
	edit->resize(width(), height());
}

QLineEdit *ELineEdit::lineEdit()
{
	return edit;
}

QLabel *ELineEdit::menuLabel()
{
	return menu;
}

void ELineEdit::getCurrentChunkPos(int &pos, int &len)
{
	QString eText=edit->text();
	int cpos=edit->cursorPosition();
	if(cpos>0) cpos--;
	QRegExp sepRX("[,;]");
			
	pos=eText.findRev(sepRX, cpos);
	if(pos==-1)
		pos=0;
	else
		pos++;
	
	int fpos=eText.find(sepRX, cpos);
	if(fpos==-1)
		len=eText.length()-pos;
	else
		len=fpos-pos;
#ifdef DEBUG_ELE
	printf(" getCurrentChunkPos cursor: %d, pos: %d, len:%d\n", edit->cursorPosition(), pos, len);
#endif
}

QString ELineEdit::getCurrentChunk()
{
	if(!multiString) return edit->text();
	
	int pos, len;
	getCurrentChunkPos(pos, len);
	return edit->text().mid(pos, len).stripWhiteSpace();
}

void ELineEdit::setCurrentChunk(const QString &newChunk)
{
	int cpos=edit->cursorPosition();
	QString myChunk=translateString(textFormat(newChunk));
	activeOption=-1;
	if(!multiString) return edit->setText(myChunk);
	
	int pos, len;
	int initPos;
	getCurrentChunkPos(pos, len);
	initPos=pos;
#ifdef DEBUG_ELE
	printf("1 ---- Current chunk position p=%d l=%d cur=%d\n", pos, len, edit->cursorPosition());
#endif
	QString tempString=edit->text().replace(pos, len, myChunk);
#ifdef DEBUG_ELE
	printf("1.5 ---- Current chunk position p=%d l=%d cur=%d\n", pos, len, edit->cursorPosition());
#endif
	edit->setText(tempString);
	edit->setCursorPosition(initPos);
#ifdef DEBUG_ELE
	printf("2 ---- Current chunk position p=%d l=%d cur=%d\n", pos, len, edit->cursorPosition());
#endif
	// insert a whitespace if needed
	if(myChunk.find(" ")!=0 && pos)
		edit->insert(" ");
	edit->setCursorPosition(initPos+myChunk.length());
#ifdef DEBUG_ELE
	printf("3 ---- Current chunk position p=%d l=%d cur=%d\n", pos, len, edit->cursorPosition());
#endif
	if(pos>0) pos --;
	// set current position at the end of the current chunk
	getCurrentChunkPos(pos, len);
#ifdef DEBUG_ELE
	printf("Current chunk position p=%d l=%d cur=%d\n", pos, len, edit->cursorPosition());
#endif
	edit->setChunk(pos, len, MatchLineEdit::Matched);
#ifdef DEBUG_ELE
	printf("Current chunk position p=%d l=%d cur=%d\n", pos, len, edit->cursorPosition());
#endif
	edit->setCursorPosition(pos+len);
}


QString ELineEdit::htmlFormat(const QString &normalText)
{
	QString text(normalText);
	int idx;
	
	while((idx=text.find("<"))!=-1)
		text.replace(idx, 1, "&lt;");
	while((idx=text.find(">"))!=-1)
		text.replace(idx, 1, "&gt;");
	while((idx=text.find(" "))!=-1)
		text.replace(idx, 1, "&nbsp;");
	
	return text;
}
QString ELineEdit::textFormat( const QString &htmlText)
{
	QString text(htmlText);
	int idx;
	
	while((idx=text.find("&lt;"))!=-1)
		text.replace(idx, 4, "<");
	while((idx=text.find("&gt;"))!=-1)
		text.replace(idx, 4, ">");
	while((idx=text.find("&nbsp;"))!=-1)
		text.replace(idx, 6, " ");
	
	return text;
	
}
QString ELineEdit::translateString(QString item)
{
	return item;
}

/*--------------------------------------------*/
MatchLineEdit::MatchLineEdit(QWidget *parent, const char *name) : QLineEdit(parent,name)
{
	p = new QPainter;
	connect(this, SIGNAL(textChanged(const QString &)), this, SLOT(updateMatchesLength(const QString &)));
	connect(this, SIGNAL(signalTextChanged(const QString &, int)),
		this, SLOT(updateBuffer(const QString &, int)));
}

MatchLineEdit::~MatchLineEdit()
{
	delete p;
}

void MatchLineEdit::paintEvent(QPaintEvent *pe)
{
#ifdef DEBUG_MLE
	printf("painting: cPos=%d  >%d<\n", cursorPosition(), edited());
#endif
	if(edited())
		emit(signalTextChanged(text(), Cut));
	QLineEdit::paintEvent(pe);
//	repaint(0, h, width(), h+1, true);
	p->begin(this);
	int offset=4;
	int h=height()/2+ p->fontMetrics().height()/2;
	int dx;
	QPen rosu(red, 1);
	QPen negru(black, 1);
	bool inToken=false;
	for(int i=0;i<matches.length() && ((const char *)matches)[i]!='\0';i++)
	{
		dx=p->fontMetrics().width(text()[i]);
		if(((const char *)text())[i]==',' || ((const char *)text())[i]==';')
			inToken=false;
		if(((const char *)text())[i]!=' ')
			inToken=true;
		if(((const char *)matches)[i]!=' ' && inToken)
		{
			p->setPen(negru);
			p->drawLine(offset,h,offset+dx,h);
//			p->drawLine(offset,h,offset+1,h);
		}
		if(((const char *)matches)[i]=='x')
		{
			p->setPen(rosu);
			p->drawLine(offset,h+2,offset+dx,h+2);
		}
		offset+=dx;
	}
//	printf(" Painted %d pixels\n", offset);
	offset=4;
	p->end();
}

void MatchLineEdit::updateMatchesLength(const QString &txt)
{
#ifdef DEBUG_MLE
	printf("-------------->>>>> Text has changed\n");
#endif
	if(txt.length()!=matches.length()+1)
		emit(signalTextChanged(txt, Paste));
	else
		emit(signalTextChanged(txt, Normal));
}

void MatchLineEdit::updateBuffer(const QString &txt, int cutPaste)
{
	cPos=cursorPosition()-1;
	if(cPos>text().length())
		cPos=text().length()-1;
	static QChar c(' ');
#ifdef DEBUG_MLE
	printf("updateBuffer: text xhanged, cut=%d\n", cutPaste);
#endif
	int difference=matches.length()-txt.length();
	switch(cutPaste)
	{
		case Cut:
#ifdef DEBUG_MLE
			printf("\tCut\n");
#endif
			cPos++;
			break;
		case Paste:
#ifdef DEBUG_MLE
			printf("\tPaste %d\n", -difference);
#endif
//			return;
		case Normal:
		default:
#ifdef DEBUG_MLE
			printf("\tNormal\n");
#endif
			if(cPos<=1 && difference>0)
			{
				repaint(true);
				return;
			}
			break;
	}
	if(difference<0)
	{
#ifdef DEBUG_MLE
		printf("insert text @ %d %d\n", cPos, -difference);
#endif
		matches.insert(cPos, &c, - difference);
	}
	else
	{
#ifdef DEBUG_MLE
		printf("remove text @ %d %d\n", cPos, difference);
#endif
		matches.remove(cPos, difference);
	}
#ifdef DEBUG_MLE
	printf(" Matches: [%s] %d\n",(const char *)matches, matches.length());
	printf(" Text   : [%s] %d\n",(const char *)txt, txt.length());
#endif
	matches.truncate(txt.length());
	setEdited(false);
	repaint(true);
}

void MatchLineEdit::setChunk(int pos, int len, int value=Unmatched)
{
	if(len <= 0)
		return;
	QString filled;
	switch(value)
	{
		case Unmatched:
			filled.fill(' ', len);
			break;
		case Matched:
			filled.fill('-', len);
			break;
		case SpecialMatched:
			filled.fill('X', len);
			break;
		default:
			filled.fill(' ', len);
	}
#ifdef DEBUG_MLE
	printf("setChunk: Pos: %d, len :%d\n",pos, len);
	printf("setChunk: Matches: [%s] %d\n",(const char *)matches, matches.length());
	printf("setChunk: Text   : [%s] %d\n",(const char *)text(), text().length());
	printf("setChunk:\t\t-- replaced --\n");
#endif
	matches.replace(pos, len, filled);
#ifdef DEBUG_MLE
	printf("setChunk: Matches: [%s] %d\n",(const char *)matches, matches.length());
	printf("setChunk: Text   : [%s] %d\n",(const char *)text(), text().length());
#endif
	repaint(true);
}
