/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <brokeriface.h>
#include <qstring.h>
#include <qstringlist.h>
#include <magellan.h>
#include <commands.h>
#include <statuscodes.h>
#include <commframe.h>
#include <commbinder.h>
#include <qcstring.h>
#include <qdatastream.h>
#include <conversions.h>
#include <queryprocessor.h>
#include <clientvfs.h>

static BrokerInterface *inst;

BrokerInterface::BrokerInterface():UI_Binder()
{
	// init the CommBinder and the loopback interface
	if(!CommBinder::thisInstance()->connect("thisInstance"))
	{
		printf("Warning: error initiating the \"thisInstance\" loopback server binding, bailing out\n");
		return;
	}
	
	// test the client-server loopback link
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::RcptOk;
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return;
	}
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		printf("Warning: the server did not complete the check request\n");
		return;
	}
	
	// everything went fine
	// printf("BrokerInterface: loopback client/server link initiated successfully\n");
}

BrokerInterface::~BrokerInterface()
{
	CommBinder::thisInstance()->disconnect("thisInstance");
}

BrokerInterface *BrokerInterface::thisInstance()
{
	return inst==0?inst=new BrokerInterface():inst;
}

bool BrokerInterface::isFolder(QString url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::UrlIsFolder;
	f.extraData=_stream(url);
	
	// debug
	// printf("param: %s, from stream: %s\n", (const char *)url, (const char *)_narrowQString(f.extraData));
			
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return _narrowBool(r.extraData);
}

bool BrokerInterface::exists(QString url)
{
	if(url.isEmpty()) return false;

	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::UrlExists;

  // debug
  // printf("brokeriface: sending exists(%s), mapped from %s\n", (const char *)ClientVFS::thisInstance()->map(url), (const char *)url);

	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return _narrowBool(r.extraData);
}

QString BrokerInterface::parent(QString url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetParent;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return QString::null;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return QString::null;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return QString::null;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return QString::null;
	}	
	
	return ClientVFS::thisInstance()->reverseMap(_narrowQString(r.extraData));
}

QString BrokerInterface::folder(QString url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetFolder;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return QString::null;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return QString::null;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return QString::null;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return QString::null;
	}	
	
	return ClientVFS::thisInstance()->reverseMap(_narrowQString(r.extraData));
}

QStringList BrokerInterface::children(QString url)
{
	QStringList eList;
	
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetChildren;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	// debug
	// printf("brokerinterface: [children] url: %s, mapping %s\n", (const char *)url, (const char *)ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return eList;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return eList;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return eList;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return eList;
	}	
	
  //debug
  // printf("brokeriface: entrylist query: Completed_Yes\n");

	return ClientVFS::thisInstance()->reverseMap(_narrowQStringList(r.extraData));
}

QStringList BrokerInterface::subfolders(QString url)
{
	QStringList eList;
	
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetSubfolders;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return eList;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return eList;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return eList;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return eList;
	}	
	
	return ClientVFS::thisInstance()->reverseMap(_narrowQStringList(r.extraData));
}

QStringList BrokerInterface::related(QString url)
{
	QStringList eList;
	
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetRelated;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return eList;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return eList;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return eList;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return eList;
	}	
	
	return _narrowQStringList(r.extraData);
}

int BrokerInterface::type(QString url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetType;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return Magellan::Unknown;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return Magellan::Unknown;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return Magellan::Unknown;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return Magellan::Unknown;
	}	
	
	return _narrowInt(r.extraData);
}

int BrokerInterface::subtype(QString url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetSubtype;
	f.extraData=_stream(url);
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return Magellan::Unknown;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return Magellan::Unknown;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return Magellan::Unknown;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return Magellan::Unknown;
	}	
	
	return _narrowInt(r.extraData);
}


bool BrokerInterface::setRelated(QString url, QStringList rList)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::SetRelated;
	QDataStream s(f.extraData, IO_WriteOnly);
	s<<ClientVFS::thisInstance()->map(url)<<rList;
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}

	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return true;
}

bool BrokerInterface::setParent(QString url, QString parentUrl)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::SetParent;
	QDataStream s(f.extraData, IO_WriteOnly);
	s<<ClientVFS::thisInstance()->map(url)<<ClientVFS::thisInstance()->map(parentUrl);
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return true;
}

bool BrokerInterface::getObjectData(QString url, QByteArray &data)
{
	// query check
	if(url.find("?")>-1)
	{
		bool res=QueryProcessor::thisInstance()->getData(url, data);
		err=QueryProcessor::thisInstance()->lastError();
		return res;
	}
	
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetData;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));

	// debug
	// printf("brokeriface: sending getData(%s), mapped from %s\n", (const char *)ClientVFS::thisInstance()->map(url), (const char *)url);
		
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}

	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	

	data=r.extraData;
	
	// debug
	// printf("brokeriface: data size: %d\n", data.size());
	
	return true;
}
		
bool BrokerInterface::getServerObjectsData(QStringList &urls, QByteArray &data)
{
// query check - unavailable
// TODO: query for a group of objects
	
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetObjectsData;
	f.extraData=_stream(urls);

	// debug
	// printf("brokeriface: sending getData(%s), mapped from %s\n", (const char *)ClientVFS::thisInstance()->map(url), (const char *)url);
		
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
//		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}

	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	

	data=r.extraData;
	return true;
}

bool BrokerInterface::getObjectData(QStringList &urls, QByteArray &data)
{
// query check - unavailable
// TODO: query for a group of objects
	
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetObjectsData;
	f.extraData=_stream(ClientVFS::thisInstance()->map(urls));

	// debug
	// printf("brokeriface: sending getData(%s), mapped from %s\n", (const char *)ClientVFS::thisInstance()->map(url), (const char *)url);
		
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}

	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	

	data=r.extraData;
	return true;
}

bool BrokerInterface::getChildrenData(QString url, QByteArray &data)
{
// query check - unavailable
// TODO: query for a group of objects
	
	CommFrame f, r;
	QString serverUrl;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::GetChildrenData;
	serverUrl=ClientVFS::thisInstance()->map(url);
	printf("Server URL = %s !",serverUrl.latin1());
	fflush(stdout);
	if( serverUrl!=QString::null )
	  f.extraData=_stream(serverUrl);
	else
	{
	  printf("\nNo map entry for %s url.",(const char *)url);
	  return false;
	}

	// debug
	printf("brokeriface: sending getChildrenData(%s), mapped from %s\n", (const char *)ClientVFS::thisInstance()->map(url), (const char *)url);
		
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}

	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	

	data=r.extraData;
	return true;
}

QByteArray BrokerInterface::getSimpleData(QString url)
{
	// this is just a wrapper over getObjectData so that we can use _narrow directly
	// it makes sense only when we receive one data structure, such as a bool or QString
	QByteArray data;
	getObjectData(url, data);
	return data;
}

bool BrokerInterface::setObjectData(QString url, const QByteArray &data)
{
	// query check
	if(url.find("?")>-1)
	{
		bool res=QueryProcessor::thisInstance()->setData(url, data);
		err=QueryProcessor::thisInstance()->lastError();
		return res;
	}
	
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::SetData;
	QDataStream s(f.extraData, IO_WriteOnly);
	s<<ClientVFS::thisInstance()->map(url)<<data;
	// debug
	// printf("brokeriface: sending setData(%s) to %s, mapped from %s\n", (const char *)ClientVFS::thisInstance()->map(url), (const char *)_narrowQString(data), (const char *)url);
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return true;
}

QString BrokerInterface::createObject(QString parent_url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::CreateObject;
	f.extraData=_stream(ClientVFS::thisInstance()->map(parent_url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return QString::null;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return QString::null;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return QString::null;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return QString::null;
	}	
	
	return ClientVFS::thisInstance()->reverseMap(_narrowQString(r.extraData));
}

QString BrokerInterface::createFolder(QStringList &paramList)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::CreateFolder;
	f.extraData=_stream(paramList);

	// debug
	// printf("brokerinterface: sending CreateFolder(%s)\n", (const char *)_narrowQString(f.extraData));
		
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return QString::null;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return QString::null;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return QString::null;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return QString::null;
	}	
	
	return _narrowQString(r.extraData);
}

QString BrokerInterface::copy(QString url, QString dest_parent_url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::CopyUrl;
	QDataStream s(f.extraData, IO_WriteOnly);
	s<<ClientVFS::thisInstance()->map(url)<<ClientVFS::thisInstance()->map(dest_parent_url);
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return QString::null;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return QString::null;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return QString::null;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return QString::null;
	}	
	
	return ClientVFS::thisInstance()->reverseMap(_narrowQString(r.extraData));
}

QString BrokerInterface::move(QString url, QString dest_parent_url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::MoveUrl;
	QDataStream s(f.extraData, IO_WriteOnly);
	s<<ClientVFS::thisInstance()->map(url)<<ClientVFS::thisInstance()->map(dest_parent_url);
	
	// debug
	// printf("brokerinterface: trying to move %s->%s...\n", (const char *)ClientVFS::thisInstance()->map(url), (const char *)ClientVFS::thisInstance()->map(dest_parent_url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return QString::null;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return QString::null;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return QString::null;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return QString::null;
	}	
	
	return ClientVFS::thisInstance()->reverseMap(_narrowQString(r.extraData));
}

bool BrokerInterface::send(QString url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::SendUrl;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return true;
}

bool BrokerInterface::remove(QString url)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::RemoveUrl;
	f.extraData=_stream(ClientVFS::thisInstance()->map(url));
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return true;
}

bool BrokerInterface::removeFolder(QStringList &paramList)
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::RemoveFolder;
	f.extraData=_stream(paramList);
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return false;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return false;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return false;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return false;
	}	
	
	return true;
}

QString BrokerInterface::regetLastError()
{
	CommFrame f, r;
	f.majorRevNo=1;
	f.minorRevNo=1;
	f.requestFrame=true;
	f.clientFrame=true;
	f.requestsReply=true;
	f.commandCode=MagellanComm::RegetLastError;
	
	if(!CommBinder::thisInstance()->send("thisInstance", f, r))
	{
		printf("Warning: no link could be established with the server, CommBinder returned an error\n");
		return QString::null;
	}
	if(!r.hasStatus)
	{
		printf("Warning: no link could be established with the server, no status was retrieved\n");
		return QString::null;
	}
	if(r.statusCode!=StatusInfo::Ok)
	{
		printf("Warning: no link could be established with the server, an invalid status was received\n");
		return QString::null;
	}
	
	if(r.completionStatus!=StatusInfo::Completed_Yes)
	{
		return QString::null;
	}	
	
	return _narrowQString(r.extraData);
}

QString BrokerInterface::lastError()
{
	return err;
}





