/******************************************/
/* Wmufo - Mini SETI@Home monitor         */
/******************************************/

/******************************************/
/* This program is free software; you can redistribute it and/or
/* modify it under the terms of the GNU General Public License
/* as published by the Free Software Foundation; either version 2
/* of the License, or (at your option) any later version.
/* 
/* This program is distributed in the hope that it will be useful,
/* but WITHOUT ANY WARRANTY; without even the implied warranty of
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/* GNU General Public License for more details.
/* 
/* You should have received a copy of the GNU General Public License
/* along with this program; if not, write to the Free Software
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
/******************************************/

#include "seti.h"

/******************************************/
/* Read statistics                        */
/******************************************/

int
read_stats(const char *dir, seti_data *seti)
{
	seti->state = SETI_STATE_ERROR;

	seti->pid = 0;
	seti->status = 'U';
	seti->progress = 0;
	seti->cpuunit = 0;
	seti->cputotal = 0;
	seti->nwus = 0;
	seti->nresults = 0;

	if (get_pid(dir, seti))
		seti->status = get_status(seti->pid);

	read_workunit_stats(dir, seti);

	read_user_stats(dir, seti);

	return seti->state;
}

/******************************************/
/* Read client pid                        */
/******************************************/

pid_t
get_pid(const char *dir, seti_data *seti)
{
	char buf[128];
	FILE *fp;

	sprintf(buf, "%s/%s", dir, SETI_FILE_PID);

	if (fp = fopen(buf, "r")) {
		while (fgets(buf, sizeof (buf), fp)) {
			seti->pid = atoi(buf);
		}
		fclose(fp);
	} else if (debug)
		fprintf(stderr, "INF: Failed to open pid file - %s\n", buf);

	return seti->pid;
}

/******************************************/
/* Read status of process                 */
/******************************************/

char
get_status(const pid_t pid)
{
	glibtop_proc_state state;

	if (pid) {
		glibtop_get_proc_state(&state, pid);

		if (strstr(state.cmd, SETI_PROG))
			return state.state;
	}

	return 'U';
}

/******************************************/
/* Read workunit statistics               */
/******************************************/

void
read_workunit_stats(const char *dir, seti_data *seti)
{
	int search = 2;
	char buf[256];
	char *right;
	FILE *fp;

	sprintf(buf, "%s/%s", dir, SETI_FILE_STATE);

	if (fp = fopen(buf, "r")) {
		while (fgets(buf, sizeof (buf), fp) && search) {
			if (right = strchr(buf, '=')) {
				*right++ = '\0';
				if (!strcmp(buf, "cpu")) {
					seti->cpuunit = atof(right);
					search--;
				} else if (!strcmp(buf, "prog")) {
					seti->progress = atof(right);
					search--;
				}
			}
		}
		fclose(fp);
		seti->state = SETI_STATE_NORMAL;

		sprintf(buf, "%s/%s", dir, SETI_FILE_RESULT);
		if (fp = fopen(buf, "r")) {
			fclose(fp);
			seti->state = SETI_STATE_UPLOAD;
		}
	} else {
		sprintf(buf, "%s/%s", dir, SETI_FILE_WTEMP);
		if (fp = fopen(buf, "r")) {
			fclose(fp);
			seti->state = SETI_STATE_DOWNLOAD;
		} else {
			if (debug)
				fprintf(stderr, "INF: Failed to open workunit file - %s/%s\n", dir, SETI_FILE_STATE);
			seti->state = SETI_STATE_CONNECT;
		}
	}
}

/******************************************/
/* Read user statistics                   */
/******************************************/

void
read_user_stats(const char *dir, seti_data *seti)
{
	char buf[256];
	char *right;
	FILE *fp;

	sprintf(buf, "%s/%s", dir, SETI_FILE_USERINFO);

	if (fp = fopen(buf, "r")) {
		while (fgets(buf, sizeof (buf), fp)) {
			if (right = strchr(buf, '=')) {
				*right++ = '\0';
				if (!strcmp(buf, "nwus"))
					seti->nwus = atol(right);
				else if (!strcmp(buf, "nresults"))
					seti->nresults = atol(right);
				else if (!strcmp(buf, "total_cpu"))
					seti->cputotal = atol(right);
			}
		}
		fclose(fp);
	} else {
		if (debug)
			fprintf(stderr, "INF: Failed to open user file - %s\n", buf);
		seti->state = SETI_STATE_ERROR;
	}
}
