/* InputMethodEvent.java -- Events from an input method 
   Copyright (C) 1999 Free Software Foundation, Inc.

This file is part of the non-peer AWT libraries of GNU Classpath.

This library is free software; you can redistribute it and/or modify
it under the terms of the GNU Library General Public License as published 
by the Free Software Foundation, either version 2 of the License, or
(at your option) any later verion.

This library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Library General Public License for more details.

You should have received a copy of the GNU Library General Public License
along with this library; if not, write to the Free Software Foundation
Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307 USA. */


package java.awt.event;

import java.awt.Component;
import java.awt.font.TextHitInfo;
import java.text.AttributedCharacterIterator;

/**
  * This class is for event generated by change in a text input method.
  *
  * @author Aaron M. Renn (arenn@urbanophile.com)
  */
public class InputMethodEvent extends java.awt.AWTEvent
             implements java.io.Serializable
{

/*
 * Static Variables
 */

/**
  * This is the first id in the range of event ids used by this class.
  */
public static final int INPUT_METHOD_FIRST = 3434;

/**
  * This is the last id in the range of event ids used by this class.
  */
public static final int INPUT_METHOD_LAST = 3435;

/**
  * This event id indicates that the input method curor point has changed.
  */
public static final int CARET_POSITION_CHANGED = 3434;

/**
  * This event id indicates that the text in the input method has changed.
  */
public static final int INPUT_METHOD_TEXT_CHANGED = 3435;

/*************************************************************************/

/*
 * Instance Variables
 */

// The input method text
private transient AttributedCharacterIterator text;

// The number of committed characters in the text
private transient int committedCharacterCount;

// The caret
private transient TextHitInfo caret;

// The most important position to be visible
private transient TextHitInfo visiblePosition;

/*************************************************************************/

/*
 * Constructors
 */

/**
  * Initializes a new instance of <code>InputMethodEvent</code> with the
  * specified source, id, text, char count, caret, and visible position.
  *
  * @param source The source that generated the event.
  * @param id The event id.
  * @param text The input text.
  * @param committedCharacterCount The number of characters in the text
  * that have been committed.
  * @param caret The caret position.
  * @param visiblePosition The position in the text that is most important
  * to be visible.
  */
public 
InputMethodEvent(Component source, int id, AttributedCharacterIterator text,
                 int committedCharacterCount, TextHitInfo caret,
                 TextHitInfo visiblePosition)
{
  super(source, id);
  this.text = text;
  this.committedCharacterCount = committedCharacterCount;
  this.caret = caret;
  this.visiblePosition = visiblePosition;
}

/*************************************************************************/

/**
  * Initializes a new instance of <code>InputMethodEvent</code> with the
  * specified source, id, caret, and visible position.
  *
  * @param source The source that generated the event.
  * @param id The event id.
  * @param caret The caret position.
  * @param visiblePosition The position in the text that is most important
  * to be visible.
  */
public 
InputMethodEvent(Component source, int id, TextHitInfo caret, 
                 TextHitInfo visiblePosition)
{
  this(source, id, null, 0, caret, visiblePosition);
}

/*************************************************************************/

/*
 * Instance Methods
 */

/**
  * This method returns the input method text.  This can be <code>null</code>,
  * and will always be <code>null</code> for <code>CARET_POSITION_CHANGED</code>
  * events.
  *
  * @return The input method text.
  */
public AttributedCharacterIterator
getText()
{
  return(text);
}

/*************************************************************************/

/**
  * Returns the number of committed characters in the input method text.
  *
  * @return The number of committed characters in the input method text.
  */
public int
getCommittedCharacterCount()
{
  return(committedCharacterCount);
}

/*************************************************************************/

/**
  * Returns the caret position.
  *
  * @return The caret position.
  */
public TextHitInfo
getCaret()
{
  return(caret);
}

/*************************************************************************/

/**
  * Returns the position that is most important to be visible.
  *
  * @param The position that is most important to be visible.
  */
public TextHitInfo
getVisiblePosition()
{
  return(visiblePosition);
}

/*************************************************************************/

/**
  * This method consumes the event.  A consumed event is not processed 
  * in the default manner by the component that generated it.
  */
public void
consume()
{
  super.consume();
}

/*************************************************************************/

/**
  * This method test whether or not this event has been consumed.
  *
  * @return <code>true</code> if the event has been consumed,
  * <code>false</code> otherwise.
  */
public boolean
isConsumed()
{
  return(super.isConsumed());
}

/*************************************************************************/

/**
  * This method returns a string identifying the event.
  *
  * @return A string identifying the event.
  */
public String
paramString()
{
  return(getClass().getName() + " source=" + getSource() + " id=" + getID());
}

} // class InputMethodEvent

