/********************************************************************\
 * HelpWindow.c -- a help window for hypertext help.                *
 * Copyright (C) 1997 Robin D. Clark                                *
 * Copyright (C) 1998,1999 Linas Vepstas                            *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
 *                                                                  *
 *   Author: Rob Clark                                              *
 * Internet: rclark@cs.hmc.edu                                      *
 *  Address: 609 8th Street                                         *
 *           Huntington Beach, CA 92648-4632                        *
\********************************************************************/

#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/PanedW.h>
#include <Xm/Frame.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>

#include <guile/gh.h>

#include "config.h"

#if HAVE_XPM
#  include <X11/xpm.h>
#endif

#if HAVE_LIBXMHTML
#include <XmHTML/XmHTML.h>
#endif 

#include "File.h"
#include "HelpWindow.h"
#include "top-level.h"
#include "ui-callbacks.h"
#include "messages.h"
#include "Sheet.h"
#include "util.h"
#include "xtutil.h"

extern Widget toplevel;
static short module = MOD_HTML; 

/********************************************************************\
 *     HTML History functions                                       * 
 * hack alert -- these are gui-independent, and should be moved
 * to somewhere were the gtk, Qt gui's can make use of them
\********************************************************************/
typedef struct _HTMLHistory {
  struct _HTMLHistory *next;
  struct _HTMLHistory *last;
  char   *htmlfile;
  char   *text;
} HTMLHistory;

/* insert an htmlfile into history.  Return TRUE if this
 * is the first element in the history.  If not last element
 * in history, all next pages are deleted */
static int
historyInsert( HTMLHistory **history, const char *htmlfile )
  {
  ENTER ("historyInsert()\n");
  if( (*history) != NULL )
    {
    HTMLHistory *temp;
    
    /* delete all next pages: */
    temp = (*history)->next;
    while( temp ) 
      {
      (*history)->next = temp->next;
      if (temp->htmlfile) free(temp->htmlfile);
      if (temp->text) free(temp->text);
      _free(temp);

      temp = (*history)->next;
      }
    
    /* Add new node to history: */
    temp = (HTMLHistory *)_malloc(sizeof(HTMLHistory));
    if (htmlfile) {
       temp->htmlfile = strdup (htmlfile);
    } else {
       temp->htmlfile = NULL;
    }
    temp->text = NULL;
    temp->next = NULL;
    temp->last = (*history);
    (*history)->next = temp;
    (*history) = temp;
    
    LEAVE ("historyInsert(): have history\n");
    return FALSE;
    }
  else
    {
    /* This must be the first node in the history... */
    (*history) = (HTMLHistory *)_malloc(sizeof(HTMLHistory));
    if (htmlfile) {
       (*history)->htmlfile = strdup (htmlfile);
    } else {
       (*history)->htmlfile = NULL;
    }
    (*history)->text = NULL;
    (*history)->last = NULL;
    (*history)->next = NULL;
    
    /* ...so return TRUE */
    LEAVE ("historyInsert(): new history \n");
    return TRUE;
    }
  }

/* Move forward in history, and return current htmlfile */
static char *
historyFwd( HTMLHistory **history )
  {
  if( (*history) != NULL ) {
    if( (*history)->next != NULL ) { 
      (*history) = (*history)->next; 
       }
    return (*history)->htmlfile;
    }
  else
    return NULL;
  }

/* Move back in history, and return current htmlfile */
static char *
historyBack( HTMLHistory **history )
  {
  if( (*history) != NULL ) {
    if( (*history)->last != NULL ) {
      (*history) = (*history)->last;
      }
    return (*history)->htmlfile;
    }
  else
    return NULL;
  }

#ifdef NOT_USED
/* Return current htmlfile */
static char *
historyCurrent( HTMLHistory **history )
  {
  if( (*history) != NULL )
    return (*history)->htmlfile;
  else
    return NULL;
  }
#endif

/* Remove all entries from history: */
static void
historyClear( HTMLHistory **history )
  {
  /* move to beginning of history: */
  while( (*history)->last != NULL )
    (*history) = (*history)->last;
  
  /* delete all next pages: */
  while( (*history)->next != NULL )
    {
    HTMLHistory *temp = (*history)->next;
    
    (*history)->next = temp->next;
    if(temp->htmlfile) free(temp->htmlfile);
    if(temp->text) free(temp->text);
    _free(temp);
    }
  
  /* delete current page: */
  if ((*history)->htmlfile) free((*history)->htmlfile);
  if ((*history)->text) free((*history)->text);
  _free(*history);
  (*history) = NULL;
  }


/********************************************************************\
 *     HTML Window stuff...                                         * 
\********************************************************************/

/** GLOBALS *********************************************************/

struct _HTMLWindow {
   Widget htmlwidget;
   Widget data_target_frame;
   Widget menu_target_frame;
   short  installed_data_frame_form_cb;
   short  installed_menu_frame_form_cb;
   HTMLHistory *history;
};

typedef struct _HTMLWindow HTMLWindow;

HTMLWindow *helpwindow = NULL;
HTMLWindow *reportwindow = NULL;

/** PROTOTYPES ******************************************************/
static void   htmlWindow( Widget parent, HTMLWindow **hwinp,
                          const char * const title, 
                          const char * const htmlfile, 
                          const char * const text);
static void   closeHtmlWin( Widget mw, XtPointer cd, XtPointer cb );
static void   frameCB( Widget mw, XtPointer cd, XtPointer cb );
static void   formCB( Widget mw, XtPointer cd, XtPointer cb );
static void   htmlBackCB( Widget mw, XtPointer cd, XtPointer cb );
static void   htmlFwdCB( Widget mw, XtPointer cd, XtPointer cb );
static void   htmlAnchorCB( Widget mw, XtPointer cd, XtPointer cb );

#if HAVE_LIBXMHTML
static char * xaccJumpToLabel (Widget mw, const char * jumpfile, char * text);
static XmImageInfo * htmlReadImageProc (Widget w, String file);
#endif /* WANT_XMHTML */

/********************************************************************\
 * reportWindow                                                     * 
 *   opens up a report window, and displays html                    * 
 *                                                                  * 
 * Args:   parent   - the parent widget                             * 
 *         title    - the title of the window                       * 
 *         htmlfile - the file name of the help file to display     * 
 * Return: none                                                     * 
\********************************************************************/
void
reportWindow( Widget parent, const char *title, const char *file)
{
  if (!reportwindow) {
    reportwindow = (HTMLWindow *) malloc (sizeof (HTMLWindow));
    reportwindow->htmlwidget = 0;
    reportwindow->data_target_frame = 0;
    reportwindow->menu_target_frame = 0;
    reportwindow->installed_data_frame_form_cb = 0;
    reportwindow->installed_menu_frame_form_cb = 0;
    reportwindow->history = NULL;
  } 
 
  htmlWindow (parent, &reportwindow, title, file, NULL);
}
  
/********************************************************************\
 * helpWindow                                                       * 
 *   opens up a help window, and displays html                      * 
 *                                                                  * 
 * Args:   parent   - the parent widget                             * 
 *         title    - the title of the window                       * 
 *         htmlfile - the file name of the help file to display     * 
 * Return: none                                                     * 
\********************************************************************/
void
helpWindow( Widget parent, const char *title, const char *htmlfile )
{
  if (!helpwindow) {
    helpwindow = (HTMLWindow *) malloc (sizeof (HTMLWindow));
    helpwindow->htmlwidget = 0;
    helpwindow->data_target_frame = 0;
    helpwindow->menu_target_frame = 0;
    helpwindow->installed_data_frame_form_cb = 0;
    helpwindow->installed_menu_frame_form_cb = 0;
    helpwindow->history = NULL;
  } 
 
  htmlWindow (parent, &helpwindow, title, htmlfile, NULL);
}
  
/********************************************************************\
 * helpWindow                                                       * 
 *   opens up a help window, and displays html                      * 
 *                                                                  * 
 * Args:   parent   - the parent widget                             * 
 *         title    - the title of the window                       * 
 *         htmlfile - the file name of the help file to display     * 
 * Return: none                                                     * 
\********************************************************************/
static void
htmlWindow( Widget parent, 
            HTMLWindow **hwinp,
            const char * const title, 
            const char * const htmlfile,
            const char * const htmltext)
  {
  Widget dialog,
         pane,
         controlform,
         buttonform,
         widget;
  int position=0;
  HTMLWindow *hw = *hwinp;
  char * text=0x0;
  
  gnc_set_busy_cursor( parent );
  ENTER ("htmlWindow()\n");
  
  historyInsert (&(hw->history), htmlfile); 
  if (htmltext) text = strdup (htmltext);
  hw->history->text = text;

  /* if the help window isn't open, then open it... otherwise, load
   * new help page into current help window */
  if( 0 == hw->htmlwidget )
    {
    /* Create the dialog box... XmNdeleteResponse is set to
     * XmDESTROY so the dialog's memory is freed when it is closed */
    dialog = XtVaCreatePopupShell( "dialog", 
				   xmDialogShellWidgetClass, parent,
				   XmNdialogStyle,  XmDIALOG_APPLICATION_MODAL,
				   XmNtitle,        title,
				   XmNdeleteResponse, XmDESTROY,
				   XmNwidth,        600,
				   XmNheight,       500,
                                   XmNtransient,    FALSE,  /* allow window to be repositioned */

				   XmNminWidth,     500,
				   XmNminHeight,    200,
				   NULL );
    
    XtAddCallback( dialog, XmNdestroyCallback, closeHtmlWin, (XtPointer) hwinp);
    
    /* Create a PanedWindow Manager for the dialog box... the child 
     * of optiondialog the paned window is the parent of the two 
     * forms which comprise the two areas of the dialog box...
     * The sash is set to minimun size to make it invisible */
    pane = XtVaCreateWidget( "pane", 
			     xmPanedWindowWidgetClass, dialog,
			     XmNsashWidth,     1,
			     XmNsashHeight,    1,
			     XmNtraversalOn,   False,
			     NULL );
    
    /** CONTROLFORM ****************************************
     * Create a controlform for control area of dialog box */
    controlform = XtVaCreateWidget( "frame", 
				    xmFrameWidgetClass, pane,
				    NULL );
    
    hw->htmlwidget =
      XtVaCreateManagedWidget( "html",

#if HAVE_LIBXMHTML
                               xmHTMLWidgetClass,       controlform,
                               XmNanchorButtons,        False, 
                               XmNimageProc,            htmlReadImageProc,
#endif 
			       XmNtopAttachment,        XmATTACH_FORM,
			       XmNbottomAttachment,     XmATTACH_FORM,
			       XmNleftAttachment,       XmATTACH_FORM,
			       XmNrightAttachment,      XmATTACH_FORM,
			       XmNwidth,                600,
			       XmNheight,               500,
			       NULL );
    
#if HAVE_LIBXMHTML
    XtAddCallback( hw->htmlwidget, XmNactivateCallback, htmlAnchorCB, (XtPointer) hw);
    XtAddCallback( hw->htmlwidget, XmNframeCallback,    frameCB, (XtPointer) hw);
    XtAddCallback( hw->htmlwidget, XmNformCallback,     formCB, (XtPointer) hw);
#endif
    
    hw->data_target_frame = hw->htmlwidget;

    /** ACTIONFORM ********************************************
     * Create a Form buttonform for action area of dialog box */
    buttonform = XtVaCreateWidget( "buttonform", 
				   xmFormWidgetClass, pane,
				   XmNfractionBase,   7,
				   NULL );
    position=1;
    
    /* The "Back" button */
    widget = XtVaCreateManagedWidget( BACK_STR,
				      xmPushButtonWidgetClass, buttonform,
				      XmNtopAttachment,      XmATTACH_FORM,
				      XmNbottomAttachment,   XmATTACH_FORM,
				      XmNleftAttachment,     XmATTACH_POSITION,
				      XmNleftPosition,       position,
				      XmNrightAttachment,    XmATTACH_POSITION,
				      XmNrightPosition,      position+1,
				      XmNshowAsDefault,      True,
				      NULL );
    
    XtAddCallback( widget, XmNactivateCallback, htmlBackCB, (XtPointer) hw);
    
    /* The "Forward" button */
    position +=2;
    widget = XtVaCreateManagedWidget( FORWARD_STR,
				      xmPushButtonWidgetClass, buttonform,
				      XmNtopAttachment,      XmATTACH_FORM,
				      XmNbottomAttachment,   XmATTACH_FORM,
				      XmNleftAttachment,     XmATTACH_POSITION,
				      XmNleftPosition,       position,
				      XmNrightAttachment,    XmATTACH_POSITION,
				      XmNrightPosition,      position+1,
				      XmNshowAsDefault,      True,
				      NULL );
    
    XtAddCallback( widget, XmNactivateCallback, htmlFwdCB, (XtPointer) hw);
    
    /* The "Close" button */
    position +=2;
    widget = XtVaCreateManagedWidget( CLOSE_STR,
				      xmPushButtonWidgetClass, buttonform,
				      XmNtopAttachment,      XmATTACH_FORM,
				      XmNbottomAttachment,   XmATTACH_FORM,
				      XmNleftAttachment,     XmATTACH_POSITION,
				      XmNleftPosition,       position,
				      XmNrightAttachment,    XmATTACH_POSITION,
				      XmNrightPosition,      position+1,
				      XmNshowAsDefault,      True,
				      NULL );
    
    XtAddCallback( widget, XmNactivateCallback, destroyShellCB, dialog );
    
    XtManageChild( hw->htmlwidget );
    
#if HAVE_LIBXMHTML
    /* HTML Widget seems happier if it get text before its fully managed. */
    text = xaccJumpToLabel( hw->htmlwidget, htmlfile, text);
    hw->history->text = text;
#endif
    /* Fix action area of the pane to its current size, and not let it
     *  resize. */
    XtManageChild( buttonform );
    
    {
    Dimension h;
    XtVaGetValues( widget, XmNheight, &h, NULL );
    XtVaSetValues( buttonform, XmNpaneMaximum, h, XmNpaneMinimum, h, NULL );
    }
    
    XtManageChild( controlform );
    XtManageChild( pane );
    XtPopup( dialog, XtGrabNone );
    }
  else 
    {
#if HAVE_LIBXMHTML
    text = xaccJumpToLabel( hw->htmlwidget, htmlfile, text);
    hw->history->text = text;
#endif
    }
  
  gnc_unset_busy_cursor( parent );
  LEAVE ("htmlWindow()\n");
  }

/********************************************************************\
 *     callback functions...                                        * 
\********************************************************************/

/* sample usage for data target frame: 
 * <frame name="report-target" src="xxx.html">
 * design hack alert -- to be general and flexible and less of a hack,
 * we should be keeping track of the named frame elements, and pushing
 * them onto a linked list.  Then, the URL that gets clicked on 
 * should contain the name of the target frame that we want to draw
 * draw to ... and we could look up that name in our list.
 */
#define DATA_TARGET "report-target"
#define MENU_TARGET "menu-target"

#if HAVE_LIBXMHTML
static void
frameCB( Widget mw, XtPointer cd, XtPointer cb )
{
  XmHTMLFrameCallbackStruct *fcs = (XmHTMLFrameCallbackStruct *) cb;
  HTMLWindow *hw = (HTMLWindow *) cd;
  ENTER ("frameCB()\n");
  
  if (XmCR_HTML_FRAMECREATE == fcs->reason) { 
     fcs->doit = True;
  } else
  if (XmCR_HTML_FRAME == fcs->reason) { 
     XtVaSetValues (fcs->html, XmNimageProc, htmlReadImageProc,  NULL);
     XtAddCallback (fcs->html, XmNactivateCallback, htmlAnchorCB, cd);
     xaccJumpToLabel (fcs->html, fcs->src, NULL); 

     /* keep track of the frame where we will be placing the reports. */
     if (!strcmp (fcs->name, DATA_TARGET)) {
        hw->data_target_frame = fcs->html;
        if (0 == hw->installed_data_frame_form_cb) {
           hw->installed_data_frame_form_cb = 1;
           XtAddCallback( hw->data_target_frame, 
               XmNformCallback, formCB, (XtPointer) hw);
        }
     } else 
     if (!strcmp (fcs->name, MENU_TARGET)) {
        hw->menu_target_frame = fcs->html;
        if (0 == hw->installed_menu_frame_form_cb) {
           hw->installed_menu_frame_form_cb = 1;
           XtAddCallback( hw->menu_target_frame, 
               XmNformCallback, formCB, (XtPointer) hw);
        }
     }
  }
  LEAVE ("frameCB()\n");
}

/********************************************************************\
\********************************************************************/
static void
formCB( Widget mw, XtPointer cd, XtPointer cb )
{
  int i,n;
  XmHTMLFormCallbackStruct *fcs = (XmHTMLFormCallbackStruct *) cb;
  // HTMLWindow *hw = (HTMLWindow *) cd;
  
   PINFO ("formCB(): action=%s\n", fcs->action);
   PINFO ("formCB(): encoding=%s\n", fcs->enctype);
   PINFO ("formCB(): num keys=%d \n", fcs->ncomponents);

   DEBUGCMD ({
      n = fcs->ncomponents;
      for (i=0; i<n; i++) {
         DEBUG ("formCB(): %s=%s\n", 
             fcs->components[i].name, fcs->components[i].value);
      }
   });
}
#endif

/********************************************************************\
 * htmlBackCB - called when user clicks "Back" button... shows last * 
 *   help page in history                                           * 
 *                                                                  * 
 * Args:   mw -                                                     * 
 *         cd -                                                     * 
 *         cb -                                                     * 
 * Return: none                                                     * 
\********************************************************************/
static void
htmlBackCB( Widget mw, XtPointer cd, XtPointer cb )
  {
  HTMLWindow *hw = (HTMLWindow *) cd;
  char *file = historyBack(&(hw->history));
  char *text = hw->history->text;
#if HAVE_LIBXMHTML
  text = xaccJumpToLabel( hw->htmlwidget, file, text);
  hw->history->text = text;
#endif
  }

/********************************************************************\
 * htmlFwdCB - called when user clicks "Forward" button... shows    * 
 *   next help page in the history                                  * 
 *                                                                  * 
 * Args:   mw -                                                     * 
 *         cd -                                                     * 
 *         cb -                                                     * 
 * Return: none                                                     * 
\********************************************************************/
static void
htmlFwdCB( Widget mw, XtPointer cd, XtPointer cb )
  {
  HTMLWindow *hw = (HTMLWindow *) cd;
  char *file = historyFwd(&(hw->history));
  char *text = hw->history->text;
#if HAVE_LIBXMHTML
  text = xaccJumpToLabel( hw->htmlwidget, file, text);
  hw->history->text = text;
#endif
  }

/********************************************************************\
 * closeHtmlWin - called when the help window is closed             * 
 *                                                                  * 
 * Args:   mw -                                                     * 
 *         cd -                                                     * 
 *         cb -                                                     * 
 * Return: none                                                     * 
\********************************************************************/
static void
closeHtmlWin( Widget mw, XtPointer cd, XtPointer cb )
  {
  HTMLWindow **hw = (HTMLWindow **) cd;
  /* Delete the history: */
  historyClear (&((*hw)->history));
  (*hw)->history=NULL;
  (*hw)->htmlwidget=0;
  free (*hw);
  (*hw) = NULL;
  }

/********************************************************************\
 * htmlAnchorCB - called when user clicks on html anchor tag        * 
 *                                                                  * 
 * Args:   mw - the html widget that called us                      * 
 *         cd -                                                     * 
 *         cb - the anchor call-back struct                         * 
 * Return: none                                                     * 
\********************************************************************/
static void
htmlAnchorCB( Widget mw, XtPointer cd, XtPointer cb )
  {
  HTMLWindow *hw = (HTMLWindow *) cd;

#if HAVE_LIBXMHTML
   XmHTMLAnchorCallbackStruct *acbs = (XmHTMLAnchorCallbackStruct *) cb;

   if(acbs->reason != XmCR_ACTIVATE) return;

   switch(acbs->url_type) {

      /* a named anchor on a page that is already displayed */
      case ANCHOR_JUMP: {
         XmHTMLAnchorScrollToName(mw, acbs->href);
      }
      break;

      /* a local file with a possible jump to label */
      case ANCHOR_FILE_LOCAL: {
         historyInsert(&(hw->history), acbs->href);
/* hack alert -- the target widget thing is a hack */
         hw->history->text = xaccJumpToLabel (hw->data_target_frame, acbs->href, NULL);
      }
      break;

      /*  other types are unsupported, but it would be fun if they were ... */
      case ANCHOR_FTP:
         PERR ("this help window doesn't support ftp: %s\n", acbs->href);
         break;
      case ANCHOR_HTTP:
         PERR ("this help window doesn't support http: %s\n", acbs->href);
         break;
      case ANCHOR_MAILTO:
         PERR("this help window doesn't support email: %s\n", acbs->href);
         break;
      case ANCHOR_UNKNOWN:
      default:
         PERR("don't know this type of url: %s\n", acbs->href);
         break;
   }

#endif
  }

/********************************************************************\
 *     utility functions...                                         * 
\********************************************************************/

#if HAVE_LIBXMHTML
static char * 
xaccJumpToLabel (Widget mw, const char * jumpfile, char * text)
{
   char *label=0x0, *file=0x0;

   if (jumpfile) {
      file = strdup (jumpfile);
   
      /*  see if this anchor contains a jump */
      label = strpbrk (file, "#?");
      if (label) {
         file [label - file] = 0x0;  /* truncate # from name */
      }
   
      /* see if the anchor is an "active gnucash page" */
      if (strstr (file, ".phtml")) {
        text = gncReport (file);
      }

      /* if text to display wasn't specified, use the truncated name to read */
      if (!text) text = gncReadFile (file);
   }
   if (!text) return NULL;

   XmHTMLTextSetString(mw, text);
   if (label) {
      XmHTMLAnchorScrollToName(mw, label);
   } else {
      XmHTMLTextScrollToLine(mw, 0);
   }
   if (file) free (file);

   return text;
}

/********************************************************************\
 *     HTML functions...                                            * 
\********************************************************************/
/********************************************************************\
 * htmlReadImageProc                                                * 
 *                                                                  * 
 * Args:   file - the name of the html file to read                 * 
 * Return: none                                                     * 
 * Global: helpPath - the path to the help files                    * 
\********************************************************************/
static XmImageInfo *
htmlReadImageProc (Widget w, String file) 
  {
  char  *filename;
  XmImageInfo *retval;

  /* construct absolute path -- twiddle the relative path we recieved */  
  filename = gncFindFile (file);
  
  /* use the default proc for the hard work */
  retval = XmHTMLImageDefaultProc(w, filename, NULL, 0);

  free(filename);
  return retval;
}
#endif /* WANT_XMHTML */

#if USE_HTMLW
#if HAVE_XPM
/********************************************************************\
 * htmlResolveImage                                                 * 
 *                                                                  * 
 * Args:   mw   - the html widge                                    *
 *         file - the name of the html file to read                 * 
 *         nl   - ???                                               * 
 * Return: none                                                     * 
 * Global: helpPath - the path to the help files                    * 
\********************************************************************/

struct _image_info {
  int width;
  int height;
  int num_colors;
  Pixmap pixmap;
};

typedef struct _image_info ImageInfo;

static ImageInfo *
htmlResolveImage( Widget mw, char *file, int nl )
  {
  ImageInfo *img = (ImageInfo *)malloc(sizeof(ImageInfo));
  XpmImage  xpm;
  XpmAttributes attr;
  int err;
  char *filename;
  
  /* construct absolute path -- twiddle the relative path we recieved */
  filename = gncFindFile (file);
  
  /* initialize stuff: */
  memset( img, 0, sizeof(ImageInfo) );
  memset( &attr, 0, sizeof(XpmAttributes) );
  memset( &xpm, 0, sizeof(XpmImage) );
  
  err = XpmReadFileToXpmImage( filename, &xpm, NULL );

  free(filename); filename = NULL;

  img->width  = xpm.width;
  img->height = xpm.height;
  img->num_colors = xpm.ncolors;
  img->pixmap = None;
  
  err = XpmCreatePixmapFromXpmImage( XtDisplay(mw), XtWindow(mw), &xpm,
				     &(img->pixmap), NULL, NULL );
  if( err != 0 )
    ERROR();
  
  return img;
  }

#endif /* HAVE_XPM */
#endif /* USE_HTMLW */
/* ----------------------- END OF FILE ---------------------  */
