/********************************************************************\
 * FileBox.c -- the file dialog box                                 *
 * Copyright (C) 1997 Robin D. Clark                                *
 * Copyright (C) 1998 Linas Vepstas                                 *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
 *                                                                  *
 *   Author: Rob Clark                                              *
 * Internet: rclark@cs.hmc.edu                                      *
 *  Address: 609 8th Street                                         *
 *           Huntington Beach, CA 92648-4632                        *
\********************************************************************/

#include <stdio.h>
#include <Xm/Xm.h>
#include <Xm/FileSB.h>

#include "config.h"

#include "FileBox.h"
#include "messages.h"
#include "util.h"
#include "xtutil.h"
#include "top-level.h"

/** GLOBALS *********************************************************/
extern XtAppContext app;
Boolean done=True;
char * previous_path = NULL;

/* This static indicates the debugging module that this .o belongs to.  */
static short module = MOD_GUI;

/** PROTOTYPES ******************************************************/
void fileBoxCB( Widget mw, XtPointer cd, XtPointer cb );
void closeBoxCB( Widget mw, XtPointer cd, XtPointer cb );

/* hack alert -- 
 * this filebox tries to remembe the last dir it worked in.
 * in fact, this should probably be (1) overriden by the app, and
 * (2) the app should remember this dir accross sessions.
 */

/********************************************************************\
 * fileBox                                                          * 
 *   pops up a file selection dialog (either a "Save As" or an      * 
 *   "Open"), and returns the name of the file the users selected.  * 
 *   (This function does not return until the user selects a file   * 
 *   or pressed "Cancel")                                           * 
 *                                                                  * 
 *   NOTE: fileBox is not re-entrant... if an instance of fileBox   * 
 *         already exists, the latter call will return NULL         * 
 *                                                                  * 
 * Args:   parent  - the parent of this window                      *
 *         type    - either OPEN or SAVE                            * 
 * Return: none                                                     * 
 * Global: app     - the XtAppContext                               * 
 *         done    - whether fileBox should return                  * 
\********************************************************************/
char *
fileBox( const char * title, const char * filter)
  {
  Widget   dialog=0;
  char*    fileName = NULL;
  XmString filterpattern, dialogname=NULL;
  Widget parent = gnc_get_ui_data();
  
  if( !done )
    return NULL;                   /* Don't open if there already is
				    * an instance of fileBox */
  if (!filter) {
    filterpattern = XmStringCreateSimple( "*.xac" );
  } else {
    char *tmp = strdup(filter);
    filterpattern = XmStringCreateSimple( tmp );
    free(tmp);
  }
  
  done = False;
  
  ENTER("fileBox");

  /* gnc_set_busy_cursor( parent ); */
  
  if (NULL == title) 
    {
    title = OPEN_STR;
    }
 {
   char *tmp = strdup(title);   
   dialogname = XmStringCreateSimple( tmp );
   dialog = XmCreateFileSelectionDialog( parent, tmp, NULL, 0 );
   free(tmp);
 }
   
  XtVaSetValues( dialog,
		 XmNpattern,      filterpattern, 
		 XmNdialogTitle,  dialogname,
                 /* lesstif 0.85 can't hack this pathMode stuff */
                 /* XmNpathMode,     XmPATH_MODE_RELATIVE, */
                 /* XmNpathMode,     XmPATH_MODE_FULL, */
		 NULL );
  XmStringFree( dialogname );
  /* XXX hack alert this results in a memory leak but fixes a core dump.
   * go figure it out someday.  */
  /* XmStringFree( filterpattern ); --electric fence flagged as error */
  
  /* initial directory is same as previous directory */
  if (previous_path) {
     XmString dirpath;
     dirpath = XmStringCreateSimple( previous_path );
     XtVaSetValues( dialog,
		 XmNdirectory,      dirpath, 
		 NULL );
     /* XXX hack alert this results in a memory leak but fixes a core dump.
      * go figure it out someday.  */
     /* XmStringFree( dirpath ); --electric fence flagged as error */ 
  }
  
  
  XtUnmanageChild( XmFileSelectionBoxGetChild(dialog,XmDIALOG_HELP_BUTTON) );
  
  /* Add the "ok" button callbacks... first save the name, then
   * destroy the dialog */
  XtAddCallback( dialog, XmNokCallback, 
		 fileBoxCB, (XtPointer)&fileName );
  XtAddCallback( dialog, XmNokCallback, 
		 destroyShellCB, (XtPointer)dialog );
  
  /* Add the cancel button callback */
  XtAddCallback( dialog, XmNcancelCallback, 
		 fileBoxCB, (XtPointer)NULL );
  XtAddCallback( dialog, XmNcancelCallback, 
		 destroyShellCB, (XtPointer)dialog );
  
  /* if user unmaps by using the window maanger, should be same as cancel */
  XtAddCallback( dialog, XmNunmapCallback, 
		 closeBoxCB, (XtPointer)&fileName );
  XtAddCallback( dialog, XmNunmapCallback, 
		 destroyShellCB, (XtPointer)dialog );

  XtManageChild( dialog );

  /* gnc_unset_busy_cursor( parent ); */
  /* while the user hasn't pushed "Ok", simulate XtMainLoop.*/
  while( !done || XtAppPending(app) )
    XtAppProcessEvent( app, XtIMAll );
  
  /* save the last directory that the user worked in. 
   * we will offer up this same directory to the user next 
   * time they get to this box.
   */
  if (fileName) {
    char *slash;
    if (previous_path) free (previous_path);
    previous_path = strdup (fileName);
    slash = strrchr (previous_path, '/');
    if (slash) { *(slash+1) = 0x0; }
  }

  LEAVE("fileBox");
  return fileName;
  }

/********************************************************************\
 * fileBoxCB                                                        * 
 *   callback that saves the name of the file so that fileBox       * 
 *   can return                                                     * 
 *                                                                  * 
 * Args:   mw - the widget that called us                           * 
 *         cd - fileName                                            * 
 *         cb -                                                     * 
 * Return: none                                                     * 
 * Global: done    - whether fileBox should return                  * 
\********************************************************************/
void
fileBoxCB( Widget mw, XtPointer cd, XtPointer cb )
  {
  char **fileName = (char **)cd;
  ENTER("fileBoxCB");
  
  if( cd != NULL )
    {
    XmFileSelectionBoxCallbackStruct *cbs = 
    (XmFileSelectionBoxCallbackStruct *)cb;
    
    if( !XmStringGetLtoR(cbs->value,charset,fileName) )
      *fileName = NULL;
    }
  done = True;
  LEAVE("fileBoxCB");
  }

void
closeBoxCB( Widget mw, XtPointer cd, XtPointer cb )
  {
  char **fileName = (char **)cd;
  ENTER("closeBoxCB");
  if (! done) {
    *fileName = NULL;
    done = True;
  }
  LEAVE("closeBoxCB");
  }

/* ======================== END OF FILE ======================== */
