;;; liece.el --- IRC client for Emacsen
;; Copyright (C) 1998, 1999 Daiki Ueno

;; Author: Daiki Ueno <ueno@ueda.info.waseda.ac.jp>
;; Created: 1998-09-28
;; Revised: 1999-01-31
;; Keywords: IRC, liece

;; This file is part of Liece.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.


;;; Commentary:
;; 

;;; Code:

(require 'liece-inlines)
(require 'liece-crypt)
(require 'liece-handle)
(require 'liece-filter)
(require 'liece-hilit)
(require 'liece-intl)
(require 'liece-menu)
(require 'liece-window)
(require 'liece-tcp)
(if running-xemacs
    (require 'liece-xemacs)
  (require 'liece-emacs))
(require 'liece-commands)

(autoload 'mule-caesar-region "mule-caesar" nil t)
(autoload 'liece-command-browse-url "liece-url" nil t)
(autoload 'liece-command-dcc-send "liece-dcc" nil t)
(autoload 'liece-command-dcc-receive "liece-dcc" nil t)
(autoload 'liece-command-dcc-list "liece-dcc" nil t)
(autoload 'liece-command-dcc-chat-listen "liece-dcc" nil t)
(autoload 'liece-command-dcc-chat-connect "liece-dcc" nil t)
(autoload 'liece-command-dcc-accept "liece-dcc" nil t)
(autoload 'liece-command-mail-compose "liece-mail" nil t)
(autoload 'liece-command-submit-bug-report "liece-mail" nil t)

(eval-and-compile
  (defvar liece-server-keyword-map
    '((:host (getenv "IRCSERVER"))
      (:service liece-service)
      (:password liece-password)
      (:prescript)
      (:prescript-delay)
      (:type)
      (:relay))
    "Mapping from keywords to default values.
All keywords that can be used must be listed here."))

(defadvice save-buffers-kill-emacs
  (before liece-save-buffers-kill-emacs activate)
  "Prompt user to quit IRC explicitly."
  (run-hooks 'liece-before-kill-emacs-hook) )

(add-hook 'liece-before-kill-emacs-hook 'liece-command-quit)

(defvar liece-tmp-server-name nil "Temporaly server name.")
(defvar liece-last-checkbuffer-time nil "Last time buffers were checked.")
(defvar liece-timers-list-initialized-p nil
  "Are liece internal timers in place?")

(defconst liece-obarray-size 1327
  "The size of obarray used by liece on channelname and username space.
For efficiency this should be prime.  See documentation of intern and
`make-vector' for more information.  Here is a list of some small primes...

13, 29, 37, 47, 59, 71, 89, 107, 131, 163, 197, 239, 293, 353, 431, 521,
631, 761, 919, 1103, 1327, 1597, 1931, 2333, 2801, 3371, 4049, 4861, 5839,
7013, 8419, 10103, 12143, 14591, 17519, 21023, 25229, 30293, 36353,
43627, 52361, 62851, 75431, 90523, 108631, 130363, 156437, 187751,
225307, 270371, 324449, 389357, 467237, 560689, 672827, 807403, 968897,
1162687, 1395263, 1674319, 2009191, 2411033, 2893249.")

(defvar liece-command-mode-map nil)
(defvar liece-dialogue-mode-map nil)
(defvar liece-channel-list-mode-map nil)
(defvar liece-nick-mode-map nil)

(defvar liece-command-map nil)
(defvar liece-client-query-map nil)
(defvar liece-dcc-map nil)
(defvar liece-crypt-map nil)
(defvar liece-friends-map nil)

(defvar liece-command-mode-syntax-table nil)

(put 'liece-command-mode 'mode-class 'special)
(put 'liece-dialogue-mode 'mode-class 'special)
(put 'liece-channel-list-mode 'mode-class 'special)
(put 'liece-nick-mode 'mode-class 'special)
(put 'liece-channel-mode 'derived-mode-parent 'liece-dialogue-mode)
(put 'liece-others-mode 'derived-mode-parent 'liece-dialogue-mode)

(defvar liece-buffer-mode-alist
  '((liece-dialogue-buffer liece-dialogue-mode)
    (liece-others-buffer liece-others-mode)
    (liece-channel-list-buffer liece-channel-list-mode)
    (liece-private-buffer liece-dialogue-mode)
    (liece-KILLS-buffer)
    (liece-IGNORED-buffer)
    (liece-WALLOPS-buffer)
    (liece-CRYPT-buffer liece-dialogue-mode)))
    
(defvar liece-client-query-keys
  '(
    ("a" liece-command-client-action)
    ("v" liece-command-client-version)
    ("u" liece-command-client-userinfo)
    ("h" liece-command-client-help)
    ("c" liece-command-client-clientinfo)
    ("g" liece-command-client-generic)
    ("p" liece-command-client-ping)
    ("t" liece-command-client-time)
    ("x" liece-command-client-x-face)
    ("X" liece-command-client-x-face-from-xbm-file)
    ("\C-x" liece-command-client-x-face-from-commandbuffer)
    ("U" liece-command-client-userinfo-from-minibuffer)
    ("\C-u" liece-command-client-userinfo-from-commandbuffer)
    )
  "Key definition table for client-query-map.")
  
(defvar liece-dialogue-keys
  '(
    ("\177" scroll-down)
    ([delete] scroll-down)
    ([backspace] scroll-down)
    (" " scroll-up)
    ("$" end-of-buffer)
    ("/" liece-command-generic)
    (">" end-of-buffer)
    ("<" beginning-of-buffer)
    ("!" liece-command-exec)
    ("|" liece-command-show-last-kill)
    ("%" liece-crypt-prefix)
    ("a" liece-command-away)
    ("b" liece-command-submit-bug-report)
    ("c" liece-command-point-back-to-command-buffer)
    ("f" liece-command-finger)
    ("F" liece-dialogue-freeze)
    ("M" liece-dialogue-own-freeze)
    ("i" liece-command-invite)
    ("j" liece-command-join)
    ("k" liece-command-kill)
    ("\C-k" liece-command-kick)
    ("\C-l" liece-command-list)
    ("l" liece-command-load-vars)
    ("s" liece-command-save-vars)
    ("m" liece-dialogue-enter-message)
    ("\C-m" liece-command-modec)
    ("n" liece-command-nickname)
    ("o" other-window)
    ("p" liece-command-mta-private)
    ("P" liece-command-toggle-private)
    ("q" liece-command-quit)
    ("r" liece-command-reconfigure-windows)
    ("x" liece-command-tag-region)
    ("t" liece-command-topic)
    ("T" liece-command-timestamp)
    ("\C-t" liece-command-find-timestamp)
    ("u" liece-command-lusers)
    ("U" liece-command-userhost)
    ("v" liece-command-browse-url)
    ("w" liece-command-who)
    )
  "Key definition table for dialogue mode.")

(eval-and-compile
  (dotimes (n 20)
    (fset (intern (format "liece-switch-to-channel-no-%d" (1+ n)))
	  `(lambda ()
	     (interactive)
	     (funcall (function liece-switch-to-channel-no) ,n)))
    ))

(defvar liece-select-keys
  '(
    ("1" liece-switch-to-channel-no-1)
    ("2" liece-switch-to-channel-no-2)
    ("3" liece-switch-to-channel-no-3)
    ("4" liece-switch-to-channel-no-4)
    ("5" liece-switch-to-channel-no-5)
    ("6" liece-switch-to-channel-no-6)
    ("7" liece-switch-to-channel-no-7)
    ("8" liece-switch-to-channel-no-8)
    ("9" liece-switch-to-channel-no-9)
    ("0" liece-switch-to-channel-no-10)
    ("\C-c1" liece-switch-to-channel-no-11)
    ("\C-c2" liece-switch-to-channel-no-12)
    ("\C-c3" liece-switch-to-channel-no-13)
    ("\C-c4" liece-switch-to-channel-no-14)
    ("\C-c5" liece-switch-to-channel-no-15)
    ("\C-c6" liece-switch-to-channel-no-16)
    ("\C-c7" liece-switch-to-channel-no-17)
    ("\C-c8" liece-switch-to-channel-no-18)
    ("\C-c9" liece-switch-to-channel-no-19)
    ("\C-c0" liece-switch-to-channel-no-20)
    )
  "Key definition table for select channels.")
    
(defvar liece-crypt-keys
  '(
    ("t" liece-command-toggle-crypt)
    ("k" liece-command-set-encryption-key)
    ("a" liece-command-add-decryption-key)
    ("d" liece-command-delete-decryption-key)
    )
  "Key definition table for crypt keys (dialogue and command mode).")

(defvar liece-dcc-keys
  '(
    ("s" liece-command-dcc-send)
    ("r" liece-command-dcc-receive)
    ("l" liece-command-dcc-list)
    ("cl" liece-command-dcc-chat-listen)
    ("cc" liece-command-dcc-chat-connect)
    ("g" liece-command-dcc-accept)
    )
  "Key definition table for DCC keys.")

(defvar liece-friends-keys
  '(
    (" " liece-command-ison)
    ("a" liece-command-activate-friends)
    ("d" liece-command-deactivate-friends)
    ("s" liece-command-display-friends)
    )
  "Key definition table for friends functions.")

(defvar liece-overriding-command-keys
  '(
    ("\177" liece-command-scroll-down)
    ([delete] liece-command-scroll-down)
    ([backspace] liece-command-scroll-down)
    (" " liece-command-scroll-up)
    ("$" liece-command-end-of-buffer)
    (">" liece-command-next-channel)
    ("<" liece-command-previous-channel)
    ("%" liece-crypt-prefix)
    ("a" liece-command-away)
    ("c" liece-command-inline)
    ("C" liece-channel-list-prefix)
    ("\C-a" liece-command-previous-channel)
    ("\C-c" liece-client-query-prefix)
    ("\C-d" liece-dcc-prefix)
    ("\C-f" liece-command-freeze)
    ("\C-i" liece-friends-prefix)
    ("\C-j" liece-command-next-channel)
    ("\C-n" liece-command-names)
    ("l" liece-command-list)
    ("L" liece-command-load-vars)
    ("M" liece-command-own-freeze)
    ("N" liece-nick-prefix)
    ("o" liece-command-mode+o)
    ("O" liece-command-toggle-nick-buffer-mode)
    ("\C-o" liece-command-toggle-channel-buffer-mode)
    ("\C-p" liece-command-part)
    ("r" liece-command-reconfigure-windows)
    ("\C-r" mule-caesar-region)
    ("s" liece-command-set-window-style)
    ("S" liece-command-save-vars)
    ("v" liece-command-mode+v)
    ("\C-v" liece-command-browse-url)
    ("\C-y" liece-command-yank-send)
    )
  "Key definition table for command mode which overrides dialogue keys.")

(defvar liece-command-keys
  '(
    ("\C-c" liece-command-prefix)
    ("\r" liece-command-enter-message)
    ([(meta return)] liece-command-enter-message-opposite-crypt-mode)
    ([tab] liece-command-complete)
    ([(meta control c) >] liece-command-push)
    ([(meta control c) <] liece-command-pop)
    ([(meta control c) o] liece-command-mode+o)
    ([(meta control c) O] liece-command-mode-o)
    ([(meta control c) v] liece-command-mode+v)
    ([(meta control c) V] liece-command-mode-v)
    )
  "Key definition table for command mode.")

(defvar liece-nick-keys
  '(
    ("o" liece-command-mode+o)
    ("O" liece-command-mode-o)
    ("v" liece-command-mode+v)
    ("V" liece-command-mode-v)
    ("f" liece-command-finger)
    (" " liece-command-nick-scroll-up)
    ("\177" liece-command-nick-scroll-down)
    ([delete] liece-command-nick-scroll-down)
    ([backspace] liece-command-nick-scroll-down)
    ("m" liece-command-mail-compose)
    ("c" liece-command-point-back-to-command-buffer)
    )
  "Key definition table for nick mode.")

(defvar liece-channel-list-keys
  '(
    (">" liece-command-next-channel)
    ("<" liece-command-previous-channel)
    ("o" other-window)
    ("c" liece-command-point-back-to-command-buffer)
    )
  "Key definition table for channel list mode.")

(defmacro liece-define-keys (map keys)
  (cons 'progn
	(mapcar (lambda (keydef)
		  `(define-key ,map ,(car keydef) ',(cadr keydef)))
		(eval keys))))

(put 'liece-define-keys 'lisp-indent-function 'defun)

(unless liece-client-query-map
  (define-prefix-command 'liece-client-query-map)
  (setq liece-client-query-map (make-keymap))
  (liece-define-keys liece-client-query-map liece-client-query-keys)
  (fset 'liece-client-query-prefix liece-client-query-map)
  )

(unless liece-dcc-map
  (define-prefix-command 'liece-dcc-map)
  (setq liece-dcc-map (make-keymap))
  (liece-define-keys liece-dcc-map liece-dcc-keys)
  (fset 'liece-dcc-prefix liece-dcc-map)
  )

(unless liece-crypt-map
  (define-prefix-command 'liece-crypt-map)
  (setq liece-crypt-map (make-keymap))
  (liece-define-keys liece-crypt-map liece-crypt-keys)
  (fset 'liece-crypt-prefix liece-crypt-map)
  )

(unless liece-command-map
  (define-prefix-command 'liece-command-map)
  (setq liece-command-map (make-keymap))
  (liece-define-keys liece-command-map liece-dialogue-keys)
  (liece-define-keys liece-command-map liece-overriding-command-keys)
  (fset 'liece-command-prefix liece-command-map)
  )

(unless liece-friends-map
  (define-prefix-command 'liece-friends-map)
  (setq liece-friends-map (make-keymap))
  (liece-define-keys liece-friends-map liece-friends-keys)
  (fset 'liece-friends-prefix liece-friends-map)
  )

(unless liece-dialogue-mode-map
  (setq liece-dialogue-mode-map (make-keymap))
  (suppress-keymap liece-dialogue-mode-map)
  (liece-define-keys liece-dialogue-mode-map liece-dialogue-keys)
  (liece-define-keys liece-dialogue-mode-map liece-select-keys)
  )

(unless liece-command-mode-map
  (setq liece-command-mode-map (make-sparse-keymap))
  (liece-define-keys liece-command-mode-map liece-command-keys)
  (liece-define-keys liece-command-mode-map
    (mapcar
     (lambda (key)
       (cons (concat "\C-c" (car key)) (cdr key)))
     liece-select-keys))
  (define-key liece-command-mode-map "\C-c%" liece-crypt-map)
  (if liece-want-traditional
      (define-key liece-command-mode-map "/"
	'liece-command-irc-compatible))
  )

(unless liece-channel-list-mode-map
  (setq liece-channel-list-mode-map (make-keymap))
  (liece-define-keys liece-channel-list-mode-map liece-channel-list-keys)
  (liece-define-keys liece-channel-list-mode-map liece-select-keys)
  )

(unless liece-nick-mode-map
  (define-prefix-command 'liece-nick-mode-map)
  (setq liece-nick-mode-map (make-keymap))
  (liece-define-keys liece-nick-mode-map liece-nick-keys)
  (fset 'liece-nick-prefix liece-nick-mode-map)
  )

;;;###liece-autoload
(defmacro liece-truncate-nickname (str)
  (list 'truncate-string-to-width str 9))

;;;###liece-autoload
(defmacro liece-server-opened ()
  "Return server process status, T or NIL.
If the stream is opened, return T, otherwise return NIL."
  '(and liece-server-process
	(memq (process-status liece-server-process)
	      '(open run))))

(defun liece-start-server (&optional confirm)
  "Open network stream to remote irc server.
If optional argument CONFIRM is non-nil, ask the host that the server
is running on."
  (if (liece-server-opened)
      ;; Stream is already opened.
      nil
    ;; Open IRC server.
    (when (or confirm (null liece-server))
      (setq liece-server
	    (liece-minibuffer-completing-default-read
	     (_ "IRC server: ")
	     liece-server-alist)))
    (and confirm
	 liece-ask-for-nickname
	 (setq liece-nickname
	       (read-string (_ "Enter your nickname: ") liece-nickname)))
    ;; If no server name is given, local host is assumed.
    (and
     (stringp liece-server)
     (string-equal liece-server "")
     (setq liece-server (system-name)))
    (let ((host (liece-server-host)))
      (liece-message
       (_ "Connecting to IRC server on %s...") host)
      (cond
       ((liece-open-server liece-server liece-service))
       ((and (stringp liece-status-message-string)
	     (> (length liece-status-message-string) 0))
	;; Show valuable message if available.
	(error liece-status-message-string))
       (t (error (_ "Cannot open IRC server on %s") host))))))

(defun liece-close-server-internal ()
  "Close connection to chat server."
  (if (liece-server-opened)
      (delete-process liece-server-process))
  (if liece-server-buffer
      (kill-buffer liece-server-buffer))
  (setq liece-server-buffer nil
	liece-server-process nil
	liece-server nil))

;;;###liece-autoload
(defun liece-close-server ()
  "Close chat server."
  (unwind-protect
      (progn
	;; Un-set default sentinel function before closing connection.
	(and
	 liece-server-process
	 (eq (quote liece-sentinel)
	     (process-sentinel liece-server-process))
	 (set-process-sentinel liece-server-process nil))
	;; We cannot send QUIT command unless the process is running.
	(if (liece-server-opened)
	    (liece-send "QUIT"))
	)
    (liece-close-server-internal)))

(defmacro liece-server-keyword-bind (plist &rest body)
  "Return a `let' form that binds all variables in PLIST.
After this is done, BODY will be executed in the scope
of the `let' form.

The variables bound and their default values are described by
the `liece-server-keyword-map' variable."
  `(let ,(mapcar
	  (lambda (keyword)
	    (list (intern (substring (symbol-name (car keyword)) 1))
		  (if (cadr keyword)
		      `(or (plist-get plist ',(car keyword))
			   ,(cadr keyword))
		    `(plist-get plist ',(car keyword)))))
	  liece-server-keyword-map)
     ,@body))

(put 'liece-server-keyword-bind 'lisp-indent-function 1)
(put 'liece-server-keyword-bind 'edebug-form-spec '(form body))

(defun liece-server-parse-string (string)
  (when (or (string-match "^\\([^:]+\\):?\\([0-9]*\\)" string)
	    (string-match "^[\\([^\\[]+\\)]:?\\([0-9]*\\)" string))
    (let ((host (match-string 1 string))
	  (service (match-string 2 string))
	  (password (substring string (match-end 0)))
	  plist)
      (push `(:host ,host) plist)
      (unless (string= service "")
	(push `(:service ,(string-to-int service)) plist))
      (cond
       ((string= password ":")
	(setq liece-ask-for-password t))
       ((string= password ""))
       (t (push `(:password ,(substring password 1)) plist)))
      (apply #'nconc plist)
      )))

(defun liece-open-server (host &optional service)
  "Open chat server on HOST.
If HOST is nil, use value of environment variable \"IRCSERVER\".
If optional argument SERVICE is non-nil, open by the service name."
  (let* ((host (or host (getenv "IRCSERVER")))
	 (plist
	  (if (valid-plist-p host)
	      host
	    (or (cdr (string-assoc-ignore-case host liece-server-alist))
		(liece-server-parse-string host))))
	 status)
    (setq liece-status-message-string "")
    (when (stringp plist) ;; Old style server entry...
      (setq plist (liece-server-parse-string host)))
    (when (and (stringp host)
	       (null (string-assoc-ignore-case host liece-server-alist)))
      (push (cons host plist) liece-server-alist)
      (setq liece-save-variables-are-dirty t))
    (liece-server-keyword-bind plist
      ;; Execute preconnecting script
      (when prescript
	(if (fboundp prescript)
	    (funcall prescript)
	  (call-process shell-file-name nil nil nil
			shell-command-switch prescript))
	(when prescript-delay
	  (sleep-for prescript-delay)))
      (if password
	  (setq liece-ask-for-password nil
		liece-password password))
      (if (and (memq type '(rlogin telnet)) relay)
	  (setq liece-tcp-relay-host relay))
      (setq liece-tmp-server-name host);; temporary
      (liece-message (_ "Connecting to IRC server %s...") host)
      (cond
       ((null host)
	(setq liece-status-message-string
	      (_ "IRC server is not specified.")))
       ((liece-open-server-internal host service type)
	(setq liece-after-registration nil)
	(liece-maybe-poll)
	(setq status (liece-wait-for-response "^:[^ ]+ [4P][5O][1N][ G]"))
	(if (null status)
	    (progn
	      (setq liece-status-message-string
		    (format (_ "Connection to %s timed out") host))
	      ;; We have to close connection here, since the function
	      ;;  `liece-server-opened' may return incorrect status.
	      (liece-close-server-internal))
	  (setq liece-after-registration t)
	  (set-process-sentinel liece-server-process 'liece-sentinel)
	  (set-process-filter liece-server-process 'liece-filter)
	  (if (or liece-ask-for-password liece-reconnect-with-password)
	      (let ((passwd-echo ?*) password)
		(setq password (read-passwd (_ "Server Password: ")))
		(or (string= password "")
		    (setq liece-password password))))
	  (if liece-password
	      (liece-send "PASS %s" liece-password))
	  (setq liece-reconnect-with-password nil)
	  (liece-send "USER %s * * :%s"
		      (or (user-real-login-name) "Nobody")
		      (if (and liece-name (not (string= liece-name "")))
			  liece-name
			"No Name"))
	  (or liece-real-nickname
	      (setq liece-real-nickname liece-nickname))
	  (setq liece-real-nickname
		(liece-truncate-nickname liece-real-nickname)
		liece-nickname-last liece-real-nickname
		liece-nick-accepted 'sent
		liece-after-registration t)
	  (liece-send "NICK %s" liece-real-nickname))
	)))
    status))

(defun liece-open-server-internal (host &optional service type)
  "Open connection to chat server on HOST by SERVICE (default is irc).
Optional argument TYPE specifies connection types such as `program'."
  (condition-case err
      (save-excursion
	;; Initialize communication buffer.
	(setq liece-server-buffer (liece-get-buffer-create " *IRC*"))
	(set-buffer liece-server-buffer)
	(kill-all-local-variables)
	(buffer-disable-undo)
	(erase-buffer)
	(cond
	 ((string-match "^[^\\[]" host)
	  (setq liece-server-process
		(liece-open-network-stream-as-binary
		 "IRC" (current-buffer) host (or service "irc") type)))
	 ((not
	   (or
	    (string-match
	     "^\\[\\([0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+\\)\\]$" host)
	    (string-match
	     "^\\[\\([0-9A-Za-z]*:[0-9A-Za-z:]*\\)\\]$" host)
	    (string-match
	     "^\\[\\([0-9]+\\)\\]$" host)))
	  (setq liece-status-message-string
		(_ "Use [nnn.nnn.nnn.nnn]")
		liece-server-process nil)))
	(setq liece-server-name host)
	(run-hooks 'liece-server-hook)
	;; Return the server process.
	liece-server-process)
    (error
     (setq liece-status-message-string (cadr err)) nil)))

(defun liece-initialize-timers ()
  (dolist (timer liece-timers)
    (if (caddr timer)
	(liece-cancel-timer (caddr timer))
      (let ((handler (car timer)) (interval (cadr timer)))
	(and (liece-functionp handler)
	     (symbolp interval) (boundp interval)
	     (setq interval (symbol-value interval))
	     (setcdr (cdr timer)
		     (list (liece-run-at-time 1 interval handler))))
	)))
  (setq liece-timers-list-initialized-p t))

(defun liece-read-variables-files (&optional file)
  "Read variables FILEs."
  (and (not (file-directory-p liece-directory))
       (file-exists-p liece-directory)
       (yes-or-no-p "Upgrade the location of the data files? ")
       (let ((file
	      (expand-file-name
	       (make-temp-name "liece") temporary-file-directory)))
	 (unwind-protect
	     (progn
	       (rename-file liece-directory file 'ok-if-exists)
	       (make-directory liece-directory)
	       (copy-file file (expand-file-name
				(file-name-nondirectory liece-variables-file)
				liece-directory)))
	   (ignore-errors (delete-file file)))))
  (or (file-directory-p liece-directory)
      (make-directory liece-directory))
  (let ((files (if file
		   (progn
		     (setq liece-variables-file file
			   liece-variables-files (list file)))
		 liece-variables-files)))
    (dolist (file files)
      (if (file-readable-p (expand-file-name file))
	  (load (expand-file-name file) t)))))

;;;###autoload
(defun liece (&optional confirm)
  "Connect to the IRC server and start chatting.
If optional argument CONFIRM is non-nil, ask which IRC server to connect.
If already connected, just pop up the windows."
  (interactive "P")
  (liece-read-variables-files
   (car command-line-args-left))
  (pop command-line-args-left)
  (run-hooks 'liece-after-load-startup-hook)
  ;; Save initial state of window configuration.
  (when (interactive-p)
    (liece-window-configuration-push))
  (unless liece-intl-message-alist
    (liece-intl-load-catalogue))
  (if (liece-server-opened)
      (liece-configure-windows)
    (unwind-protect
	(progn
	  (switch-to-buffer
	   (liece-get-buffer-create liece-command-buffer))
	  (unless (eq major-mode 'liece-command-mode)
	    (liece-command-mode))
	  (liece-start-server confirm))
      (if (not (liece-server-opened))
	  (liece-command-quit)
	;; IRC server is successfully open.
	(setq mode-line-process (format " {%s}" liece-server))
	(let (buffer-read-only)
	  (unless liece-keep-buffers
	    (erase-buffer))
	  (sit-for 0))

	(liece-set-crypt-indicator)
	(liece-crypt-initialize)

	(liece-initialize-buffers)
	(liece-configure-windows)
	(setq liece-current-channels nil)
	(cond
	 (liece-current-channel
	  (liece-command-join liece-current-channel))
	 (liece-startup-channel
	  (liece-command-join liece-startup-channel))
	 (liece-startup-channel-list
	  (dolist (chnl liece-startup-channel-list)
	    (if (listp chnl)
		(liece-command-join (car chnl) (cadr chnl))
	      (liece-command-join chnl)))))
	(unless (string-equal liece-away-message "")
	  (liece-command-away liece-away-message))
	(run-hooks 'liece-startup-hook)
	(setq liece-obarray
	      (or liece-obarray (make-vector liece-obarray-size nil)))
	(unless liece-timers-list-initialized-p
	  (liece-initialize-timers))
	(liece-command-timestamp)
	(message (substitute-command-keys 
		  "Type \\[describe-mode] for help"))))))

;;;###liece-autoload
(defun liece-command-mode ()
  "Major mode for IRCHAT.  Normal edit function are available.
Typing Return or Linefeed enters the current line in the dialogue.
The following special commands are available:
For a list of the generic commands type \\[liece-command-generic] ? RET.
\\{liece-command-mode-map}"
  (interactive)
  (kill-all-local-variables)

  (make-local-variable 'mode-line-format)

  (liece-set-crypt-indicator)
  (setq liece-nick-alist (list (list liece-nickname))
	mode-line-modified "--- "
	major-mode 'liece-command-mode
	mode-name "IRCHAT Commands"
	liece-privmsg-partner nil
	liece-private-indicator nil
	liece-away-indicator "-"
	liece-freeze-indicator "-"
	liece-own-freeze-indicator "-"
	mode-line-buffer-identification
	(liece-mode-line-buffer-identification (list "Liece:"))
	
	mode-line-format
	'(" "
	  mode-line-buffer-identification
	  " Commands -"
	  liece-private-indicator
	  liece-away-indicator
	  liece-crypt-indicator
	  "- " liece-current-channel
	  " -%[" minor-mode-alist "%] "
	  liece-real-nickname " (" liece-server ")"
	  (-3 . "%p") " %-"))

  (use-local-map liece-command-mode-map)

  (unless window-system
    (setq liece-display-frame-title nil))
  
  (when liece-display-frame-title
    (make-local-variable 'frame-title-format)
    (setq frame-title-format 'liece-channel-status-indicator))
  
  (unless liece-blink-parens
    (make-local-variable 'blink-matching-paren)
    (setq blink-matching-paren nil))
  
  (unless liece-command-mode-syntax-table
    (setq liece-command-mode-syntax-table
	  (copy-syntax-table (syntax-table)))
    (set-syntax-table liece-command-mode-syntax-table)
    (mapcar
     (function (lambda (c) (modify-syntax-entry c "w")))
     "^[]{}'`"))

  (run-hooks 'liece-command-mode-hook))
  
;;;###liece-autoload
(defun liece-dialogue-mode ()
  "Major mode for displaying the IRC dialogue.
All normal editing commands are turned off.
Instead, these commands are available:
\\{liece-dialogue-mode-map}"
  (kill-all-local-variables)

  (make-local-variable 'mode-line-format)
  (make-local-variable 'liece-freeze)
  (make-local-variable 'liece-freeze-indicator)
  (make-local-variable 'liece-own-freeze)
  (make-local-variable 'liece-own-freeze-indicator)
  (make-local-variable 'tab-stop-list)

  (setq liece-freeze liece-default-freeze
	liece-freeze-indicator (if liece-freeze "F" "-")
	liece-own-freeze liece-default-own-freeze
	liece-own-freeze-indicator (if liece-own-freeze "M" "-")
	
	mode-line-modified "--- "
	major-mode 'liece-dialogue-mode
	mode-name "IRCHAT Dialogue"
	mode-line-buffer-identification
	(liece-mode-line-buffer-identification (list "Liece:"))
	
	mode-line-format
	'(" "
	  mode-line-buffer-identification
	  " Dialogue -"
	  liece-away-indicator
	  liece-crypt-indicator
	  liece-freeze-indicator
	  liece-own-freeze-indicator
	  "- " liece-channels-indicator " "
	  (-3 . "%p") "-%-")
	buffer-read-only t
	tab-stop-list liece-tab-stop-list)
  
  (use-local-map liece-dialogue-mode-map)
  (buffer-disable-undo)

  (unless liece-keep-buffers
    (erase-buffer))
  
  (run-hooks 'liece-dialogue-mode-hook))

;;;###liece-autoload
(define-derived-mode liece-others-mode liece-dialogue-mode
  "Other channels"
  "Major mode for displaying the IRC others message except current channel.
All normal editing commands are turned off.
Instead, these commands are available:
\\{liece-others-mode-map}"
  (setq mode-line-modified "--- "
	mode-line-buffer-identification
	(liece-mode-line-buffer-identification (list "Liece:"))

        mode-line-format
	'(" "
	  mode-line-buffer-identification
	  " Others -"
	  liece-away-indicator
	  liece-crypt-indicator
	  liece-freeze-indicator
	  liece-own-freeze-indicator
	  "- " liece-channels-indicator " "
	  (-3 . "%p") "-%-")
	buffer-read-only t))

;;;###liece-autoload
(define-derived-mode liece-channel-mode liece-dialogue-mode
  "Current channel"
  "Major mode for displaying the IRC current channel buffer.
All normal editing commands are turned off.
Instead, these commands are available:
\\{liece-channel-mode-map}"
  (setq mode-line-modified "--- "
	mode-line-buffer-identification
	(liece-mode-line-buffer-identification (list "Liece:"))
	
        mode-line-format
	'(" "
	  mode-line-buffer-identification
	  " Channel -"
	  liece-away-indicator
	  liece-crypt-indicator
	  liece-freeze-indicator
	  liece-own-freeze-indicator
	  "- " liece-channel-indicator " "
	  (-3 . "%p") "-%-")
	buffer-read-only t))

;;;###liece-autoload
(defun liece-channel-list-mode ()
  "Major mode for displaying channel list
All normal editing commands are turned off."
  (kill-all-local-variables)
  (setq mode-line-modified "--- "
        major-mode 'liece-channel-list-mode
        mode-name "IRCHAT Channel list"
	mode-line-buffer-identification
	(liece-mode-line-buffer-identification (list "Liece:"))

        mode-line-format
	'(" "
	  mode-line-buffer-identification
	  " " liece-command-buffer-mode-indicator " "
	  "-" (-3 . "%p") "-%-")
	truncate-lines t
	buffer-read-only t)
  (use-local-map liece-channel-list-mode-map)
  (run-hooks 'liece-channel-list-mode-hook))

;;;###liece-autoload
(defun liece-nick-mode ()
  "Major mode for displaying members in the IRC current channel buffer.
All normal editing commands are turned off.
Instead, these commands are available:
\\{liece-nick-mode-map}"
  (kill-all-local-variables)
  (setq mode-line-modified "--- "
        major-mode 'liece-nick-mode
        mode-name "IRCHAT Channel member"
	mode-line-buffer-identification
	(liece-mode-line-buffer-identification (list "Liece:"))

        mode-line-format
	'(" "
	  mode-line-buffer-identification
	  " Nick "
	  "{" liece-channel-indicator "} "
	  "-" (-3 . "%p") "-%-")
	truncate-lines t
	buffer-read-only t)
  (use-local-map liece-nick-mode-map)
  (run-hooks 'liece-nick-mode-hook))

(fset 'liece-dialogue-freeze 'liece-command-freeze)
(fset 'liece-dialogue-own-freeze 'liece-command-own-freeze)

(defun liece-initialize-buffers ()
  "Initialize buffers."
  (dolist (spec liece-buffer-mode-alist)
    (let ((buffer (symbol-value (car spec)))
	  (mode (cadr spec)))
      (or (get-buffer buffer)
	  (save-excursion
	    (set-buffer (liece-get-buffer-create buffer))
	    (or (eq major-mode mode)
		(null mode)
		(funcall mode)))))
    ))

;;;###liece-autoload
(defun liece-clear-system ()
  "Clear all IRCHAT variables and buffers."
  (interactive)
  (dolist (buffer liece-buffer-list)
    (when (get-buffer buffer)
      (bury-buffer buffer)))
  (if (vectorp liece-obarray)
      (dotimes (i liece-obarray-size)
	(aset liece-obarray i nil)))
  (dolist (timer liece-timers)
    (if (caddr timer)
	(liece-cancel-timer (caddr timer)))
    (if (cdr timer)
	(setcdr (cdr timer) nil)))
  (setq liece-channel-buffer-alist nil
	liece-nick-buffer-alist nil
	liece-current-channels nil
	liece-current-channel nil
	liece-current-chat-partners nil
	liece-current-chat-partner nil
	liece-timers-list-initialized-p nil
	liece-friends-last nil
	liece-polling 0
	liece-channel-indicator "No channel"))

(defun liece-wait-for-response (regexp &optional timeout)
  "Wait for server response which match REGEXP.
Optional argument TIMEOUT specifies connection timeout."
  (save-excursion
    (let ((status t) (wait t) (timeout (or timeout liece-connection-timeout)))
      (set-buffer liece-server-buffer)
      (with-timeout (timeout nil)
	(while wait
	  (liece-accept-response)
	  (goto-char (point-min))
	  (cond ((looking-at "ERROR") (setq status nil wait nil))
		((looking-at ".") (setq wait nil))))
	;; Save status message.
	(end-of-line)
	(setq liece-status-message-string
	      (buffer-substring (point-min) (point)))
	(when status
	  (while wait
	    (goto-char (point-max))
	    (forward-line -1)
	    (if (looking-at regexp)
		(setq wait 0)
	      (liece-message (_ "Reading..."))
	      (liece-accept-response))))
	;; Successfully received server response.
	t))))

(defun liece-accept-process-output (process &optional timeout)
  "Wait for output from PROCESS and message some dots.
Optional argument TIMEOUT specifies connection timeout."
  (save-excursion
    (set-buffer liece-server-buffer)
    (accept-process-output process (or timeout 1))))

(defun liece-accept-response ()
  "Read response of server.  Only used at startup time."
  (unless (liece-server-opened)
    (cond
     ((not liece-reconnect-automagic)
      (error "IRCHAT: Connection closed"))
     (liece-grow-tail
      (let ((liece-nickname (concat liece-nickname liece-grow-tail)))
	(liece)))
     (t (liece))))
  (condition-case code
      (liece-accept-process-output liece-server-process)
    (error
     (or (string-equal "select error: Invalid argument" (nth 1 code))
	 (signal (car code) (cdr code))))))

(defmacro liece-replace-internal (buffer match defstring oldstring newstring)
  `(save-excursion
     (set-buffer (get-buffer ,buffer))
     (let (buffer-read-only (inhibit-read-only t))
       (goto-char (point-max))
       (previous-line liece-compress-treshold)
       (save-match-data
	 (if (not (re-search-forward ,match nil t))
	     (liece-insert ,buffer ,defstring)
	   (while (re-search-forward ,match nil t))
	   (beginning-of-line)
	   (if (re-search-forward ,oldstring nil t)
	       (replace-match ,newstring nil t)
	     ;; This should't happen (kny)
	     (liece-insert ,buffer ,defstring))
	   (liece-insert ,buffer ""))))))

;;;###liece-autoload
(defun liece-replace (buffer match defstring oldstring newstring)
  "Replace in buffer or list of buffers BUFFER with matching MATCH.
Argument DEFSTRING used when no matches are there.
Argument OLDSTRING is replaced with NEWSTRING."
  (unless (listp buffer)
    (setq buffer (list buffer)))
  (dolist (buf buffer)
    (when (get-buffer buf)
      (liece-replace-internal buf match defstring oldstring newstring))))

(defmacro liece-check-buffers ()
  '(let ((liece-checkbuffer-interval 0))
     (when (> liece-buffer-maxsize 0)
       (save-excursion
	 (dolist (buffer liece-channel-buffer-alist)
	   (set-buffer (cdr buffer))
	   (when (< liece-buffer-maxsize (buffer-size))
	     (let (buffer-read-only)
	       (goto-char (- (buffer-size) liece-buffer-defsize))
	       (forward-line -1)
	       (delete-region (point-min) (point)))))))
     (setq liece-last-checkbuffer-time (current-time))))

(defun liece-check-buffers-if-interval-expired ()
  (if (and (numberp liece-checkbuffer-interval)
	   (> liece-checkbuffer-interval 0)
	   (or (null liece-last-checkbuffer-time)
	       (> (liece-time-difference
		   liece-last-checkbuffer-time
		   (current-time))
		  liece-checkbuffer-interval)))
      (progn (liece-check-buffers) t)
    nil))

(defun liece-refresh-buffer-window (buffer)
  (let ((window (liece-get-buffer-window buffer)))
    (when (and window (not (pos-visible-in-window-p (point-max) window)))
      (save-selected-window
	(select-window window)
	(goto-char (point-max))
	(if (null liece-scroll-step)
	    (recenter (- (liece-window-height window) 1))
	  (vertical-motion
	   (- (or liece-scroll-step
		  (1+ (/ (liece-window-height window) 2)))
	      (liece-window-height window)))
	  (set-window-start window (point))
	  (goto-char (point-max)))
	))))

(defmacro liece-save-point (&rest body)
  `(let ((liece-save-point (point-marker)))
     (unwind-protect
	 (progn ,@body)
       (goto-char liece-save-point)
       (set-marker liece-save-point nil))))

(defvar liece-before-insert-hook
  '(liece-check-buffers-if-interval-expired
    liece-command-timestamp-if-interval-expired))

(defun liece-insert-internal (buffer string)
  (run-hooks 'liece-before-insert-hook)
  (with-current-buffer (liece-get-buffer-create buffer)
    (or (eq (derived-mode-class major-mode) 'liece-dialogue-mode)
	(liece-dialogue-mode))
    (liece-save-point
     (let ((inhibit-read-only t)
	   buffer-read-only
	   (from (progn (goto-char(point-max)) (point))))
       (unless (liece-is-message-ignored string (current-buffer))
	 (and liece-display-time (not (string-equal string ""))
	      (liece-insert-time-string))
	 (insert string)
	 (run-hook-with-args 'liece-insert-hook from (point))
	 )))
    (unless (liece-frozen (current-buffer))
      (liece-refresh-buffer-window (current-buffer)))
    ))

;;;###liece-autoload
(defun liece-insert (buffer string)
  (or (listp buffer)
      (setq buffer (list buffer)))
  (dolist (buf buffer)
    (when (get-buffer buf)
      (liece-insert-internal buf string))))

(provide 'liece)
;;; liece.el ends here
