/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include <stdio.h>
#include "instP.h"
#include "streampool.h"
#include "transobj.h"
#include "handleP.h"

static char *locations[] = {
    "none",		/* L_NONE */
    "local",		/* L_LOCAL */
    "global",		/* L_GLOBAL */
    "camera",		/* L_CAMERA */
    "ndc",		/* L_NDC */
    "screen",		/* L_SCREEN */
};

static int getlocation(char *name)
{
  int i;
  if(name == NULL)
    return -1;
  for(i = COUNT(locations); --i >= 0 && strcasecmp(name, locations[i]) != 0; )
    ;
  return i;	/* Return location number, or -1 if not found. */
}

Geom *
InstImport( Pool *p )
{
    register Inst *inst = NULL;
    FILE *file;
    char *expect = NULL;
    char *w;
    int c;

    if(p == NULL || (file = p->inf) == NULL)
	return 0;

    if(strcmp(GeomToken(file), "INST"))
	return 0;

    for(;;) {
	switch((c = fnextc(file, 0))) {
	case EOF:
	case CKET:
	    goto done;

	case 'l':
	    if(fexpectstr(file, expect = "location"))
		goto syntax;
	    if(inst == NULL)
		inst = (Inst *)GeomCCreate(NULL, InstMethods(), NULL);
	    inst->location = getlocation( fdelimtok("(){}", file, 0) );
	    expect = "location [local|global|camera|ndc|screen]";
	    if(inst->location < 0)
		goto syntax;
	    break;

	case 'o':
	    if(fexpectstr(file, expect = "origin"))
		goto syntax;
	    if(inst == NULL)
		inst = (Inst *)GeomCCreate(NULL, InstMethods(), NULL);

	    expect = "origin [local|global|camera|ndc|screen] X Y Z";
	    inst->origin = getlocation( fdelimtok("(){}", file, 0) );
	    if(inst->origin < 0)
		goto syntax;
	    if(fgetnf(file, 3, &inst->originpt.x, 0) < 3)
		goto syntax;
	    break;
	    
	    
	case 'u':
	    if(fexpectstr(file, expect = "unit"))
		goto syntax;
	    goto geom;

	case 'g':
	    if(fexpectstr(file, expect = "geom"))
		goto syntax;

	  geom:
	    if(inst == NULL)
		inst = (Inst *)GeomCCreate(NULL, InstMethods(), NULL);
	    expect = "geometry";
	    if(!GeomStreamIn(p, &inst->geomhandle, &inst->geom))
		goto failed;
	    if(inst->geomhandle)
		HandleRegister(&inst->geomhandle, (Ref *)inst,
				&inst->geom, HandleUpdRef);
	    break;

	case 't':		/* tlist ... or transform ... */
	    if(inst == NULL)
		inst = (Inst *)GeomCCreate(NULL, InstMethods(), NULL);
	    fgetc(file);
	    switch((c = fgetc(file))) {
	    case 'l':
		if(fexpectstr(file, (expect = "tlist")+2)) 	/* "tlist" */
		    goto syntax;
	     transforms:
		if(inst == NULL)
		    inst = (Inst *)GeomCCreate(NULL, InstMethods(), NULL);
		expect = "TLIST object";
		if(!GeomStreamIn(p, &inst->tlisthandle, &inst->tlist))
		    goto failed;
		if(inst->tlisthandle)
		    HandleRegister(&inst->tlisthandle, (Ref *)inst,
				&inst->tlist, HandleUpdRef);
		break;

	    case 'r':
		if(fexpectstr(file, (expect = "transform")+2))		/* "transform" */
		    goto syntax;
		if(fexpectstr(file, "s") == 0)	/* transforms = tlist */
		    goto transforms;
		if(inst == NULL)
		    inst = (Inst *)GeomCCreate(NULL, InstMethods(), NULL);
		expect = "transform matrix";
		if(!TransStreamIn(p, &inst->axishandle, inst->axis))
		    goto failed;
		if(inst->axishandle)
		    HandleRegister(&inst->axishandle, (Ref *)inst,
				inst->axis, TransUpdate);
		break;

	    default:
		expect = "something";
		goto syntax;

	    }
	    break;

	default:
	syntax:
	  OOGLSyntax(file, "Couldn't read INST in \"%s\": syntax error, expected %s",
			p->poolname, expect);
	  goto bogus;

	failed:
	  OOGLSyntax(file, "Couldn't read INST in \"%s\": expected %s",
			PoolName(p), expect);

        bogus:
	  GeomDelete((Geom *)inst);
	  return NULL;
	}
    }

  done:
    return (Geom *)inst;
}

int
InstExport( Inst *inst, Pool *pool )
{
    int ok = 1;

    if(inst == NULL || pool == NULL || pool->outf == NULL)
	return NULL;

    fprintf(pool->outf, "INST\n");
    if(inst->origin != L_NONE && (unsigned)inst->origin < COUNT(locations)) {
	fprintf(pool->outf, "  origin %s ", locations[inst->origin]);
	fputnf(pool->outf, 3, &inst->originpt.x, 0);
	fputc('\n', pool->outf);
    }

    if(inst->location != L_LOCAL && inst->location != L_NONE &&
			(unsigned)inst->location < COUNT(locations))
	fprintf(pool->outf, "  location %s\n", locations[inst->location]);

    if(inst->tlist != NULL || inst->tlisthandle != NULL) {
	fprintf(pool->outf, "  tlist ");
	ok &= GeomStreamOut(pool, inst->tlisthandle, inst->tlist);
    } else {
	ok &= TransStreamOut(pool, inst->axishandle, inst->axis);
    }

    if(inst->geom != NULL || inst->geomhandle != NULL) {
	fprintf(pool->outf, "  geom ");
	ok &= GeomStreamOut(pool, inst->geomhandle, inst->geom);
    }
    return ok;
}
