// $Id: CommandStack.cpp,v 1.4 2002/04/16 23:21:12 xfred_1 Exp $

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef WIN32
# include <assert.h>
#endif

#include "Command.h"
#include "CommandStack.h"

/**
 * Clears part of the stack.
 * 
 * TODO use algorithm style.
 */
void
CommandStack::Clear(const CLIterator& iterator)
{
   for (CLIterator i = iterator; i != m_Stack.end() ; ++i)
   {
      Command* command = *i;
      delete command;
   }
   
   m_Stack.erase(iterator, m_Stack.end());
}


/**
 * Removes the oldest undoable command.
 */
void
CommandStack::Forget()
{
   CLIterator iterator = m_Stack.begin();

   Command* command = *iterator;
   delete command;
   
   m_Stack.pop_front();
}


CommandStack::CommandStack(unsigned size) : 
    m_Stack(),
    m_Current(m_Stack.end()),
    m_MaxSize(size)
{}

CommandStack::~CommandStack()
{
   Clear();
}

/**
 * Try to merge this command with the one on the top of the stack, and
 * if it's not possible, push it to the top of the command stack. 
 */
void
CommandStack::Push(Command* command)
{
    // We can put only valid commands
    assert( command && "Pushing a null Command on the stack" );

    // If the command is not undoable, flush the stack
    if (! command->IsUndoable() )
    {
	Clear();
	m_Current = m_Stack.end();
	return;
    }
    
    // We have undone something, remove the redoable state
    Clear(m_Current);

    // Too many items ?
    if (m_Stack.size() == m_MaxSize)
    {
	Forget();
    }

    // Push the command
    m_Stack.push_back( command );
    m_Current = m_Stack.end();
}


/// Undo the last command.
void CommandStack::Undo()
{
   if (CannotUndo())
   {
       // Maybe raise an assert, sometimes
       return;
   }
   
   // Take the last command
   --m_Current;
   Command* command = *m_Current;
   
   assert( command && "Commands in the stack should never be null" );
   
   command->Unexecute();
}

/// Redo the last undone command.
void
CommandStack::Redo()
{
   if (CannotRedo()) 
   {
      // Maybe raise an assert, sometimes
      return;
   }
   
   // Take the current and move to next
   Command *command = *m_Current;
   ++m_Current;
   
   assert( command && "Commands in the stack should never be null" );
   
   command->Execute();
}


