# FIAIF is an Intelligent firewall, version: $Revision: 1.22 $
#
# description: Setup Traffic Shaping.
#
# Script Author:	Hubert <ahu at ds9a dot nl>
#                       Anders Fugmann <afu at fugmann dot net>
#                       
# FIAIF is an Intelligent firewall
# Copyright (C) 2002 Bert Hubert
# Copyright (C) 2002-2003 Anders Peter Fugmann
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

###############################################################################
# Stop TC for an interface
###############################################################################
function tc_stop_zone ()
{
    local DEVICE=$1
    # clean existing down- and uplink qdiscs, hide any errors
    TC qdisc del dev ${DEVICE} root    > /dev/null 2>&1
    TC qdisc del dev ${DEVICE} ingress > /dev/null 2>&1
}

###############################################################################
# Setup CBQ TC for an interface
###############################################################################
function tc_setup_zone_cbq ()
{
    local DEVICE=$1
    local SPEED_UPLINK=$2
    local SPEED_DOWNLINK=$3

    # install root CBQ

    TC qdisc add dev ${DEVICE} root handle 1: cbq avpkt 1000 \
	bandwidth 100mbit 

    # shape everything at ${SPEED_UPLINK} speed - 
    # this prevents huge queues in your
    # DSL modem which destroy latency:
    # main class

    TC class add dev ${DEVICE} parent 1: classid 1:1 cbq \
	rate $((SPEED_UPLINK))kbit \
	allot 1500 prio 5 bounded isolated 

    # high prio class 1:10:

    TC class add dev ${DEVICE} parent 1:1 classid 1:10 cbq \
	rate $((SPEED_UPLINK))kbit \
	allot 1600 prio 1 avpkt 1000

    # bulk and default class 1:20 - gets slightly less traffic, 
    #  and a lower priority:

    TC class add dev ${DEVICE} parent 1:1 classid 1:20 cbq \
	rate $((9*SPEED_UPLINK/10))kbit allot 1600 prio 2 avpkt 1000

    # low prio class 1:30 - gets slightly less traffic, 
    #  and a even lower priority:
    TC class add dev ${DEVICE} parent 1:1 classid 1:30 cbq \
	rate $((8*SPEED_UPLINK/10))kbit allot 1600 prio 3 avpkt 1000

    # All get Stochastic Fairness:
    TC qdisc add dev ${DEVICE} parent 1:10 handle 10: sfq perturb 10
    TC qdisc add dev ${DEVICE} parent 1:20 handle 20: sfq perturb 10
    TC qdisc add dev ${DEVICE} parent 1:30 handle 30: sfq perturb 10

    # start filters
    # TOS Minimum Delay (ssh, NOT scp) in 1:10:
    TC filter add dev ${DEVICE} parent 1:0 protocol ip prio 10 u32 \
	match ip tos 0x10 0xff  flowid 1:10

    # ICMP (ip protocol 1) in the interactive class 1:10 so we 
    # can do measurements & impress our friends:
    TC filter add dev ${DEVICE} parent 1:0 protocol ip prio 11 u32 \
	match ip protocol 1 0xff flowid 1:10

    # To speed up downloads while an upload is going on, put ACK packets in
    # the interactive class:
 
    TC filter add dev ${DEVICE} parent 1: protocol ip prio 12 u32 \
	match ip protocol 6 0xff \
	match u8 0x05 0x0f at 0 \
	match u16 0x0000 0xffc0 at 2 \
	match u8 0x10 0xff at 33 \
	flowid 1:10
    
    # iptables mark for low latency
    TC filter add dev ${DEVICE} parent 1: protocol ip prio 13 \
	handle 1 fw flowid 1:30

    # If maximize-throughput TOS bit is set, it gets low prio.
    TC filter add dev ${DEVICE} parent 1: protocol ip prio 14 u32 \
      match ip tos 0x08 0xff flowid 1:30

    # Everything else goes into normal/middle prio queue.
    TC filter add dev ${DEVICE} parent 1: protocol ip prio 15 u32 \
    	match ip dst 0.0.0.0/0 flowid 1:20


    ########## downlink ####################################
    # slow downloads down to somewhat less than the real speed  to prevent
    # queuing at our ISP. Tune to see how high you can set it.
    # ISPs tend to have *huge* queues to make sure big downloads are fast
    #
    # attach ingress policer:
    TC qdisc add dev ${DEVICE} handle ffff: ingress

    # filter *everything* to it (0.0.0.0/0), drop everything that's
    # coming in too fast:

    TC filter add dev ${DEVICE} parent ffff: protocol ip prio 50 u32 \
	match ip src 0.0.0.0/0 police \
	rate $((SPEED_DOWNLINK))kbit burst 10k drop flowid :1

    # Use iptables mark functionality instead
    # tc filter add dev eth1 parent ffff: protocol ip prio 50 handle 1 fw \
    # police rate ${DOWNLINK}kbit burst 10k drop 

}

###############################################################################
# Setup HTB TC for an interface
###############################################################################
function tc_setup_zone_htb ()
{
    local DEVICE=$1
    local SPEED_UPLINK=$2
    local SPEED_DOWNLINK=$3

    ###### uplink

    # install root HTB, point default traffic to 1:20:

    TC qdisc add dev $DEVICE root handle 1: htb default 20

    # shape everything at $UPLINK speed - this prevents huge queues in your
    # DSL modem which destroy latency:

    TC class add dev $DEVICE parent 1: classid 1:1 htb rate $((SPEED_UPLINK))kbit \
        burst 6k

    # high prio class 1:10:

    TC class add dev $DEVICE parent 1:1 classid 1:10 htb rate $((SPEED_UPLINK))kbit \
    	burst 6k prio 1

    # bulk & default class 1:20 - gets slightly less traffic,
    # and a lower priority:

    TC class add dev $DEVICE parent 1:1 classid 1:20 htb rate $((9*SPEED_UPLINK/10))kbit \
    	burst 6k prio 2

    # bulk & default class 1:30 - gets even less traffic,
    # and a even lower priority:

    TC class add dev $DEVICE parent 1:1 classid 1:30 htb rate $((8*SPEED_UPLINK/10))kbit \
    	burst 6k prio 3

    # all get Stochastic Fairness:

    TC qdisc add dev $DEVICE parent 1:10 handle 10: sfq perturb 10
    TC qdisc add dev $DEVICE parent 1:20 handle 20: sfq perturb 10
    TC qdisc add dev $DEVICE parent 1:30 handle 30: sfq perturb 10

    # TOS Minimum Delay (ssh, NOT scp) in 1:10:

    TC filter add dev $DEVICE parent 1:0 protocol ip prio 10 u32 \
	match ip tos 0x10 0xff flowid 1:10

    # ICMP (ip protocol 1) in the interactive class 1:10 so we
    # can do measurements & impress our friends:

    TC filter add dev $DEVICE parent 1:0 protocol ip prio 10 u32 \
	match ip protocol 1 0xff flowid 1:10

    # To speed up downloads while an upload is going on, put ACK packets in
    # the interactive class:

    TC filter add dev $DEVICE parent 1: protocol ip prio 10 u32 \
	match ip protocol 6 0xff \
	match u8 0x05 0x0f at 0 \
	match u16 0x0000 0xffc0 at 2 \
	match u8 0x10 0xff at 33 \
	flowid 1:10

    # rest is 'non-interactive' ie 'bulk' and ends up in 1:20

    TC filter add dev $DEVICE parent 1: protocol ip prio 18 u32 \
    	 match ip dst 0.0.0.0/0 flowid 1:20

    ########## downlink #############
    # slow downloads down to somewhat less than the real speed  to prevent
    # queuing at our ISP. Tune to see how high you can set it.
    # ISPs tend to have *huge* queues to make sure big downloads are fast
    #
    # attach ingress policer:

    TC qdisc add dev $DEVICE handle ffff: ingress

    # filter *everything* to it (0.0.0.0/0), drop everything that's
    # coming in too fast:

    TC filter add dev $DEVICE parent ffff: protocol ip prio 50 u32 \
       match ip src 0.0.0.0/0 police rate $((SPEED_DOWNLINK))kbit \
       burst 10k drop flowid :1
}

###############################################################################
# Status 
###############################################################################
function tc_status ()
{
    for ZONE in ${ZONES}; do
	# Retrieve each zone.
	read_zone ${ZONE}
	if (( $? != 0 )); then
	    continue
	fi

	echo "---- qdisc parameters ----"
	TC qdisc ls dev ${DEV}
	echo "---- Class parameters ----"
	TC class ls dev ${DEV}
	echo "---- filter parameters ----"
	TC filter ls dev ${DEV}
    done 
}

###############################################################################
# Stats 
###############################################################################
function tc_stat()
{
    TC -s qdisc list
}

###############################################################################
# Setup Traffic Shaping
###############################################################################
function tc_setup ()
{
    # Load the configuration file:
    if (( ENABLE_TC == 1 )); then
    echo -n "Setting up Traffic Shaping: "
	for ZONE in ${ZONES}; do
	    # Retrieve each zone.
	    read_zone ${ZONE}
	    if (( $? != 0 )); then
		continue
	    fi
	    if (( TC_ENABLE == 1 )); then
		echo -n "${ZONE} "
		debug_out "Traffic shaping for zone: ${ZONE}"
		debug_out "Traffic shaping type: ${TC_TYPE}"
		case "${TC_TYPE}" in
		    "CBQ" | "cbq")
			tc_setup_zone_cbq ${DEV} ${TC_UPLINK} ${TC_DOWNLINK}
			;;
		    "HTB" | "htb")
			tc_setup_zone_htb ${DEV} ${TC_UPLINK} ${TC_DOWNLINK}
			;;
		    *)
			debug_out "Error: Unknown type"
			;;
		esac
		debug_out "Traffic shaping done"
	    fi
	done
	echo 
    fi
}

###############################################################################
# Stop Traffic Shaping
###############################################################################
function tc_stop ()
{
    if (( ENABLE_TC == 1 )); then  
	echo -n "Stopping Traffic Shaping: "
	debug_out "Stopping Traffic Shaping"
	for ZONE in ${ZONES}; do
	    # Retrieve each zone.
	    read_zone ${ZONE}
	    if (( $? != 0 )); then
		continue
	    fi
	    tc_stop_zone ${DEV}
	done
	echo "Done."
    fi
}

