//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qlayout.h>
#include <qlabel.h>
#include <qfont.h>
#include <qtextbrowser.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qframe.h>
#include <qstringlist.h>
#include <qdir.h>
#include <qpushbutton.h>
#include <qfiledialog.h>
#include <qtooltip.h>

//Projectwide includes
#include "saveDialog.h"
#include "../clickableLabel.h"
#include "../../config.h"

//==============================================
SaveDialog::SaveDialog( QString actionMessage,
                                    QString defaultPath,
                                    QString defaultTheme,
                                    QWidget* parent,
                                    const char* name ) :
                                    QDialog(parent,name)
{
  //set window title
  setCaption( actionMessage );

  //set the background of the widget to be white
//  setPaletteBackgroundColor( QColor(255, 255, 255) );


  //create location frame and widgets
  locationFrame = new QFrame( this );
  locationLabel = new QLabel( tr("Save to:"), locationFrame );

  QFont boldFont = locationLabel->font();
  boldFont.setWeight(QFont::Bold);
  
  locationLabel->setFont( boldFont );
  locationVal = new QLineEdit( locationFrame );
  locationVal->setText( defaultPath );
  locationVal->setFont( boldFont );

  browseButton = new ClickableLabel( locationFrame );
  browseButton->setPixmap( QPixmap(QString(IMAGE_PATH)+"buttonIcons/browse.png") );
  QToolTip::add( browseButton, tr("Browse to save destination") );
  connect( browseButton, SIGNAL(clicked()), SLOT(browse()) );
  locationGrid = new QGridLayout( locationFrame, 1, 3, 0 );
  locationGrid->addWidget( locationLabel, 0, 0 );
  locationGrid->addWidget( locationVal, 0, 1 );
  locationGrid->addWidget( browseButton, 0, 2);
  locationGrid->setColStretch( 1, 1 );
  locationGrid->setSpacing(WIDGET_SPACING); 
  
  //create theme selection frame and widgets
  themeSelectionFrame = new QFrame( this );
  themesLabel = new QLabel( tr("Themes:"), themeSelectionFrame );
  themesLabel->setFont( boldFont );
  themesList = new QListBox( themeSelectionFrame );
  QToolTip::add( themesList, tr("Select theme for saving album") );
  QDir localDir( THEMES_PATH );
  QStringList list = localDir.entryList( QDir::Dirs );
  bool itemsAdded = false;
  QStringList::Iterator file;
  for ( file = list.begin(); file != list.end(); ++file )
  {
    if(localDir.exists( QString(*file) + "/theme.xsl" ))
    {
      themesList->insertItem( *file );
      itemsAdded = true;
    }
  }

  //attempt to select default theme passed in, if not found select first theme in list
  bool themeFound = false;
  uint i=0;
  for(i=0; i<themesList->count(); i++)
  {
    if(themesList->text(i) == defaultTheme )
    {
      themeFound = true;
      themesList->setCurrentItem( i );
      break;
    }
  }
  if(!themeFound && itemsAdded )
  {
    themesList->setCurrentItem( 0 );
  }

  connect( themesList, SIGNAL( highlighted(int) ), this, SLOT( updatePreview() ) );

  themeSelectionGrid = new QGridLayout( themeSelectionFrame, 2, 1, 0 );
  themeSelectionGrid->addWidget( themesLabel, 0, 0 );
  themeSelectionGrid->addWidget( themesList, 1, 0 );

  //create theme preview frame and widgets
  themePreviewFrame = new QFrame( this );
  themePreviewLabel = new QLabel( tr("Theme Preview:"), themePreviewFrame );
  themePreviewLabel->setFont( boldFont );
  themeScreenShot = new QLabel( themePreviewFrame );
  screenShotLabel = new QLabel( themePreviewFrame );
  screenShotLabel->setFont( boldFont );
  
  themeScreenPrev = new ClickableLabel( themePreviewFrame );
  themeScreenPrev->setPixmap( QPixmap(QString(IMAGE_PATH)+"buttonIcons/previous.png") );
  QToolTip::add( themeScreenPrev, tr("View previous theme screenshot") );
  connect( themeScreenPrev, SIGNAL(clicked()), SLOT(prevScreenShot()) );
  
  themeScreenNext = new ClickableLabel( themePreviewFrame );
  themeScreenNext->setPixmap( QPixmap(QString(IMAGE_PATH)+"buttonIcons/next.png") );
  QToolTip::add( themeScreenNext, tr("View next theme screenshot") );
  connect( themeScreenNext, SIGNAL(clicked()), SLOT(nextScreenShot()) );

  themeFeatures = new QTextBrowser( themePreviewFrame );
  themeFeatures->setFrameStyle( QFrame::Panel | QFrame::Sunken );
  themeFeatures->mimeSourceFactory()->setFilePath( QStringList(THEMES_PATH) );
  updatePreview();

  themePreviewGrid = new QGridLayout( themePreviewFrame, 5, 5, 0);
  themePreviewGrid->addMultiCellWidget( themePreviewLabel, 0,0, 0,4 );

  themePreviewGrid->addWidget( themeScreenPrev, 1, 0, Qt::AlignVCenter );
  themePreviewGrid->addColSpacing( 1, 10 );
  themePreviewGrid->setColStretch( 1, 1 );
  themePreviewGrid->addWidget( themeScreenShot, 1, 2 );
  themePreviewGrid->addColSpacing( 3, 10 );
  themePreviewGrid->setColStretch( 3, 1 );
  themePreviewGrid->addWidget( themeScreenNext, 1, 4, Qt::AlignVCenter );
  themePreviewGrid->addMultiCellWidget( screenShotLabel, 2, 2, 0, 4, Qt::AlignCenter );
  themePreviewGrid->addMultiCellWidget( themeFeatures, 4, 4, 0, 4 );

  //create buttons frame and widgets
  buttonsFrame = new QFrame( this );
  saveButton = new QPushButton( 
  //PLATFORM_SPECIFIC_CODE
 #ifndef Q_OS_MACX  
 QPixmap(QString(IMAGE_PATH)+"buttonIcons/save.png"),
 #endif
                               tr("Save"), buttonsFrame );
  saveButton->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  saveButton->setDefault(true);
  connect( saveButton, SIGNAL(clicked()), SLOT(save()) );
  cancelButton = new QPushButton( 
  //PLATFORM_SPECIFIC_CODE
  #ifndef Q_OS_MACX  
  QPixmap(QString(IMAGE_PATH)+"buttonIcons/button_cancel.png"),
  #endif
 tr("Cancel"), buttonsFrame
);
  cancelButton->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  connect( cancelButton, SIGNAL(clicked()), SLOT(cancel()) );
  buttonsGrid = new QGridLayout( buttonsFrame, 1, 5, 0 );
  buttonsGrid->setColStretch( 0, 1 );
  buttonsGrid->addWidget( saveButton, 0, 1 );
  buttonsGrid->addColSpacing( 2, 10 );
  buttonsGrid->addWidget( cancelButton, 0, 3 );
  buttonsGrid->setColStretch( 4, 1 );

  //place top level frames in grid
  mainGrid = new QGridLayout( this, 3, 2, 0);
  mainGrid->addWidget( themeSelectionFrame, 0, 0 );
  mainGrid->addWidget( themePreviewFrame, 0, 1 );
  mainGrid->addMultiCellWidget( locationFrame, 1, 1, 0, 1 );
  mainGrid->addMultiCellWidget( buttonsFrame, 2, 2, 0, 1 );

  //allow image and description region of select theme to expand to fit window
  mainGrid->setColStretch( 1, 1 );
  mainGrid->setRowStretch( 1, 1 );
  mainGrid->setMargin(WIDGET_SPACING);
  mainGrid->setSpacing(WIDGET_SPACING); 
  
  //set window to not be resizeable
  this->show();
  setFixedSize(size());
}
//==============================================
void SaveDialog::updatePreview()
{
  previewNum = 1;
  int i=1;
  QDir localDir( THEMES_PATH );
  while( localDir.exists( QString( themesList->currentText() + "/preview%1.png").arg(i) ) ) { i++; }
  numPreviews = i-1;

  //update theme description if provided
  if(localDir.exists( themesList->currentText() + "/description.html" ))
  {
    themeFeatures->setSource( themesList->currentText() + "/description.html" );
  }

  //update preview image to provide one or default otherwise
  if(localDir.exists( themesList->currentText() + "/preview1.png") )
  {
    screenShotLabel->setText( QString( tr("Screenshot") ) + QString( " %1/%2").arg(previewNum).arg(numPreviews) );
    themeScreenShot->setPixmap( QPixmap(THEMES_PATH + themesList->currentText() + "/preview1.png") );
    themeScreenPrev->setInvisible( true );    
    themeScreenNext->setInvisible( previewNum == numPreviews );
  }
  else
  {
    screenShotLabel->setText( "" );
    themeScreenShot->setPixmap( QPixmap(QString(IMAGE_PATH)+"miscImages/themePreview.png") );
    themeScreenPrev->setInvisible( true );
    themeScreenNext->setInvisible( true );
  }

}
//==============================================
void SaveDialog::save()
{
  accept();
}
//==============================================
void SaveDialog::cancel()
{
  reject();
}
//==============================================
void SaveDialog::prevScreenShot()
{
  previewNum--;
  themeScreenNext->setInvisible(false);
  themeScreenPrev->setInvisible(previewNum == 1);

  screenShotLabel->setText( QString( tr("Screenshot") ) + QString( " %1/%2").arg(previewNum).arg(numPreviews) );
  themeScreenShot->setPixmap( QPixmap( QString(THEMES_PATH + themesList->currentText() + "/preview%1.png").arg(previewNum) ) );
}
//==============================================
void SaveDialog::nextScreenShot()
{
  previewNum++;
  themeScreenPrev->setInvisible(false);
  themeScreenNext->setInvisible(previewNum == numPreviews);

  screenShotLabel->setText( QString( tr("Screenshot") ) + QString( " %1/%2").arg(previewNum).arg(numPreviews) );
  themeScreenShot->setPixmap( QPixmap( QString(THEMES_PATH + themesList->currentText() + "/preview%1.png").arg(previewNum) ) );
}
//==============================================
void SaveDialog::browse()
{
  //get directory from user
  QString dirName = QFileDialog::getSaveFileName( locationVal->text(),
                                                  NULL, this, NULL, QString(tr("Save as")) );

  if(!dirName.isNull())
    locationVal->setText( dirName );
}
//==============================================
QString SaveDialog::getTheme()
{
  return themesList->currentText();
}
//==============================================
QString SaveDialog::getPath()
{
  return locationVal->text();
}
//==============================================
bool SaveDialog::selectThemeAndPath( QString titleMessage,
                                                        QString defaultPath,
                                                        QString &theme,
                                                        QString &path )
{
  SaveDialog* dlg = new SaveDialog( titleMessage, defaultPath, theme );
  if( dlg->exec() == QDialog::Accepted )
  {
    theme = dlg->getTheme();
    path = dlg->getPath();
    delete dlg;
    return true;
  }
  else
  {
    delete dlg;
    return false;
  }
}
//==============================================
bool SaveDialog::themeAvailable(QString theme)
{
  //walk through the themes directory searching
  //for a directory with the name of the theme
  //that also has a theme.xsl file inside it
  QDir localDir( THEMES_PATH );
  QStringList list = localDir.entryList( QDir::Dirs );
  QStringList::Iterator file;
  for ( file = list.begin(); file != list.end(); ++file )
  {
    if(localDir.exists( QString(*file) + "/theme.xsl") &&
      QString(*file) == theme)
      return true;
  }
  //theme not found
  return false;
}
//==============================================
