//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

#ifndef BACKEND_ALBUM_H
#define BACKEND_ALBUM_H

//representative image sizes
#define SMALL 1
#define MEDIUM 2
#define LARGE 3

//define loading success values
#define ALBUM_LOADED 1
#define ALBUM_READ_ERROR 2
#define ALBUM_XML_ERROR 3
#define ALBUM_EXPORTED 1
#define ALBUM_ERROR_OPEN_FILE 2
//--------------------
//forward declarations
class QString;
class QPixmap;
class Subalbum;
class SubalbumPreviewWidget;
class StatusWidget;
class QStringList;
//--------------------

//=====================================
/*! \brief An album contains Subalbums.
 *
 *  Specific contents:
 * <ul>
 *  <li>Subalbums
 *  <li>Name
 *  <li>Description
 *  <li>Modification date
 *  <li>Author
 *  <li>Representative image
 * </ul>
 */
//=====================================

class Album
{
//------------------------------------------------------
public:

  ///Sets default information and create temporary directory as necessary
  Album( QString tmpDir, bool createSubalbum = true );

  ///Frees Subalbums
  ~Album();

  ///Sets the album name
  void setName(QString val);

  ///Gets the album name
  QString getName();

  ///Sets the album description
  void setDescription(QString val);

  ///Gets the album description
  QString getDescription();

  ///Sets the album author
  void setAuthor(QString val);

  ///Gets the album author
  QString getAuthor();

  ///Sets the representative image
  void setRepresentativeImages(QString imageFilename);

  ///Returns the representative image
  QPixmap* getRepresentativeImage(int size);

  ///Returns a pointer to the first Subalbum
  Subalbum* getFirstSubalbum();

  ///Returns a pointer to the last Subalbum
  Subalbum* getLastSubalbum();

  ///Appends subalbum to end of linked list
  void appendSubalbum(Subalbum* val);

  ///Removes a subalbum
  void removeSubalbum(Subalbum* val);

  ///Returns the last modified year
  int getModificationYear();

  ///Returns the last modified month
  int getModificationMonth();

  ///Returns the last modified day
  int getModificationDay();

  ///Returns the creation year
  int getCreationYear();

  ///Returns the creation month
  int getCreationMonth();

  ///Returnst he creation day
  int getCreationDay();

  ///Updates the creation date to today's date
  void Album::updateCreationDate();

  ///Updates the modification date to today's date
  void updateModificationDate();

  ///Imports album from XML format, returning int indicates success or not
  int importFromDisk(StatusWidget* status, QString fileName, bool disableCheckPhotoMods);

  ///Exports album in XML and HTML format, along with resized images
  int exportToDisk(StatusWidget* status, QString dirName, QString themeName);

  ///Exports album in XML and HTML format, along with resized images,
  ///saves all files to the last saved directory, if none set returns
  int exportToDisk(StatusWidget* status, bool forceSave=false);

  ///Export a compressed web album (excludes full size images and xml data)
  int exportCompressedWebAlbum(StatusWidget* status,
                               QString exportLocation,
                               QString exportMessage);
  
  ///Export fullsize images (excludes slideshow and thumbnail images, 
  ///album and collection iamges, and html or xml files)
  int exportLargeImages(StatusWidget* status, QString exportPath, QString exportMessage);  
  
  ///Returns true if album previously saved to disk
  bool prevSave();

  ///Returns true if album has been modified since the last save operation
  bool albumModified();

  ///Sets the album as modified
  void setModified(bool val=true);

  ///Syncs subalbum ordering with front end gui ordering
  void syncSubalbumList(SubalbumPreviewWidget* item);

  ///Returns the current save location of all images
  QString getSaveLocation();

  ///Returns the number of photos
  int getNumPhotos();
  
  ///Returns number of subalbums
  int getNumSubalbums();
  
  ///Returns currently selected theme
  QString getTheme();
  
  ///Returns the temporary directory for use when modifying and adding new images
  QString getTmpDir();
  
  ///Returns the next unique photo id
  int getNextUniquePhotoID();
  
  ///Returns a list of the most up to date thumbnail filesnames
  QStringList getThumbnailFilenames();
//------------------------------------------------------
private:        
  ///Exports album to XML
  int exportToXML(StatusWidget* status, QString exportPath);

  ///Exports top level images
  void exportTopLevelImages();

  ///Exports subalbum images
  void exportSubalbumImages(StatusWidget* status, bool forceSave);

  ///Removes any _orig images for photos which have been recently reverted
  ///to their original form (and hence we can reduce disk usage but removing these effective duplicates)
  void removeStagnantOrigFiles(StatusWidget* status);
  
  ///Checks if images need to be moved and does so if necessary
  void reorderSubalbumImages(StatusWidget* status);

  ///Removes old stagnant images caused when photos are removed from album or moved from one subalbum to another.
  void removeStagnantImages();

  ///Removes previously saved resources, copies over new resources
  void exportThemeResources( QString theme );

  ///Short name for album
  QString name;

  ///Longer description of album
  QString description;

  ///Album Creator
  QString author;

  ///Representative images
  QPixmap* smallRepresentativeImage;
  QPixmap* largeRepresentativeImage;

  ///Pointer to first Subalbum
  Subalbum* firstSubalbum;

  ///Pointer to last Subalbum
  Subalbum* lastSubalbum;

  ///Last modification year
  int modificationYear;

  ///Last modification month
  int modificationMonth;

  ///Last modification day
  int modificationDay;

  ///Creation year
  int creationYear;

  ///Creation month
  int creationMonth;

  ///Creation day
  int creationDay;

  ///Number of subalbums
  int numSubalbums;

  ///Number of loaded subalbums
  int numLoadedSubalbums;

  ///Set if album was loaded/has been saved to disk
  bool savedToDisk;

  ///Directory album saved to
  QString saveLocation;

  //Last directory album was saved to
  QString oldSaveLocation;
  
  ///Theme to save album with
  QString theme;

  ///Modification status of the album
  bool modified;
  
  ///Temporary directory for placing modified or new images before saving takes place.
  QString tmpDir;
  
  ///Next Unique ID for new photos. This counter
  ///is used to gerneate unique filenames before photos are saved. After
  ///saving we reset this counter to avoid wrap-around.
  int nextUniqueID;
//------------------------------------------------------
};

#endif //BACKEND_ALBUM_H
